"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elastictranscoder = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elastictranscoder](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastictranscoder.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elastictranscoder extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elastictranscoder](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastictranscoder.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elastictranscoder';
        this.accessLevelList = {
            "Write": [
                "CancelJob",
                "CreateJob",
                "CreatePipeline",
                "CreatePreset",
                "DeletePipeline",
                "DeletePreset",
                "TestRole",
                "UpdatePipeline",
                "UpdatePipelineNotifications",
                "UpdatePipelineStatus"
            ],
            "List": [
                "ListJobsByPipeline",
                "ListJobsByStatus",
                "ListPipelines",
                "ListPresets"
            ],
            "Read": [
                "ReadJob",
                "ReadPipeline",
                "ReadPreset"
            ]
        };
    }
    /**
     * Cancel a job that Elastic Transcoder has not begun to process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/cancel-job.html
     */
    toCancelJob() {
        this.to('elastictranscoder:CancelJob');
        return this;
    }
    /**
     * Create a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-job.html
     */
    toCreateJob() {
        this.to('elastictranscoder:CreateJob');
        return this;
    }
    /**
     * Create a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-pipeline.html
     */
    toCreatePipeline() {
        this.to('elastictranscoder:CreatePipeline');
        return this;
    }
    /**
     * Create a preset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-preset.html
     */
    toCreatePreset() {
        this.to('elastictranscoder:CreatePreset');
        return this;
    }
    /**
     * Delete a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-pipeline.html
     */
    toDeletePipeline() {
        this.to('elastictranscoder:DeletePipeline');
        return this;
    }
    /**
     * Delete a preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-preset.html
     */
    toDeletePreset() {
        this.to('elastictranscoder:DeletePreset');
        return this;
    }
    /**
     * Get a list of the jobs that you assigned to a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-pipeline.html
     */
    toListJobsByPipeline() {
        this.to('elastictranscoder:ListJobsByPipeline');
        return this;
    }
    /**
     * Get information about all of the jobs associated with the current AWS account that have a specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-status.html
     */
    toListJobsByStatus() {
        this.to('elastictranscoder:ListJobsByStatus');
        return this;
    }
    /**
     * Get a list of the pipelines associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-pipelines.html
     */
    toListPipelines() {
        this.to('elastictranscoder:ListPipelines');
        return this;
    }
    /**
     * Get a list of all presets associated with the current AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-presets.html
     */
    toListPresets() {
        this.to('elastictranscoder:ListPresets');
        return this;
    }
    /**
     * Get detailed information about a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-job.html
     */
    toReadJob() {
        this.to('elastictranscoder:ReadJob');
        return this;
    }
    /**
     * Get detailed information about a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-pipeline.html
     */
    toReadPipeline() {
        this.to('elastictranscoder:ReadPipeline');
        return this;
    }
    /**
     * Get detailed information about a preset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-preset.html
     */
    toReadPreset() {
        this.to('elastictranscoder:ReadPreset');
        return this;
    }
    /**
     * Test the settings for a pipeline to ensure that Elastic Transcoder can create and process jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/test-pipeline-role.html
     */
    toTestRole() {
        this.to('elastictranscoder:TestRole');
        return this;
    }
    /**
     * Update settings for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline.html
     */
    toUpdatePipeline() {
        this.to('elastictranscoder:UpdatePipeline');
        return this;
    }
    /**
     * Update only Amazon Simple Notification Service (Amazon SNS) notifications for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-notifications.html
     */
    toUpdatePipelineNotifications() {
        this.to('elastictranscoder:UpdatePipelineNotifications');
        return this;
    }
    /**
     * Pause or reactivate a pipeline, so the pipeline stops or restarts processing jobs, update the status for the pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-status.html
     */
    toUpdatePipelineStatus() {
        this.to('elastictranscoder:UpdatePipelineStatus');
        return this;
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-pipelines.html
     *
     * @param pipelineId - Identifier for the pipelineId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPipeline(pipelineId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:pipeline/${PipelineId}';
        arn = arn.replace('${PipelineId}', pipelineId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type preset to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-presets.html
     *
     * @param presetId - Identifier for the presetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPreset(presetId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:preset/${PresetId}';
        arn = arn.replace('${PresetId}', presetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Elastictranscoder = Elastictranscoder;
//# sourceMappingURL=data:application/json;base64,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