"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Frauddetector = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [frauddetector](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Frauddetector extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [frauddetector](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'frauddetector';
        this.accessLevelList = {
            "Write": [
                "BatchCreateVariable",
                "CreateDetectorVersion",
                "CreateModel",
                "CreateModelVersion",
                "CreateRule",
                "CreateVariable",
                "DeleteDetector",
                "DeleteDetectorVersion",
                "DeleteEntityType",
                "DeleteEvent",
                "DeleteEventType",
                "DeleteExternalModel",
                "DeleteLabel",
                "DeleteModel",
                "DeleteModelVersion",
                "DeleteOutcome",
                "DeleteRule",
                "DeleteVariable",
                "PutDetector",
                "PutEntityType",
                "PutEventType",
                "PutExternalModel",
                "PutKMSEncryptionKey",
                "PutLabel",
                "PutOutcome",
                "UpdateDetectorVersion",
                "UpdateDetectorVersionMetadata",
                "UpdateDetectorVersionStatus",
                "UpdateModel",
                "UpdateModelVersion",
                "UpdateModelVersionStatus",
                "UpdateRuleMetadata",
                "UpdateRuleVersion",
                "UpdateVariable"
            ],
            "List": [
                "BatchGetVariable",
                "GetDetectorVersion",
                "GetDetectors",
                "GetEntityTypes",
                "GetEventTypes",
                "GetExternalModels",
                "GetLabels",
                "GetModelVersion",
                "GetModels",
                "GetOutcomes",
                "GetRules",
                "GetVariables",
                "ListTagsForResource"
            ],
            "Read": [
                "DescribeDetector",
                "DescribeModelVersions",
                "GetEventPrediction",
                "GetKMSEncryptionKey"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates a batch of variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchCreateVariable
     */
    toBatchCreateVariable() {
        this.to('frauddetector:BatchCreateVariable');
        return this;
    }
    /**
     * Gets a batch of variables.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchGetVariable
     */
    toBatchGetVariable() {
        this.to('frauddetector:BatchGetVariable');
        return this;
    }
    /**
     * Creates a detector version. The detector version starts in a DRAFT status.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateDetectorVersion
     */
    toCreateDetectorVersion() {
        this.to('frauddetector:CreateDetectorVersion');
        return this;
    }
    /**
     * Creates a model using the specified model type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModel
     */
    toCreateModel() {
        this.to('frauddetector:CreateModel');
        return this;
    }
    /**
     * Creates a version of the model using the specified model type and model id.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModelVersion
     */
    toCreateModelVersion() {
        this.to('frauddetector:CreateModelVersion');
        return this;
    }
    /**
     * Creates a rule for use with the specified detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateRule.html
     */
    toCreateRule() {
        this.to('frauddetector:CreateRule');
        return this;
    }
    /**
     * Creates a variable.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateVariable.html
     */
    toCreateVariable() {
        this.to('frauddetector:CreateVariable');
        return this;
    }
    /**
     * Deletes the detector. Before deleting a detector, you must first delete all detector versions and rule versions associated with the detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetector
     */
    toDeleteDetector() {
        this.to('frauddetector:DeleteDetector');
        return this;
    }
    /**
     * Deletes the detector version. You cannot delete detector versions that are in ACTIVE status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetectorVersion
     */
    toDeleteDetectorVersion() {
        this.to('frauddetector:DeleteDetectorVersion');
        return this;
    }
    /**
     * Deletes an entity type. You cannot delete an entity type that is included in an event type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEntityType
     */
    toDeleteEntityType() {
        this.to('frauddetector:DeleteEntityType');
        return this;
    }
    /**
     * Deletes the specified event.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEvent
     */
    toDeleteEvent() {
        this.to('frauddetector:DeleteEvent');
        return this;
    }
    /**
     * Deletes an event type. You cannot delete an event type that is used in a detector or a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEventType
     */
    toDeleteEventType() {
        this.to('frauddetector:DeleteEventType');
        return this;
    }
    /**
     * Removes a SageMaker model from Amazon Fraud Detector. You can remove an Amazon SageMaker model if it is not associated with a detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteExternalModel
     */
    toDeleteExternalModel() {
        this.to('frauddetector:DeleteExternalModel');
        return this;
    }
    /**
     * Deletes a label. You cannot delete labels that are included in an event type in Amazon Fraud Detector. You cannot delete a label assigned to an event ID. You must first delete the relevant event ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteLabel
     */
    toDeleteLabel() {
        this.to('frauddetector:DeleteLabel');
        return this;
    }
    /**
     * Deletes a model. You can delete models and model versions in Amazon Fraud Detector, provided that they are not associated with a detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteModel
     */
    toDeleteModel() {
        this.to('frauddetector:DeleteModel');
        return this;
    }
    /**
     * Deletes a model version. You can delete models and model versions in Amazon Fraud Detector, provided that they are not associated with a detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteModelVersion
     */
    toDeleteModelVersion() {
        this.to('frauddetector:DeleteModelVersion');
        return this;
    }
    /**
     * Deletes an outcome. You cannot delete an outcome that is used in a rule version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteOutcome
     */
    toDeleteOutcome() {
        this.to('frauddetector:DeleteOutcome');
        return this;
    }
    /**
     * Deletes the rule. You cannot delete a rule if it is used by an ACTIVE or INACTIVE detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteRule
     */
    toDeleteRule() {
        this.to('frauddetector:DeleteRule');
        return this;
    }
    /**
     * Deletes a variable. You cannot delete variables that are included in an event type in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteVariable
     */
    toDeleteVariable() {
        this.to('frauddetector:DeleteVariable');
        return this;
    }
    /**
     * Gets all versions for a specified detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeDetector
     */
    toDescribeDetector() {
        this.to('frauddetector:DescribeDetector');
        return this;
    }
    /**
     * Gets all of the model versions for the specified model type or for the specified model type and model ID. You can also get details for a single, specified model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeModelVersions
     */
    toDescribeModelVersions() {
        this.to('frauddetector:DescribeModelVersions');
        return this;
    }
    /**
     * Gets a particular detector version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectorVersion
     */
    toGetDetectorVersion() {
        this.to('frauddetector:GetDetectorVersion');
        return this;
    }
    /**
     * Gets all detectors or a single detector if a detectorId is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetDetectorsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectors
     */
    toGetDetectors() {
        this.to('frauddetector:GetDetectors');
        return this;
    }
    /**
     * Gets all entity types or a specific entity type if a name is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEntityTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEntityTypes
     */
    toGetEntityTypes() {
        this.to('frauddetector:GetEntityTypes');
        return this;
    }
    /**
     * Evaluates an event against a detector version. If a version ID is not provided, the detector’s (ACTIVE) version is used.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPrediction
     */
    toGetEventPrediction() {
        this.to('frauddetector:GetEventPrediction');
        return this;
    }
    /**
     * Gets all event types or a specific event type if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEventTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventTypes
     */
    toGetEventTypes() {
        this.to('frauddetector:GetEventTypes');
        return this;
    }
    /**
     * Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetExternalModelsResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetExternalModels
     */
    toGetExternalModels() {
        this.to('frauddetector:GetExternalModels');
        return this;
    }
    /**
     * Gets the encryption key if a Key Management Service (KMS) customer master key (CMK) has been specified to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetKMSEncryptionKey
     */
    toGetKMSEncryptionKey() {
        this.to('frauddetector:GetKMSEncryptionKey');
        return this;
    }
    /**
     * Gets all labels or a specific label if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 10 and 50. To get the next page results, provide the pagination token from the GetGetLabelsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetLabels
     */
    toGetLabels() {
        this.to('frauddetector:GetLabels');
        return this;
    }
    /**
     * Gets the details of the specified model version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModelVersion
     */
    toGetModelVersion() {
        this.to('frauddetector:GetModelVersion');
        return this;
    }
    /**
     * Gets one or more models. Gets all models for the AWS account if no model type and no model id provided. Gets all models for the AWS account and model type, if the model type is specified but model id is not provided. Gets a specific model if (model type, model id) tuple is specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModels
     */
    toGetModels() {
        this.to('frauddetector:GetModels');
        return this;
    }
    /**
     * Gets one or more outcomes. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 100 records per page. If you provide a maxResults, the value must be between 50 and 100. To get the next page results, provide the pagination token from the GetOutcomesResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetOutcomes
     */
    toGetOutcomes() {
        this.to('frauddetector:GetOutcomes');
        return this;
    }
    /**
     * Get all rules for a detector (paginated) if ruleId and ruleVersion are not specified. Gets all rules for the detector and the ruleId if present (paginated). Gets a specific rule if both the ruleId and the ruleVersion are specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetRules.html
     */
    toGetRules() {
        this.to('frauddetector:GetRules');
        return this;
    }
    /**
     * Gets all of the variables or the specific variable. This is a paginated API. Providing null maxSizePerPage results in retrieving maximum of 100 records per page. If you provide maxSizePerPage the value must be between 50 and 100. To get the next page result, a provide a pagination token from GetVariablesResult as part of your request. Null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetVariables
     */
    toGetVariables() {
        this.to('frauddetector:GetVariables');
        return this;
    }
    /**
     * Lists all tags associated with the resource. This is a paginated API. To get the next page results, provide the pagination token from the response as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('frauddetector:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates a detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutDetector.html
     */
    toPutDetector() {
        this.to('frauddetector:PutDetector');
        return this;
    }
    /**
     * Creates or updates an entity type. An entity represents who is performing the event. As part of a fraud prediction, you pass the entity ID to indicate the specific entity who performed the event. An entity type classifies the entity. Example classifications include customer, merchant, or account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEntityType.html
     */
    toPutEntityType() {
        this.to('frauddetector:PutEntityType');
        return this;
    }
    /**
     * Creates or updates an event type. An event is a business activity that is evaluated for fraud risk. With Amazon Fraud Detector, you generate fraud predictions for events. An event type defines the structure for an event sent to Amazon Fraud Detector. This includes the variables sent as part of the event, the entity performing the event (such as a customer), and the labels that classify the event. Example event types include online payment transactions, account registrations, and authentications.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEventType.html
     */
    toPutEventType() {
        this.to('frauddetector:PutEventType');
        return this;
    }
    /**
     * Creates or updates an Amazon SageMaker model endpoint. You can also use this action to update the configuration of the model endpoint, including the IAM role and/or the mapped variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutExternalModel.html
     */
    toPutExternalModel() {
        this.to('frauddetector:PutExternalModel');
        return this;
    }
    /**
     * Specifies the Key Management Service (KMS) customer master key (CMK) to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutKMSEncryptionKey.html
     */
    toPutKMSEncryptionKey() {
        this.to('frauddetector:PutKMSEncryptionKey');
        return this;
    }
    /**
     * Creates or updates label. A label classifies an event as fraudulent or legitimate. Labels are associated with event types and used to train supervised machine learning models in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutLabel.html
     */
    toPutLabel() {
        this.to('frauddetector:PutLabel');
        return this;
    }
    /**
     * Creates or updates an outcome.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutOutcome.html
     */
    toPutOutcome() {
        this.to('frauddetector:PutOutcome');
        return this;
    }
    /**
     * Assigns tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('frauddetector:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('frauddetector:UntagResource');
        return this;
    }
    /**
     * Updates a detector version. The detector version attributes that you can update include models, external model endpoints, rules, rule execution mode, and description. You can only update a DRAFT detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersion
     */
    toUpdateDetectorVersion() {
        this.to('frauddetector:UpdateDetectorVersion');
        return this;
    }
    /**
     * Updates the detector version's description. You can update the metadata for any detector version (DRAFT, ACTIVE, or INACTIVE).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionMetadata
     */
    toUpdateDetectorVersionMetadata() {
        this.to('frauddetector:UpdateDetectorVersionMetadata');
        return this;
    }
    /**
     * Updates the detector version’s status. You can perform the following promotions or demotions using UpdateDetectorVersionStatus: DRAFT to ACTIVE, ACTIVE to INACTIVE, and INACTIVE to ACTIVE.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionStatus
     */
    toUpdateDetectorVersionStatus() {
        this.to('frauddetector:UpdateDetectorVersionStatus');
        return this;
    }
    /**
     * Updates a model. You can update the description attribute using this action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModel
     */
    toUpdateModel() {
        this.to('frauddetector:UpdateModel');
        return this;
    }
    /**
     * Updates a model version. Updating a model version retrains an existing model version using updated training data and produces a new minor version of the model. You can update the training data set location and data access role attributes using this action. This action creates and trains a new minor version of the model, for example version 1.01, 1.02, 1.03.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersion
     */
    toUpdateModelVersion() {
        this.to('frauddetector:UpdateModelVersion');
        return this;
    }
    /**
     * Updates the status of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersionStatus
     */
    toUpdateModelVersionStatus() {
        this.to('frauddetector:UpdateModelVersionStatus');
        return this;
    }
    /**
     * Updates a rule's metadata. The description attribute can be updated.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleMetadata.html
     */
    toUpdateRuleMetadata() {
        this.to('frauddetector:UpdateRuleMetadata');
        return this;
    }
    /**
     * Updates a rule version resulting in a new rule version. Updates a rule version resulting in a new rule version (version 1, 2, 3 ...).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleVersion.html
     */
    toUpdateRuleVersion() {
        this.to('frauddetector:UpdateRuleVersion');
        return this;
    }
    /**
     * Updates a variable.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateVariable.html
     */
    toUpdateVariable() {
        this.to('frauddetector:UpdateVariable');
        return this;
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type detector-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:entity-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type external-model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExternalModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:external-model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:event-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type label to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLabel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:label/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type outcome to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOutcome(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:outcome/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:rule/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type variable to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVariable(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:variable/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Frauddetector = Frauddetector;
//# sourceMappingURL=data:application/json;base64,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