"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Glacier = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [glacier](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonglacier.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Glacier extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [glacier](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonglacier.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'glacier';
        this.accessLevelList = {
            "Write": [
                "AbortMultipartUpload",
                "CompleteMultipartUpload",
                "CreateVault",
                "DeleteArchive",
                "DeleteVault",
                "DeleteVaultNotifications",
                "InitiateJob",
                "InitiateMultipartUpload",
                "PurchaseProvisionedCapacity",
                "SetVaultNotifications",
                "UploadArchive",
                "UploadMultipartPart"
            ],
            "Permissions management": [
                "AbortVaultLock",
                "CompleteVaultLock",
                "DeleteVaultAccessPolicy",
                "InitiateVaultLock",
                "SetDataRetrievalPolicy",
                "SetVaultAccessPolicy"
            ],
            "Tagging": [
                "AddTagsToVault",
                "RemoveTagsFromVault"
            ],
            "Read": [
                "DescribeJob",
                "DescribeVault",
                "GetDataRetrievalPolicy",
                "GetJobOutput",
                "GetVaultAccessPolicy",
                "GetVaultLock",
                "GetVaultNotifications"
            ],
            "List": [
                "ListJobs",
                "ListMultipartUploads",
                "ListParts",
                "ListProvisionedCapacity",
                "ListTagsForVault",
                "ListVaults"
            ]
        };
    }
    /**
     * Aborts a multipart upload identified by the upload ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html
     */
    toAbortMultipartUpload() {
        this.to('glacier:AbortMultipartUpload');
        return this;
    }
    /**
     * Aborts the vault locking process if the vault lock is not in the Locked state
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AbortVaultLock.html
     */
    toAbortVaultLock() {
        this.to('glacier:AbortVaultLock');
        return this;
    }
    /**
     * Adds the specified tags to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AddTagsToVault.html
     */
    toAddTagsToVault() {
        this.to('glacier:AddTagsToVault');
        return this;
    }
    /**
     * Completes a multipart upload process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html
     */
    toCompleteMultipartUpload() {
        this.to('glacier:CompleteMultipartUpload');
        return this;
    }
    /**
     * Completes the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-CompleteVaultLock.html
     */
    toCompleteVaultLock() {
        this.to('glacier:CompleteVaultLock');
        return this;
    }
    /**
     * Creates a new vault with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html
     */
    toCreateVault() {
        this.to('glacier:CreateVault');
        return this;
    }
    /**
     * Deletes an archive from a vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifArchiveAgeInDays()
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html
     */
    toDeleteArchive() {
        this.to('glacier:DeleteArchive');
        return this;
    }
    /**
     * Deletes a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html
     */
    toDeleteVault() {
        this.to('glacier:DeleteVault');
        return this;
    }
    /**
     * Deletes the access policy associated with the specified vault
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-DeleteVaultAccessPolicy.html
     */
    toDeleteVaultAccessPolicy() {
        this.to('glacier:DeleteVaultAccessPolicy');
        return this;
    }
    /**
     * Deletes the notification configuration set for a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html
     */
    toDeleteVaultNotifications() {
        this.to('glacier:DeleteVaultNotifications');
        return this;
    }
    /**
     * Returns information about a job you previously initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html
     */
    toDescribeJob() {
        this.to('glacier:DescribeJob');
        return this;
    }
    /**
     * Returns information about a vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get..html
     */
    toDescribeVault() {
        this.to('glacier:DescribeVault');
        return this;
    }
    /**
     * Returns the current data retrieval policy for the account and region specified in the GET request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetDataRetrievalPolicy.html
     */
    toGetDataRetrievalPolicy() {
        this.to('glacier:GetDataRetrievalPolicy');
        return this;
    }
    /**
     * Downloads the output of the job you initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html
     */
    toGetJobOutput() {
        this.to('glacier:GetJobOutput');
        return this;
    }
    /**
     * Retrieves the access-policy subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultAccessPolicy.html
     */
    toGetVaultAccessPolicy() {
        this.to('glacier:GetVaultAccessPolicy');
        return this;
    }
    /**
     * Retrieves attributes from the lock-policy subresource set on the specified vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultLock.html
     */
    toGetVaultLock() {
        this.to('glacier:GetVaultLock');
        return this;
    }
    /**
     * Retrieves the notification-configuration subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html
     */
    toGetVaultNotifications() {
        this.to('glacier:GetVaultNotifications');
        return this;
    }
    /**
     * Initiates a job of the specified type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifArchiveAgeInDays()
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html
     */
    toInitiateJob() {
        this.to('glacier:InitiateJob');
        return this;
    }
    /**
     * Initiates a multipart upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html
     */
    toInitiateMultipartUpload() {
        this.to('glacier:InitiateMultipartUpload');
        return this;
    }
    /**
     * Initiates the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-InitiateVaultLock.html
     */
    toInitiateVaultLock() {
        this.to('glacier:InitiateVaultLock');
        return this;
    }
    /**
     * Lists jobs for a vault that are in-progress and jobs that have recently finished
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html
     */
    toListJobs() {
        this.to('glacier:ListJobs');
        return this;
    }
    /**
     * Lists in-progress multipart uploads for the specified vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html
     */
    toListMultipartUploads() {
        this.to('glacier:ListMultipartUploads');
        return this;
    }
    /**
     * Lists the parts of an archive that have been uploaded in a specific multipart upload
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html
     */
    toListParts() {
        this.to('glacier:ListParts');
        return this;
    }
    /**
     * This operation lists the provisioned capacity for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListProvisionedCapacity.html
     */
    toListProvisionedCapacity() {
        this.to('glacier:ListProvisionedCapacity');
        return this;
    }
    /**
     * Lists all the tags attached to a vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListTagsForVault.html
     */
    toListTagsForVault() {
        this.to('glacier:ListTagsForVault');
        return this;
    }
    /**
     * Lists all vaults
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html
     */
    toListVaults() {
        this.to('glacier:ListVaults');
        return this;
    }
    /**
     * This operation purchases a provisioned capacity unit for an AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-PurchaseProvisionedCapacity.html
     */
    toPurchaseProvisionedCapacity() {
        this.to('glacier:PurchaseProvisionedCapacity');
        return this;
    }
    /**
     * Removes one or more tags from the set of tags attached to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-RemoveTagsFromVault.html
     */
    toRemoveTagsFromVault() {
        this.to('glacier:RemoveTagsFromVault');
        return this;
    }
    /**
     * Sets and then enacts a data retrieval policy in the region specified in the PUT request
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetDataRetrievalPolicy.html
     */
    toSetDataRetrievalPolicy() {
        this.to('glacier:SetDataRetrievalPolicy');
        return this;
    }
    /**
     * Configures an access policy for a vault and will overwrite an existing policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html
     */
    toSetVaultAccessPolicy() {
        this.to('glacier:SetVaultAccessPolicy');
        return this;
    }
    /**
     * Configures vault notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html
     */
    toSetVaultNotifications() {
        this.to('glacier:SetVaultNotifications');
        return this;
    }
    /**
     * Adds an archive to a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html
     */
    toUploadArchive() {
        this.to('glacier:UploadArchive');
        return this;
    }
    /**
     * Uploads a part of an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html
     */
    toUploadMultipartPart() {
        this.to('glacier:UploadMultipartPart');
        return this;
    }
    /**
     * Adds a resource of type vault to the statement
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-vaults.html
     *
     * @param vaultName - Identifier for the vaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVault(vaultName, account, region, partition) {
        var arn = 'arn:${Partition}:glacier:${Region}:${Account}:vaults/${VaultName}';
        arn = arn.replace('${VaultName}', vaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * How long an archive has been stored in the vault, in days.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDeleteArchive()
     * - .toInitiateJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArchiveAgeInDays(value, operator) {
        return this.if(`glacier:ArchiveAgeInDays`, value, operator || 'StringLike');
    }
    /**
     * A customer-defined tag.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`glacier:ResourceTag/`, value, operator || 'StringLike');
    }
}
exports.Glacier = Glacier;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uZ2xhY2llci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFtYXpvbmdsYWNpZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esc0NBQXNEO0FBRXREOzs7O0dBSUc7QUFDSCxNQUFhLE9BQVEsU0FBUSx3QkFBZTtJQUcxQzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQVJOLGtCQUFhLEdBQUcsU0FBUyxDQUFDO1FBNlp2QixvQkFBZSxHQUFvQjtZQUMzQyxPQUFPLEVBQUU7Z0JBQ1Asc0JBQXNCO2dCQUN0Qix5QkFBeUI7Z0JBQ3pCLGFBQWE7Z0JBQ2IsZUFBZTtnQkFDZixhQUFhO2dCQUNiLDBCQUEwQjtnQkFDMUIsYUFBYTtnQkFDYix5QkFBeUI7Z0JBQ3pCLDZCQUE2QjtnQkFDN0IsdUJBQXVCO2dCQUN2QixlQUFlO2dCQUNmLHFCQUFxQjthQUN0QjtZQUNELHdCQUF3QixFQUFFO2dCQUN4QixnQkFBZ0I7Z0JBQ2hCLG1CQUFtQjtnQkFDbkIseUJBQXlCO2dCQUN6QixtQkFBbUI7Z0JBQ25CLHdCQUF3QjtnQkFDeEIsc0JBQXNCO2FBQ3ZCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGdCQUFnQjtnQkFDaEIscUJBQXFCO2FBQ3RCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLGFBQWE7Z0JBQ2IsZUFBZTtnQkFDZix3QkFBd0I7Z0JBQ3hCLGNBQWM7Z0JBQ2Qsc0JBQXNCO2dCQUN0QixjQUFjO2dCQUNkLHVCQUF1QjthQUN4QjtZQUNELE1BQU0sRUFBRTtnQkFDTixVQUFVO2dCQUNWLHNCQUFzQjtnQkFDdEIsV0FBVztnQkFDWCx5QkFBeUI7Z0JBQ3pCLGtCQUFrQjtnQkFDbEIsWUFBWTthQUNiO1NBQ0YsQ0FBQztJQWhjRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUM1QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFnREQ7Ozs7Ozs7OztPQVNHO0lBQ0ksT0FBTyxDQUFDLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckYsSUFBSSxHQUFHLEdBQUcsbUVBQW1FLENBQUM7UUFDOUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxrQkFBa0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzlFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksbUJBQW1CLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMvRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUMxRSxDQUFDO0NBQ0Y7QUExZkQsMEJBMGZDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSBcIi4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWxcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtnbGFjaWVyXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25nbGFjaWVyLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgR2xhY2llciBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2dsYWNpZXInO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2dsYWNpZXJdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbmdsYWNpZXIuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogQWJvcnRzIGEgbXVsdGlwYXJ0IHVwbG9hZCBpZGVudGlmaWVkIGJ5IHRoZSB1cGxvYWQgSURcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1tdWx0aXBhcnQtYWJvcnQtdXBsb2FkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fib3J0TXVsdGlwYXJ0VXBsb2FkKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6QWJvcnRNdWx0aXBhcnRVcGxvYWQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBYm9ydHMgdGhlIHZhdWx0IGxvY2tpbmcgcHJvY2VzcyBpZiB0aGUgdmF1bHQgbG9jayBpcyBub3QgaW4gdGhlIExvY2tlZCBzdGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktQWJvcnRWYXVsdExvY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWJvcnRWYXVsdExvY2soKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpBYm9ydFZhdWx0TG9jaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgdGhlIHNwZWNpZmllZCB0YWdzIHRvIGEgdmF1bHRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25nbGFjaWVyL2xhdGVzdC9kZXYvYXBpLUFkZFRhZ3NUb1ZhdWx0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FkZFRhZ3NUb1ZhdWx0KCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6QWRkVGFnc1RvVmF1bHQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wbGV0ZXMgYSBtdWx0aXBhcnQgdXBsb2FkIHByb2Nlc3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1tdWx0aXBhcnQtY29tcGxldGUtdXBsb2FkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6Q29tcGxldGVNdWx0aXBhcnRVcGxvYWQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wbGV0ZXMgdGhlIHZhdWx0IGxvY2tpbmcgcHJvY2Vzc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktQ29tcGxldGVWYXVsdExvY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ29tcGxldGVWYXVsdExvY2soKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpDb21wbGV0ZVZhdWx0TG9jaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgdmF1bHQgd2l0aCB0aGUgc3BlY2lmaWVkIG5hbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS12YXVsdC1wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVmF1bHQoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpDcmVhdGVWYXVsdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYW4gYXJjaGl2ZSBmcm9tIGEgdmF1bHRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFyY2hpdmVBZ2VJbkRheXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1hcmNoaXZlLWRlbGV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBcmNoaXZlKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6RGVsZXRlQXJjaGl2ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSB2YXVsdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25nbGFjaWVyL2xhdGVzdC9kZXYvYXBpLXZhdWx0LWRlbGV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVWYXVsdCgpIHtcbiAgICB0aGlzLnRvKCdnbGFjaWVyOkRlbGV0ZVZhdWx0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgYWNjZXNzIHBvbGljeSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCB2YXVsdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktRGVsZXRlVmF1bHRBY2Nlc3NQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlVmF1bHRBY2Nlc3NQb2xpY3koKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpEZWxldGVWYXVsdEFjY2Vzc1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIG5vdGlmaWNhdGlvbiBjb25maWd1cmF0aW9uIHNldCBmb3IgYSB2YXVsdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25nbGFjaWVyL2xhdGVzdC9kZXYvYXBpLXZhdWx0LW5vdGlmaWNhdGlvbnMtZGVsZXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVZhdWx0Tm90aWZpY2F0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdnbGFjaWVyOkRlbGV0ZVZhdWx0Tm90aWZpY2F0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYSBqb2IgeW91IHByZXZpb3VzbHkgaW5pdGlhdGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1kZXNjcmliZS1qb2ItZ2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlSm9iKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6RGVzY3JpYmVKb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGEgdmF1bHRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25nbGFjaWVyL2xhdGVzdC9kZXYvYXBpLXZhdWx0LWdldC4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVWYXVsdCgpIHtcbiAgICB0aGlzLnRvKCdnbGFjaWVyOkRlc2NyaWJlVmF1bHQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBjdXJyZW50IGRhdGEgcmV0cmlldmFsIHBvbGljeSBmb3IgdGhlIGFjY291bnQgYW5kIHJlZ2lvbiBzcGVjaWZpZWQgaW4gdGhlIEdFVCByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1HZXREYXRhUmV0cmlldmFsUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERhdGFSZXRyaWV2YWxQb2xpY3koKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpHZXREYXRhUmV0cmlldmFsUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRG93bmxvYWRzIHRoZSBvdXRwdXQgb2YgdGhlIGpvYiB5b3UgaW5pdGlhdGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1qb2Itb3V0cHV0LWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRKb2JPdXRwdXQoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpHZXRKb2JPdXRwdXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgdGhlIGFjY2Vzcy1wb2xpY3kgc3VicmVzb3VyY2Ugc2V0IG9uIHRoZSB2YXVsdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktR2V0VmF1bHRBY2Nlc3NQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0VmF1bHRBY2Nlc3NQb2xpY3koKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpHZXRWYXVsdEFjY2Vzc1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhdHRyaWJ1dGVzIGZyb20gdGhlIGxvY2stcG9saWN5IHN1YnJlc291cmNlIHNldCBvbiB0aGUgc3BlY2lmaWVkIHZhdWx0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1HZXRWYXVsdExvY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0VmF1bHRMb2NrKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6R2V0VmF1bHRMb2NrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIHRoZSBub3RpZmljYXRpb24tY29uZmlndXJhdGlvbiBzdWJyZXNvdXJjZSBzZXQgb24gdGhlIHZhdWx0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS12YXVsdC1ub3RpZmljYXRpb25zLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRWYXVsdE5vdGlmaWNhdGlvbnMoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpHZXRWYXVsdE5vdGlmaWNhdGlvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBJbml0aWF0ZXMgYSBqb2Igb2YgdGhlIHNwZWNpZmllZCB0eXBlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcmNoaXZlQWdlSW5EYXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktaW5pdGlhdGUtam9iLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvSW5pdGlhdGVKb2IoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpJbml0aWF0ZUpvYicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluaXRpYXRlcyBhIG11bHRpcGFydCB1cGxvYWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1tdWx0aXBhcnQtaW5pdGlhdGUtdXBsb2FkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0luaXRpYXRlTXVsdGlwYXJ0VXBsb2FkKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6SW5pdGlhdGVNdWx0aXBhcnRVcGxvYWQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBJbml0aWF0ZXMgdGhlIHZhdWx0IGxvY2tpbmcgcHJvY2Vzc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktSW5pdGlhdGVWYXVsdExvY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvSW5pdGlhdGVWYXVsdExvY2soKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpJbml0aWF0ZVZhdWx0TG9jaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGpvYnMgZm9yIGEgdmF1bHQgdGhhdCBhcmUgaW4tcHJvZ3Jlc3MgYW5kIGpvYnMgdGhhdCBoYXZlIHJlY2VudGx5IGZpbmlzaGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1qb2JzLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Sm9icygpIHtcbiAgICB0aGlzLnRvKCdnbGFjaWVyOkxpc3RKb2JzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgaW4tcHJvZ3Jlc3MgbXVsdGlwYXJ0IHVwbG9hZHMgZm9yIHRoZSBzcGVjaWZpZWQgdmF1bHRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25nbGFjaWVyL2xhdGVzdC9kZXYvYXBpLW11bHRpcGFydC1saXN0LXVwbG9hZHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE11bHRpcGFydFVwbG9hZHMoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpMaXN0TXVsdGlwYXJ0VXBsb2FkcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBwYXJ0cyBvZiBhbiBhcmNoaXZlIHRoYXQgaGF2ZSBiZWVuIHVwbG9hZGVkIGluIGEgc3BlY2lmaWMgbXVsdGlwYXJ0IHVwbG9hZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktbXVsdGlwYXJ0LWxpc3QtcGFydHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFBhcnRzKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6TGlzdFBhcnRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBvcGVyYXRpb24gbGlzdHMgdGhlIHByb3Zpc2lvbmVkIGNhcGFjaXR5IGZvciB0aGUgc3BlY2lmaWVkIEFXUyBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktTGlzdFByb3Zpc2lvbmVkQ2FwYWNpdHkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFByb3Zpc2lvbmVkQ2FwYWNpdHkoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpMaXN0UHJvdmlzaW9uZWRDYXBhY2l0eScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGFsbCB0aGUgdGFncyBhdHRhY2hlZCB0byBhIHZhdWx0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1MaXN0VGFnc0ZvclZhdWx0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yVmF1bHQoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpMaXN0VGFnc0ZvclZhdWx0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYWxsIHZhdWx0c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktdmF1bHRzLWdldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VmF1bHRzKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6TGlzdFZhdWx0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgb3BlcmF0aW9uIHB1cmNoYXNlcyBhIHByb3Zpc2lvbmVkIGNhcGFjaXR5IHVuaXQgZm9yIGFuIEFXUyBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25nbGFjaWVyL2xhdGVzdC9kZXYvYXBpLVB1cmNoYXNlUHJvdmlzaW9uZWRDYXBhY2l0eS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXJjaGFzZVByb3Zpc2lvbmVkQ2FwYWNpdHkoKSB7XG4gICAgdGhpcy50bygnZ2xhY2llcjpQdXJjaGFzZVByb3Zpc2lvbmVkQ2FwYWNpdHknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmVzIG9uZSBvciBtb3JlIHRhZ3MgZnJvbSB0aGUgc2V0IG9mIHRhZ3MgYXR0YWNoZWQgdG8gYSB2YXVsdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktUmVtb3ZlVGFnc0Zyb21WYXVsdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZW1vdmVUYWdzRnJvbVZhdWx0KCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6UmVtb3ZlVGFnc0Zyb21WYXVsdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNldHMgYW5kIHRoZW4gZW5hY3RzIGEgZGF0YSByZXRyaWV2YWwgcG9saWN5IGluIHRoZSByZWdpb24gc3BlY2lmaWVkIGluIHRoZSBQVVQgcmVxdWVzdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktU2V0RGF0YVJldHJpZXZhbFBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXREYXRhUmV0cmlldmFsUG9saWN5KCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6U2V0RGF0YVJldHJpZXZhbFBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgYW4gYWNjZXNzIHBvbGljeSBmb3IgYSB2YXVsdCBhbmQgd2lsbCBvdmVyd3JpdGUgYW4gZXhpc3RpbmcgcG9saWN5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1TZXRWYXVsdEFjY2Vzc1BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRWYXVsdEFjY2Vzc1BvbGljeSgpIHtcbiAgICB0aGlzLnRvKCdnbGFjaWVyOlNldFZhdWx0QWNjZXNzUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ29uZmlndXJlcyB2YXVsdCBub3RpZmljYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktdmF1bHQtbm90aWZpY2F0aW9ucy1wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0VmF1bHROb3RpZmljYXRpb25zKCkge1xuICAgIHRoaXMudG8oJ2dsYWNpZXI6U2V0VmF1bHROb3RpZmljYXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBhcmNoaXZlIHRvIGEgdmF1bHRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FwaS1hcmNoaXZlLXBvc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBsb2FkQXJjaGl2ZSgpIHtcbiAgICB0aGlzLnRvKCdnbGFjaWVyOlVwbG9hZEFyY2hpdmUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGxvYWRzIGEgcGFydCBvZiBhbiBhcmNoaXZlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi9hcGktdXBsb2FkLXBhcnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBsb2FkTXVsdGlwYXJ0UGFydCgpIHtcbiAgICB0aGlzLnRvKCdnbGFjaWVyOlVwbG9hZE11bHRpcGFydFBhcnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBcIldyaXRlXCI6IFtcbiAgICAgIFwiQWJvcnRNdWx0aXBhcnRVcGxvYWRcIixcbiAgICAgIFwiQ29tcGxldGVNdWx0aXBhcnRVcGxvYWRcIixcbiAgICAgIFwiQ3JlYXRlVmF1bHRcIixcbiAgICAgIFwiRGVsZXRlQXJjaGl2ZVwiLFxuICAgICAgXCJEZWxldGVWYXVsdFwiLFxuICAgICAgXCJEZWxldGVWYXVsdE5vdGlmaWNhdGlvbnNcIixcbiAgICAgIFwiSW5pdGlhdGVKb2JcIixcbiAgICAgIFwiSW5pdGlhdGVNdWx0aXBhcnRVcGxvYWRcIixcbiAgICAgIFwiUHVyY2hhc2VQcm92aXNpb25lZENhcGFjaXR5XCIsXG4gICAgICBcIlNldFZhdWx0Tm90aWZpY2F0aW9uc1wiLFxuICAgICAgXCJVcGxvYWRBcmNoaXZlXCIsXG4gICAgICBcIlVwbG9hZE11bHRpcGFydFBhcnRcIlxuICAgIF0sXG4gICAgXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCI6IFtcbiAgICAgIFwiQWJvcnRWYXVsdExvY2tcIixcbiAgICAgIFwiQ29tcGxldGVWYXVsdExvY2tcIixcbiAgICAgIFwiRGVsZXRlVmF1bHRBY2Nlc3NQb2xpY3lcIixcbiAgICAgIFwiSW5pdGlhdGVWYXVsdExvY2tcIixcbiAgICAgIFwiU2V0RGF0YVJldHJpZXZhbFBvbGljeVwiLFxuICAgICAgXCJTZXRWYXVsdEFjY2Vzc1BvbGljeVwiXG4gICAgXSxcbiAgICBcIlRhZ2dpbmdcIjogW1xuICAgICAgXCJBZGRUYWdzVG9WYXVsdFwiLFxuICAgICAgXCJSZW1vdmVUYWdzRnJvbVZhdWx0XCJcbiAgICBdLFxuICAgIFwiUmVhZFwiOiBbXG4gICAgICBcIkRlc2NyaWJlSm9iXCIsXG4gICAgICBcIkRlc2NyaWJlVmF1bHRcIixcbiAgICAgIFwiR2V0RGF0YVJldHJpZXZhbFBvbGljeVwiLFxuICAgICAgXCJHZXRKb2JPdXRwdXRcIixcbiAgICAgIFwiR2V0VmF1bHRBY2Nlc3NQb2xpY3lcIixcbiAgICAgIFwiR2V0VmF1bHRMb2NrXCIsXG4gICAgICBcIkdldFZhdWx0Tm90aWZpY2F0aW9uc1wiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJMaXN0Sm9ic1wiLFxuICAgICAgXCJMaXN0TXVsdGlwYXJ0VXBsb2Fkc1wiLFxuICAgICAgXCJMaXN0UGFydHNcIixcbiAgICAgIFwiTGlzdFByb3Zpc2lvbmVkQ2FwYWNpdHlcIixcbiAgICAgIFwiTGlzdFRhZ3NGb3JWYXVsdFwiLFxuICAgICAgXCJMaXN0VmF1bHRzXCJcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHZhdWx0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmdsYWNpZXIvbGF0ZXN0L2Rldi93b3JraW5nLXdpdGgtdmF1bHRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHZhdWx0TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB2YXVsdE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25WYXVsdCh2YXVsdE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06Z2xhY2llcjoke1JlZ2lvbn06JHtBY2NvdW50fTp2YXVsdHMvJHtWYXVsdE5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtWYXVsdE5hbWV9JywgdmF1bHROYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBIb3cgbG9uZyBhbiBhcmNoaXZlIGhhcyBiZWVuIHN0b3JlZCBpbiB0aGUgdmF1bHQsIGluIGRheXMuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25nbGFjaWVyL2xhdGVzdC9kZXYvYWNjZXNzLWNvbnRyb2wtb3ZlcnZpZXcuaHRtbCNzcGVjaWZ5aW5nLWNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0RlbGV0ZUFyY2hpdmUoKVxuICAgKiAtIC50b0luaXRpYXRlSm9iKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBcmNoaXZlQWdlSW5EYXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBnbGFjaWVyOkFyY2hpdmVBZ2VJbkRheXNgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBIGN1c3RvbWVyLWRlZmluZWQgdGFnLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZ2xhY2llci9sYXRlc3QvZGV2L2FjY2Vzcy1jb250cm9sLW92ZXJ2aWV3Lmh0bWwjc3BlY2lmeWluZy1jb25kaXRpb25zXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVzb3VyY2VUYWdFeGlzdHModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGdsYWNpZXI6UmVzb3VyY2VUYWcvYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cbn1cbiJdfQ==