"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Guardduty = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [guardduty](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonguardduty.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Guardduty extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [guardduty](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonguardduty.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'guardduty';
        this.accessLevelList = {
            "Write": [
                "AcceptInvitation",
                "ArchiveFindings",
                "CreateDetector",
                "CreateFilter",
                "CreateIPSet",
                "CreateMembers",
                "CreatePublishingDestination",
                "CreateSampleFindings",
                "CreateThreatIntelSet",
                "DeclineInvitations",
                "DeleteDetector",
                "DeleteFilter",
                "DeleteIPSet",
                "DeleteInvitations",
                "DeleteMembers",
                "DeletePublishingDestination",
                "DeleteThreatIntelSet",
                "DisableOrganizationAdminAccount",
                "DisassociateFromMasterAccount",
                "DisassociateMembers",
                "EnableOrganizationAdminAccount",
                "InviteMembers",
                "StartMonitoringMembers",
                "StopMonitoringMembers",
                "TagResource",
                "UnarchiveFindings",
                "UntagResource",
                "UpdateDetector",
                "UpdateFilter",
                "UpdateFindingsFeedback",
                "UpdateIPSet",
                "UpdateOrganizationConfiguration",
                "UpdatePublishingDestination",
                "UpdateThreatIntelSet"
            ],
            "Read": [
                "DescribeOrganizationConfiguration",
                "DescribePublishingDestination",
                "GetDetector",
                "GetFilter",
                "GetFindings",
                "GetFindingsStatistics",
                "GetIPSet",
                "GetInvitationsCount",
                "GetMasterAccount",
                "GetMembers",
                "GetThreatIntelSet"
            ],
            "List": [
                "ListDetectors",
                "ListFilters",
                "ListFindings",
                "ListIPSets",
                "ListInvitations",
                "ListMembers",
                "ListOrganizationAdminAccounts",
                "ListPublishingDestinations",
                "ListTagsForResource",
                "ListThreatIntelSets"
            ]
        };
    }
    /**
     * Grants permission to accept invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        this.to('guardduty:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to archive GuardDuty findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ArchiveFindings.html
     */
    toArchiveFindings() {
        this.to('guardduty:ArchiveFindings');
        return this;
    }
    /**
     * Grants permission to create a detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateDetector.html
     */
    toCreateDetector() {
        this.to('guardduty:CreateDetector');
        return this;
    }
    /**
     * Grants permission to create GuardDuty filters. A filters defines finding attributes and conditions used to filter findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateFilter.html
     */
    toCreateFilter() {
        this.to('guardduty:CreateFilter');
        return this;
    }
    /**
     * Grants permission to create an IPSet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        this.to('guardduty:CreateIPSet');
        return this;
    }
    /**
     * Grants permission to create GuardDuty member accounts, where the account used to create a member becomes the GuardDuty administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        this.to('guardduty:CreateMembers');
        return this;
    }
    /**
     * Grants permission to create a publishing destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreatePublishingDestination.html
     */
    toCreatePublishingDestination() {
        this.to('guardduty:CreatePublishingDestination');
        return this;
    }
    /**
     * Grants permission to create sample findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateSampleFindings.html
     */
    toCreateSampleFindings() {
        this.to('guardduty:CreateSampleFindings');
        return this;
    }
    /**
     * Grants permission to create GuardDuty ThreatIntelSets, where a ThreatIntelSet consists of known malicious IP addresses used by GuardDuty to generate findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatIntelSet.html
     */
    toCreateThreatIntelSet() {
        this.to('guardduty:CreateThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to decline invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        this.to('guardduty:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteDetector.html
     */
    toDeleteDetector() {
        this.to('guardduty:DeleteDetector');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty filters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteFilter.html
     */
    toDeleteFilter() {
        this.to('guardduty:DeleteFilter');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty IPSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        this.to('guardduty:DeleteIPSet');
        return this;
    }
    /**
     * Grants permission to delete invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        this.to('guardduty:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        this.to('guardduty:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to delete a publishing destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeletePublishingDestination.html
     */
    toDeletePublishingDestination() {
        this.to('guardduty:DeletePublishingDestination');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty ThreatIntelSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatIntelSet.html
     */
    toDeleteThreatIntelSet() {
        this.to('guardduty:DeleteThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to retrieve details about the delegated administrator associated with a GuardDuty detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        this.to('guardduty:DescribeOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve details about a publishing destination
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribePublishingDestination.html
     */
    toDescribePublishingDestination() {
        this.to('guardduty:DescribePublishingDestination');
        return this;
    }
    /**
     * Grants permission to disable the organization delegated administrator for GuardDuty
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisableOrganizationAdminAccount.html
     */
    toDisableOrganizationAdminAccount() {
        this.to('guardduty:DisableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to disassociate a GuardDuty member account from its GuardDuty master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        this.to('guardduty:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants permission to disassociate GuardDuty member accounts from their master GuardDuty account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        this.to('guardduty:DisassociateMembers');
        return this;
    }
    /**
     * Grants permission to enable an organization delegated administrator for GuardDuty
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_EnableOrganizationAdminAccount.html
     */
    toEnableOrganizationAdminAccount() {
        this.to('guardduty:EnableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty detectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetDetector.html
     */
    toGetDetector() {
        this.to('guardduty:GetDetector');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFilter.html
     */
    toGetFilter() {
        this.to('guardduty:GetFilter');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindings.html
     */
    toGetFindings() {
        this.to('guardduty:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty finding statistics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindingsStatistics.html
     */
    toGetFindingsStatistics() {
        this.to('guardduty:GetFindingsStatistics');
        return this;
    }
    /**
     * Grants permsission to retrieve GuardDuty IPSets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        this.to('guardduty:GetIPSet');
        return this;
    }
    /**
     * Grants permission to retrieve the count of all GuardDuty invitations sent to a specified account, which does not include the accepted invitation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        this.to('guardduty:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve details of the GuardDuty master account associated with a member account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        this.to('guardduty:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the member accounts associated with a master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        this.to('guardduty:GetMembers');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty ThreatIntelSets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatIntelSet.html
     */
    toGetThreatIntelSet() {
        this.to('guardduty:GetThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to invite other AWS accounts to enable GuardDuty and become GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        this.to('guardduty:InviteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty detectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html
     */
    toListDetectors() {
        this.to('guardduty:ListDetectors');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFilters.html
     */
    toListFilters() {
        this.to('guardduty:ListFilters');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFindings.html
     */
    toListFindings() {
        this.to('guardduty:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty IPSets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListIPSets.html
     */
    toListIPSets() {
        this.to('guardduty:ListIPSets');
        return this;
    }
    /**
     * Grants permission to retrieve a lists of all of the GuardDuty membership invitations that were sent to an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        this.to('guardduty:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrierve a lsit of GuardDuty member accounts associated with a master account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        this.to('guardduty:ListMembers');
        return this;
    }
    /**
     * Grants permission to list details about the organization delegated administrator for GuardDuty
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListOrganizationAdminAccounts.html
     */
    toListOrganizationAdminAccounts() {
        this.to('guardduty:ListOrganizationAdminAccounts');
        return this;
    }
    /**
     * Grants permission to retrieve a list of publishing destinations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListPublishingDestinations.html
     */
    toListPublishingDestinations() {
        this.to('guardduty:ListPublishingDestinations');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags associated with a GuardDuty resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('guardduty:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty ThreatIntelSets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatIntelSets.html
     */
    toListThreatIntelSets() {
        this.to('guardduty:ListThreatIntelSets');
        return this;
    }
    /**
     * Grants permission to a GuardDuty administrator account to monitor findings from GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMonitoringMembers.html
     */
    toStartMonitoringMembers() {
        this.to('guardduty:StartMonitoringMembers');
        return this;
    }
    /**
     * Grants permission to disable monitoring findings from member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StopMonitoringMembers.html
     */
    toStopMonitoringMembers() {
        this.to('guardduty:StopMonitoringMembers');
        return this;
    }
    /**
     * Grants permission to add tags to a GuardDuty resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('guardduty:TagResource');
        return this;
    }
    /**
     * Grants permission to unarchive GuardDuty findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UnarchiveFindings.html
     */
    toUnarchiveFindings() {
        this.to('guardduty:UnarchiveFindings');
        return this;
    }
    /**
     * Grants permission to remove tags from a GuardDuty resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('guardduty:UntagResource');
        return this;
    }
    /**
     * Grants permission to update GuardDuty detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateDetector.html
     */
    toUpdateDetector() {
        this.to('guardduty:UpdateDetector');
        return this;
    }
    /**
     * Grants permission to updates GuardDuty filters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFilter.html
     */
    toUpdateFilter() {
        this.to('guardduty:UpdateFilter');
        return this;
    }
    /**
     * Grants permission to update findings feedback to mark GuardDuty findings as useful or not useful
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFindingsFeedback.html
     */
    toUpdateFindingsFeedback() {
        this.to('guardduty:UpdateFindingsFeedback');
        return this;
    }
    /**
     * Grants permission to update GuardDuty IPSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        this.to('guardduty:UpdateIPSet');
        return this;
    }
    /**
     * Grants permission to update the delegated administrator configuration associated with a GuardDuty detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        this.to('guardduty:UpdateOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to update a publishing destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdatePublishingDestination.html
     */
    toUpdatePublishingDestination() {
        this.to('guardduty:UpdatePublishingDestination');
        return this;
    }
    /**
     * Grants permission to updates the GuardDuty ThreatIntelSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatIntelSet.html
     */
    toUpdateThreatIntelSet() {
        this.to('guardduty:UpdateThreatIntelSet');
        return this;
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(detectorId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type filter to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param filterName - Identifier for the filterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFilter(detectorId, filterName, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/filter/${FilterName}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${FilterName}', filterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param iPSetId - Identifier for the iPSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(detectorId, iPSetId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/ipset/${IPSetId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${IPSetId}', iPSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type threatintelset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param threatIntelSetId - Identifier for the threatIntelSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThreatintelset(detectorId, threatIntelSetId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${ThreatIntelSetId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${ThreatIntelSetId}', threatIntelSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type publishingDestination to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param publishingDestinationId - Identifier for the publishingDestinationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPublishingDestination(detectorId, publishingDestinationId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/publishingDestination/${PublishingDestinationId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${PublishingDestinationId}', publishingDestinationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Guardduty = Guardduty;
//# sourceMappingURL=data:application/json;base64,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