"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Honeycode = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [honeycode](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonhoneycode.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Honeycode extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [honeycode](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonhoneycode.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'honeycode';
        this.accessLevelList = {
            "Write": [
                "ApproveTeamAssociation",
                "InvokeScreenAutomation",
                "RejectTeamAssociation"
            ],
            "Read": [
                "GetScreenData"
            ],
            "List": [
                "ListTeamAssociations"
            ]
        };
    }
    /**
     * Grants permission to approve a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#approve-team-association
     */
    toApproveTeamAssociation() {
        this.to('honeycode:ApproveTeamAssociation');
        return this;
    }
    /**
     * Grants permission to load the data from a screen
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_GetScreenData.html
     */
    toGetScreenData() {
        this.to('honeycode:GetScreenData');
        return this;
    }
    /**
     * Grants permission to invoke a screen automation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_InvokeScreenAutomation.html
     */
    toInvokeScreenAutomation() {
        this.to('honeycode:InvokeScreenAutomation');
        return this;
    }
    /**
     * Grants permission to list all pending and approved team associations with your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#list-team-associations
     */
    toListTeamAssociations() {
        this.to('honeycode:ListTeamAssociations');
        return this;
    }
    /**
     * Grants permission to reject a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#reject-team-association
     */
    toRejectTeamAssociation() {
        this.to('honeycode:RejectTeamAssociation');
        return this;
    }
    /**
     * Adds a resource of type screen to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreen(workbookId, appId, screenId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type screen-automation to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen-automation.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param automationId - Identifier for the automationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreenAutomation(workbookId, appId, screenId, automationId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen-automation:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}/automation/${AutomationId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${AutomationId}', automationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Honeycode = Honeycode;
//# sourceMappingURL=data:application/json;base64,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