"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ivs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ivs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninteractivevideoservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ivs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ivs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninteractivevideoservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ivs';
        this.accessLevelList = {
            "Read": [
                "BatchGetChannel",
                "BatchGetStreamKey",
                "GetChannel",
                "GetPlaybackKeyPair",
                "GetStream",
                "GetStreamKey"
            ],
            "Write": [
                "CreateChannel",
                "CreateStreamKey",
                "DeleteChannel",
                "DeletePlaybackKeyPair",
                "DeleteStreamKey",
                "ImportPlaybackKeyPair",
                "PutMetadata",
                "StopStream",
                "UpdateChannel"
            ],
            "List": [
                "ListChannels",
                "ListPlaybackKeyPairs",
                "ListStreamKeys",
                "ListStreams"
            ],
            "Tagging": [
                "ListTagsForResource",
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to get multiple channels simultaneously by channel ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_BatchGetChannel.html
     */
    toBatchGetChannel() {
        this.to('ivs:BatchGetChannel');
        return this;
    }
    /**
     * Grants permission to get multiple stream keys simultaneously by stream key ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_BatchGetStreamKey.html
     */
    toBatchGetStreamKey() {
        this.to('ivs:BatchGetStreamKey');
        return this;
    }
    /**
     * Grants permission to create a new channel and an associated stream key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        this.to('ivs:CreateChannel');
        return this;
    }
    /**
     * Grants permission to create a stream key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_CreateStreamKey.html
     */
    toCreateStreamKey() {
        this.to('ivs:CreateStreamKey');
        return this;
    }
    /**
     * Grants permission to delete a channel and channel's stream keys.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeleteChannel.html
     */
    toDeleteChannel() {
        this.to('ivs:DeleteChannel');
        return this;
    }
    /**
     * Grants permission to delete the playback key pair for a specified ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeletePlaybackKeyPair.html
     */
    toDeletePlaybackKeyPair() {
        this.to('ivs:DeletePlaybackKeyPair');
        return this;
    }
    /**
     * Grants permission to delete the stream key for a specified ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeleteStreamKey.html
     */
    toDeleteStreamKey() {
        this.to('ivs:DeleteStreamKey');
        return this;
    }
    /**
     * Grants permission to get the channel configuration for a specified channel ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetChannel.html
     */
    toGetChannel() {
        this.to('ivs:GetChannel');
        return this;
    }
    /**
     * Grants permission to get the playback keypair information for a specified ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetPlaybackKeyPair.html
     */
    toGetPlaybackKeyPair() {
        this.to('ivs:GetPlaybackKeyPair');
        return this;
    }
    /**
     * Grants permission to get information about the active (live) stream on a specified channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetStream.html
     */
    toGetStream() {
        this.to('ivs:GetStream');
        return this;
    }
    /**
     * Grants permission to get stream-key information for a specified ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetStreamKey.html
     */
    toGetStreamKey() {
        this.to('ivs:GetStreamKey');
        return this;
    }
    /**
     * Grants permission to import the public key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ImportPlaybackKeyPair.html
     */
    toImportPlaybackKeyPair() {
        this.to('ivs:ImportPlaybackKeyPair');
        return this;
    }
    /**
     * Grants permission to get summary information about channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        this.to('ivs:ListChannels');
        return this;
    }
    /**
     * Grants permission to get summary information about playback key pairs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListPlaybackKeyPairs.html
     */
    toListPlaybackKeyPairs() {
        this.to('ivs:ListPlaybackKeyPairs');
        return this;
    }
    /**
     * Grants permission to get summary information about stream keys
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListStreamKeys.html
     */
    toListStreamKeys() {
        this.to('ivs:ListStreamKeys');
        return this;
    }
    /**
     * Grants permission to get summary information about live streams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListStreams.html
     */
    toListStreams() {
        this.to('ivs:ListStreams');
        return this;
    }
    /**
     * Grants permission to get information about the tags for a specified ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('ivs:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to insert metadata into an RTMP stream for a specified channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_PutMetadata.html
     */
    toPutMetadata() {
        this.to('ivs:PutMetadata');
        return this;
    }
    /**
     * Grants permission to disconnect a streamer on a specified channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_StopStream.html
     */
    toStopStream() {
        this.to('ivs:StopStream');
        return this;
    }
    /**
     * Grants permission to add or update tags for a resource with a specified ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('ivs:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags for a resource with a specified ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('ivs:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a channel's configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        this.to('ivs:UpdateChannel');
        return this;
    }
    /**
     * Adds a resource of type Channel to the statement
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_Channel.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(resourceId, account, partition) {
        var arn = 'arn:${Partition}:ivs::${Account}:channel/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Stream-Key to the statement
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_StreamKey.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamKey(resourceId, account, partition) {
        var arn = 'arn:${Partition}:ivs::${Account}:stream-key/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Playback-Key-Pair to the statement
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_PlaybackKeyPair.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPlaybackKeyPair(resourceId, account, partition) {
        var arn = 'arn:${Partition}:ivs::${Account}:playback-key/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Ivs = Ivs;
//# sourceMappingURL=data:application/json;base64,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