"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kendra = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kendra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkendra.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kendra extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kendra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkendra.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kendra';
        this.accessLevelList = {
            "Write": [
                "BatchDeleteDocument",
                "BatchPutDocument",
                "CreateDataSource",
                "CreateFaq",
                "CreateIndex",
                "DeleteDataSource",
                "DeleteFaq",
                "DeleteIndex",
                "StartDataSourceSyncJob",
                "StopDataSourceSyncJob",
                "SubmitFeedback",
                "UpdateDataSource",
                "UpdateIndex"
            ],
            "Read": [
                "DescribeDataSource",
                "DescribeFaq",
                "DescribeIndex",
                "Query"
            ],
            "List": [
                "ListDataSourceSyncJobs",
                "ListDataSources",
                "ListFaqs",
                "ListIndices",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Batch Delete document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_BatchDeleteDocument.html
     */
    toBatchDeleteDocument() {
        this.to('kendra:BatchDeleteDocument');
        return this;
    }
    /**
     * Batch put document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_BatchPutDocument.html
     */
    toBatchPutDocument() {
        this.to('kendra:BatchPutDocument');
        return this;
    }
    /**
     * Create a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateDataSource.html
     */
    toCreateDataSource() {
        this.to('kendra:CreateDataSource');
        return this;
    }
    /**
     * Create an Faq
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateFaq.html
     */
    toCreateFaq() {
        this.to('kendra:CreateFaq');
        return this;
    }
    /**
     * Create an Index
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateIndex.html
     */
    toCreateIndex() {
        this.to('kendra:CreateIndex');
        return this;
    }
    /**
     * Delete a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        this.to('kendra:DeleteDataSource');
        return this;
    }
    /**
     * Delete an Faq
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteFaq.html
     */
    toDeleteFaq() {
        this.to('kendra:DeleteFaq');
        return this;
    }
    /**
     * Delete an Index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteIndex.html
     */
    toDeleteIndex() {
        this.to('kendra:DeleteIndex');
        return this;
    }
    /**
     * Describe a data source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeDataSource.html
     */
    toDescribeDataSource() {
        this.to('kendra:DescribeDataSource');
        return this;
    }
    /**
     * Describe an Faq
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeFaq.html
     */
    toDescribeFaq() {
        this.to('kendra:DescribeFaq');
        return this;
    }
    /**
     * Describe an Index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeIndex.html
     */
    toDescribeIndex() {
        this.to('kendra:DescribeIndex');
        return this;
    }
    /**
     * Get Data Source sync job history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListDataSourceSyncJobs.html
     */
    toListDataSourceSyncJobs() {
        this.to('kendra:ListDataSourceSyncJobs');
        return this;
    }
    /**
     * List the data sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListDataSources.html
     */
    toListDataSources() {
        this.to('kendra:ListDataSources');
        return this;
    }
    /**
     * List the Faqs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListFaqs.html
     */
    toListFaqs() {
        this.to('kendra:ListFaqs');
        return this;
    }
    /**
     * List the indexes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListIndices.html
     */
    toListIndices() {
        this.to('kendra:ListIndices');
        return this;
    }
    /**
     * Lists tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('kendra:ListTagsForResource');
        return this;
    }
    /**
     * Query documents and Faqs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html
     */
    toQuery() {
        this.to('kendra:Query');
        return this;
    }
    /**
     * Start Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_StartDataSourceSyncJob.html
     */
    toStartDataSourceSyncJob() {
        this.to('kendra:StartDataSourceSyncJob');
        return this;
    }
    /**
     * Stop Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_StopDataSourceSyncJob.html
     */
    toStopDataSourceSyncJob() {
        this.to('kendra:StopDataSourceSyncJob');
        return this;
    }
    /**
     * Send feedback about a query results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_SubmitFeedback.html
     */
    toSubmitFeedback() {
        this.to('kendra:SubmitFeedback');
        return this;
    }
    /**
     * Tags a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.to('kendra:TagResource');
        return this;
    }
    /**
     * Removes the tag with the given key from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('kendra:UntagResource');
        return this;
    }
    /**
     * Update a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        this.to('kendra:UpdateDataSource');
        return this;
    }
    /**
     * Update an Index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateIndex.html
     */
    toUpdateIndex() {
        this.to('kendra:UpdateIndex');
        return this;
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/index.html
     *
     * @param indexId - Identifier for the indexId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(indexId, account, region, partition) {
        var arn = 'arn:${Partition}:kendra:${Region}:${Account}:index/${IndexId}';
        arn = arn.replace('${IndexId}', indexId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type data-source to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/data-source.html
     *
     * @param indexId - Identifier for the indexId.
     * @param dataSourceId - Identifier for the dataSourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataSource(indexId, dataSourceId, account, region, partition) {
        var arn = 'arn:${Partition}:kendra:${Region}:${Account}:index/${IndexId}/data-source/${DataSourceId}';
        arn = arn.replace('${IndexId}', indexId);
        arn = arn.replace('${DataSourceId}', dataSourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type faq to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/faq.html
     *
     * @param indexId - Identifier for the indexId.
     * @param faqId - Identifier for the faqId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFaq(indexId, faqId, account, region, partition) {
        var arn = 'arn:${Partition}:kendra:${Region}:${Account}:index/${IndexId}/faq/${FaqId}';
        arn = arn.replace('${IndexId}', indexId);
        arn = arn.replace('${FaqId}', faqId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kendra = Kendra;
//# sourceMappingURL=data:application/json;base64,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