"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lightsail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lightsail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlightsail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lightsail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lightsail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlightsail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lightsail';
        this.accessLevelList = {
            "Write": [
                "AllocateStaticIp",
                "AttachDisk",
                "AttachInstancesToLoadBalancer",
                "AttachLoadBalancerTlsCertificate",
                "AttachStaticIp",
                "CloseInstancePublicPorts",
                "CopySnapshot",
                "CreateCloudFormationStack",
                "CreateDisk",
                "CreateDiskFromSnapshot",
                "CreateDiskSnapshot",
                "CreateDomain",
                "CreateDomainEntry",
                "CreateInstanceSnapshot",
                "CreateInstances",
                "CreateInstancesFromSnapshot",
                "CreateKeyPair",
                "CreateLoadBalancer",
                "CreateLoadBalancerTlsCertificate",
                "CreateRelationalDatabase",
                "CreateRelationalDatabaseFromSnapshot",
                "CreateRelationalDatabaseSnapshot",
                "DeleteDisk",
                "DeleteDiskSnapshot",
                "DeleteDomain",
                "DeleteDomainEntry",
                "DeleteInstance",
                "DeleteInstanceSnapshot",
                "DeleteKeyPair",
                "DeleteKnownHostKeys",
                "DeleteLoadBalancer",
                "DeleteLoadBalancerTlsCertificate",
                "DeleteRelationalDatabase",
                "DeleteRelationalDatabaseSnapshot",
                "DetachDisk",
                "DetachInstancesFromLoadBalancer",
                "DetachStaticIp",
                "DownloadDefaultKeyPair",
                "ExportSnapshot",
                "GetInstanceAccessDetails",
                "GetRelationalDatabaseMasterUserPassword",
                "ImportKeyPair",
                "OpenInstancePublicPorts",
                "PeerVpc",
                "PutInstancePublicPorts",
                "RebootInstance",
                "RebootRelationalDatabase",
                "ReleaseStaticIp",
                "StartInstance",
                "StartRelationalDatabase",
                "StopInstance",
                "StopRelationalDatabase",
                "TagResource",
                "UnpeerVpc",
                "UntagResource",
                "UpdateDomainEntry",
                "UpdateLoadBalancerAttribute",
                "UpdateRelationalDatabase",
                "UpdateRelationalDatabaseParameters"
            ],
            "Read": [
                "GetActiveNames",
                "GetDisk",
                "GetDiskSnapshot",
                "GetDomain",
                "GetDomains",
                "GetInstance",
                "GetInstanceMetricData",
                "GetInstancePortStates",
                "GetInstanceSnapshot",
                "GetInstanceState",
                "GetInstances",
                "GetKeyPairs",
                "GetLoadBalancer",
                "GetLoadBalancerMetricData",
                "GetLoadBalancerTlsCertificates",
                "GetLoadBalancers",
                "GetOperation",
                "GetOperations",
                "GetOperationsForResource",
                "GetRelationalDatabaseEvents",
                "GetRelationalDatabaseLogEvents",
                "GetRelationalDatabaseLogStreams",
                "GetRelationalDatabaseMetricData",
                "GetRelationalDatabases",
                "GetStaticIp",
                "GetStaticIps",
                "IsVpcPeered"
            ],
            "List": [
                "GetBlueprints",
                "GetBundles",
                "GetCloudFormationStackRecords",
                "GetDiskSnapshots",
                "GetDisks",
                "GetExportSnapshotRecords",
                "GetInstanceSnapshots",
                "GetKeyPair",
                "GetRegions",
                "GetRelationalDatabase",
                "GetRelationalDatabaseBlueprints",
                "GetRelationalDatabaseBundles",
                "GetRelationalDatabaseParameters",
                "GetRelationalDatabaseSnapshot",
                "GetRelationalDatabaseSnapshots"
            ]
        };
    }
    /**
     * Creates a static IP address that can be attached to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html
     */
    toAllocateStaticIp() {
        this.to('lightsail:AllocateStaticIp');
        return this;
    }
    /**
     * Attaches a disk to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html
     */
    toAttachDisk() {
        this.to('lightsail:AttachDisk');
        return this;
    }
    /**
     * Attaches one or more instances to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html
     */
    toAttachInstancesToLoadBalancer() {
        this.to('lightsail:AttachInstancesToLoadBalancer');
        return this;
    }
    /**
     * Attaches a TLS certificate to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html
     */
    toAttachLoadBalancerTlsCertificate() {
        this.to('lightsail:AttachLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Attaches a static IP address to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html
     */
    toAttachStaticIp() {
        this.to('lightsail:AttachStaticIp');
        return this;
    }
    /**
     * Closes a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html
     */
    toCloseInstancePublicPorts() {
        this.to('lightsail:CloseInstancePublicPorts');
        return this;
    }
    /**
     * Copies a snapshot from one AWS Region to another in Amazon Lightsail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.to('lightsail:CopySnapshot');
        return this;
    }
    /**
     * Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html
     */
    toCreateCloudFormationStack() {
        this.to('lightsail:CreateCloudFormationStack');
        return this;
    }
    /**
     * Creates a disk.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html
     */
    toCreateDisk() {
        this.to('lightsail:CreateDisk');
        return this;
    }
    /**
     * Creates a disk from snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html
     */
    toCreateDiskFromSnapshot() {
        this.to('lightsail:CreateDiskFromSnapshot');
        return this;
    }
    /**
     * Creates a disk snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html
     */
    toCreateDiskSnapshot() {
        this.to('lightsail:CreateDiskSnapshot');
        return this;
    }
    /**
     * Creates a domain resource for the specified domain name.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html
     */
    toCreateDomain() {
        this.to('lightsail:CreateDomain');
        return this;
    }
    /**
     * Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html
     */
    toCreateDomainEntry() {
        this.to('lightsail:CreateDomainEntry');
        return this;
    }
    /**
     * Creates an instance snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html
     */
    toCreateInstanceSnapshot() {
        this.to('lightsail:CreateInstanceSnapshot');
        return this;
    }
    /**
     * Creates one or more instances.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html
     */
    toCreateInstances() {
        this.to('lightsail:CreateInstances');
        return this;
    }
    /**
     * Creates one or more instances based on an instance snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html
     */
    toCreateInstancesFromSnapshot() {
        this.to('lightsail:CreateInstancesFromSnapshot');
        return this;
    }
    /**
     * Creates a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html
     */
    toCreateKeyPair() {
        this.to('lightsail:CreateKeyPair');
        return this;
    }
    /**
     * Creates a load balancer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html
     */
    toCreateLoadBalancer() {
        this.to('lightsail:CreateLoadBalancer');
        return this;
    }
    /**
     * Creates a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html
     */
    toCreateLoadBalancerTlsCertificate() {
        this.to('lightsail:CreateLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Creates a new relational database.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html
     */
    toCreateRelationalDatabase() {
        this.to('lightsail:CreateRelationalDatabase');
        return this;
    }
    /**
     * Creates a new relational database from a snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html
     */
    toCreateRelationalDatabaseFromSnapshot() {
        this.to('lightsail:CreateRelationalDatabaseFromSnapshot');
        return this;
    }
    /**
     * Creates a relational database snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html
     */
    toCreateRelationalDatabaseSnapshot() {
        this.to('lightsail:CreateRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Deletes a disk.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html
     */
    toDeleteDisk() {
        this.to('lightsail:DeleteDisk');
        return this;
    }
    /**
     * Deletes a disk snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html
     */
    toDeleteDiskSnapshot() {
        this.to('lightsail:DeleteDiskSnapshot');
        return this;
    }
    /**
     * Deletes a domain resource and all of its DNS records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.to('lightsail:DeleteDomain');
        return this;
    }
    /**
     * Deletes a DNS record entry for a domain resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html
     */
    toDeleteDomainEntry() {
        this.to('lightsail:DeleteDomainEntry');
        return this;
    }
    /**
     * Deletes an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html
     */
    toDeleteInstance() {
        this.to('lightsail:DeleteInstance');
        return this;
    }
    /**
     * Deletes an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html
     */
    toDeleteInstanceSnapshot() {
        this.to('lightsail:DeleteInstanceSnapshot');
        return this;
    }
    /**
     * Deletes a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html
     */
    toDeleteKeyPair() {
        this.to('lightsail:DeleteKeyPair');
        return this;
    }
    /**
     * Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html
     */
    toDeleteKnownHostKeys() {
        this.to('lightsail:DeleteKnownHostKeys');
        return this;
    }
    /**
     * Deletes a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html
     */
    toDeleteLoadBalancer() {
        this.to('lightsail:DeleteLoadBalancer');
        return this;
    }
    /**
     * Deletes a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    toDeleteLoadBalancerTlsCertificate() {
        this.to('lightsail:DeleteLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Deletes a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html
     */
    toDeleteRelationalDatabase() {
        this.to('lightsail:DeleteRelationalDatabase');
        return this;
    }
    /**
     * Deletes relational database snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html
     */
    toDeleteRelationalDatabaseSnapshot() {
        this.to('lightsail:DeleteRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Detaches a disk from an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html
     */
    toDetachDisk() {
        this.to('lightsail:DetachDisk');
        return this;
    }
    /**
     * Detaches one or more instances from a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    toDetachInstancesFromLoadBalancer() {
        this.to('lightsail:DetachInstancesFromLoadBalancer');
        return this;
    }
    /**
     * Detaches a static IP from an instance to which it is attached.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html
     */
    toDetachStaticIp() {
        this.to('lightsail:DetachStaticIp');
        return this;
    }
    /**
     * Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html
     */
    toDownloadDefaultKeyPair() {
        this.to('lightsail:DownloadDefaultKeyPair');
        return this;
    }
    /**
     * Exports an Amazon Lightsail snapshot to Amazon EC2.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html
     */
    toExportSnapshot() {
        this.to('lightsail:ExportSnapshot');
        return this;
    }
    /**
     * Returns the names of all active (not deleted) resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html
     */
    toGetActiveNames() {
        this.to('lightsail:GetActiveNames');
        return this;
    }
    /**
     * Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html
     */
    toGetBlueprints() {
        this.to('lightsail:GetBlueprints');
        return this;
    }
    /**
     * Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html
     */
    toGetBundles() {
        this.to('lightsail:GetBundles');
        return this;
    }
    /**
     * Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html
     */
    toGetCloudFormationStackRecords() {
        this.to('lightsail:GetCloudFormationStackRecords');
        return this;
    }
    /**
     * Returns information about a disk.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html
     */
    toGetDisk() {
        this.to('lightsail:GetDisk');
        return this;
    }
    /**
     * Returns information about a disk snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html
     */
    toGetDiskSnapshot() {
        this.to('lightsail:GetDiskSnapshot');
        return this;
    }
    /**
     * Returns information about all disk snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html
     */
    toGetDiskSnapshots() {
        this.to('lightsail:GetDiskSnapshots');
        return this;
    }
    /**
     * Returns information about all disks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html
     */
    toGetDisks() {
        this.to('lightsail:GetDisks');
        return this;
    }
    /**
     * Returns DNS records for a domain resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html
     */
    toGetDomain() {
        this.to('lightsail:GetDomain');
        return this;
    }
    /**
     * Returns DNS records for all domain resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html
     */
    toGetDomains() {
        this.to('lightsail:GetDomains');
        return this;
    }
    /**
     * Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html
     */
    toGetExportSnapshotRecords() {
        this.to('lightsail:GetExportSnapshotRecords');
        return this;
    }
    /**
     * Returns information about an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html
     */
    toGetInstance() {
        this.to('lightsail:GetInstance');
        return this;
    }
    /**
     * Returns temporary keys you can use to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html
     */
    toGetInstanceAccessDetails() {
        this.to('lightsail:GetInstanceAccessDetails');
        return this;
    }
    /**
     * Returns the data points for the specified metric of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html
     */
    toGetInstanceMetricData() {
        this.to('lightsail:GetInstanceMetricData');
        return this;
    }
    /**
     * Returns the port states of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html
     */
    toGetInstancePortStates() {
        this.to('lightsail:GetInstancePortStates');
        return this;
    }
    /**
     * Returns information about an instance snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html
     */
    toGetInstanceSnapshot() {
        this.to('lightsail:GetInstanceSnapshot');
        return this;
    }
    /**
     * Returns information about all instance snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html
     */
    toGetInstanceSnapshots() {
        this.to('lightsail:GetInstanceSnapshots');
        return this;
    }
    /**
     * Returns the state of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html
     */
    toGetInstanceState() {
        this.to('lightsail:GetInstanceState');
        return this;
    }
    /**
     * Returns information about all instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html
     */
    toGetInstances() {
        this.to('lightsail:GetInstances');
        return this;
    }
    /**
     * Returns information about a key pair.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html
     */
    toGetKeyPair() {
        this.to('lightsail:GetKeyPair');
        return this;
    }
    /**
     * Returns information about all key pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html
     */
    toGetKeyPairs() {
        this.to('lightsail:GetKeyPairs');
        return this;
    }
    /**
     * Returns information about a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html
     */
    toGetLoadBalancer() {
        this.to('lightsail:GetLoadBalancer');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html
     */
    toGetLoadBalancerMetricData() {
        this.to('lightsail:GetLoadBalancerMetricData');
        return this;
    }
    /**
     * Returns information about a load balancer TLS certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    toGetLoadBalancerTlsCertificates() {
        this.to('lightsail:GetLoadBalancerTlsCertificates');
        return this;
    }
    /**
     * Returns information about load balancers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html
     */
    toGetLoadBalancers() {
        this.to('lightsail:GetLoadBalancers');
        return this;
    }
    /**
     * Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html
     */
    toGetOperation() {
        this.to('lightsail:GetOperation');
        return this;
    }
    /**
     * Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html
     */
    toGetOperations() {
        this.to('lightsail:GetOperations');
        return this;
    }
    /**
     * Returns operations for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html
     */
    toGetOperationsForResource() {
        this.to('lightsail:GetOperationsForResource');
        return this;
    }
    /**
     * Returns a list of all valid AWS Regions for Amazon Lightsail.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html
     */
    toGetRegions() {
        this.to('lightsail:GetRegions');
        return this;
    }
    /**
     * Returns information about a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html
     */
    toGetRelationalDatabase() {
        this.to('lightsail:GetRelationalDatabase');
        return this;
    }
    /**
     * Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html
     */
    toGetRelationalDatabaseBlueprints() {
        this.to('lightsail:GetRelationalDatabaseBlueprints');
        return this;
    }
    /**
     * Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html
     */
    toGetRelationalDatabaseBundles() {
        this.to('lightsail:GetRelationalDatabaseBundles');
        return this;
    }
    /**
     * Returns events for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html
     */
    toGetRelationalDatabaseEvents() {
        this.to('lightsail:GetRelationalDatabaseEvents');
        return this;
    }
    /**
     * Returns events for the specified log stream of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html
     */
    toGetRelationalDatabaseLogEvents() {
        this.to('lightsail:GetRelationalDatabaseLogEvents');
        return this;
    }
    /**
     * Returns the log streams available for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html
     */
    toGetRelationalDatabaseLogStreams() {
        this.to('lightsail:GetRelationalDatabaseLogStreams');
        return this;
    }
    /**
     * Returns the master user password of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html
     */
    toGetRelationalDatabaseMasterUserPassword() {
        this.to('lightsail:GetRelationalDatabaseMasterUserPassword');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html
     */
    toGetRelationalDatabaseMetricData() {
        this.to('lightsail:GetRelationalDatabaseMetricData');
        return this;
    }
    /**
     * Returns the parameters of a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html
     */
    toGetRelationalDatabaseParameters() {
        this.to('lightsail:GetRelationalDatabaseParameters');
        return this;
    }
    /**
     * Returns information about a relational database snapshot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html
     */
    toGetRelationalDatabaseSnapshot() {
        this.to('lightsail:GetRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Returns information about all relational database snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html
     */
    toGetRelationalDatabaseSnapshots() {
        this.to('lightsail:GetRelationalDatabaseSnapshots');
        return this;
    }
    /**
     * Return information about all relational databases.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html
     */
    toGetRelationalDatabases() {
        this.to('lightsail:GetRelationalDatabases');
        return this;
    }
    /**
     * Returns information about a static IP.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html
     */
    toGetStaticIp() {
        this.to('lightsail:GetStaticIp');
        return this;
    }
    /**
     * Returns information about all static IPs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html
     */
    toGetStaticIps() {
        this.to('lightsail:GetStaticIps');
        return this;
    }
    /**
     * Imports a public key from a key pair.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html
     */
    toImportKeyPair() {
        this.to('lightsail:ImportKeyPair');
        return this;
    }
    /**
     * Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html
     */
    toIsVpcPeered() {
        this.to('lightsail:IsVpcPeered');
        return this;
    }
    /**
     * Adds, or opens a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html
     */
    toOpenInstancePublicPorts() {
        this.to('lightsail:OpenInstancePublicPorts');
        return this;
    }
    /**
     * Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html
     */
    toPeerVpc() {
        this.to('lightsail:PeerVpc');
        return this;
    }
    /**
     * Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html
     */
    toPutInstancePublicPorts() {
        this.to('lightsail:PutInstancePublicPorts');
        return this;
    }
    /**
     * Reboots an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html
     */
    toRebootInstance() {
        this.to('lightsail:RebootInstance');
        return this;
    }
    /**
     * Reboots a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html
     */
    toRebootRelationalDatabase() {
        this.to('lightsail:RebootRelationalDatabase');
        return this;
    }
    /**
     * Deletes a static IP.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html
     */
    toReleaseStaticIp() {
        this.to('lightsail:ReleaseStaticIp');
        return this;
    }
    /**
     * Starts an instance that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html
     */
    toStartInstance() {
        this.to('lightsail:StartInstance');
        return this;
    }
    /**
     * Starts a relational database that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html
     */
    toStartRelationalDatabase() {
        this.to('lightsail:StartRelationalDatabase');
        return this;
    }
    /**
     * Stops an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html
     */
    toStopInstance() {
        this.to('lightsail:StopInstance');
        return this;
    }
    /**
     * Stops a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html
     */
    toStopRelationalDatabase() {
        this.to('lightsail:StopRelationalDatabase');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html
     */
    toTagResource() {
        this.to('lightsail:TagResource');
        return this;
    }
    /**
     * Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html
     */
    toUnpeerVpc() {
        this.to('lightsail:UnpeerVpc');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('lightsail:UntagResource');
        return this;
    }
    /**
     * Updates a domain recordset after it is created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html
     */
    toUpdateDomainEntry() {
        this.to('lightsail:UpdateDomainEntry');
        return this;
    }
    /**
     * Updates a load balancer attribute, such as the health check path and session stickiness.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    toUpdateLoadBalancerAttribute() {
        this.to('lightsail:UpdateLoadBalancerAttribute');
        return this;
    }
    /**
     * Updates a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html
     */
    toUpdateRelationalDatabase() {
        this.to('lightsail:UpdateRelationalDatabase');
        return this;
    }
    /**
     * Updates the parameters of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html
     */
    toUpdateRelationalDatabaseParameters() {
        this.to('lightsail:UpdateRelationalDatabaseParameters');
        return this;
    }
    /**
     * Adds a resource of type Domain to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstance(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type InstanceSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstanceSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type KeyPair to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKeyPair(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type StaticIp to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStaticIp(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Disk to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDisk(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type DiskSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDiskSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancer to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLoadBalancer(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type PeeredVpc to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPeeredVpc(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancerTlsCertificate to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadBalancerTlsCertificate(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ExportSnapshotRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExportSnapshotRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CloudFormationStackRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCloudFormationStackRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabase to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRelationalDatabase(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabaseSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRelationalDatabaseSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Lightsail = Lightsail;
//# sourceMappingURL=data:application/json;base64,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