"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Machinelearning = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [machinelearning](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmachinelearning.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Machinelearning extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [machinelearning](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmachinelearning.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'machinelearning';
        this.accessLevelList = {
            "Tagging": [
                "AddTags",
                "DeleteTags"
            ],
            "Write": [
                "CreateBatchPrediction",
                "CreateDataSourceFromRDS",
                "CreateDataSourceFromRedshift",
                "CreateDataSourceFromS3",
                "CreateEvaluation",
                "CreateMLModel",
                "CreateRealtimeEndpoint",
                "DeleteBatchPrediction",
                "DeleteDataSource",
                "DeleteEvaluation",
                "DeleteMLModel",
                "DeleteRealtimeEndpoint",
                "Predict",
                "UpdateBatchPrediction",
                "UpdateDataSource",
                "UpdateEvaluation",
                "UpdateMLModel"
            ],
            "List": [
                "DescribeBatchPredictions",
                "DescribeDataSources",
                "DescribeEvaluations",
                "DescribeMLModels",
                "DescribeTags"
            ],
            "Read": [
                "GetBatchPrediction",
                "GetDataSource",
                "GetEvaluation",
                "GetMLModel"
            ]
        };
    }
    /**
     * Adds one or more tags to an object, up to a limit of 10. Each tag consists of a key and an optional value
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        this.to('machinelearning:AddTags');
        return this;
    }
    /**
     * Generates predictions for a group of observations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_CreateBatchPrediction.html
     */
    toCreateBatchPrediction() {
        this.to('machinelearning:CreateBatchPrediction');
        return this;
    }
    /**
     * Creates a DataSource object from an Amazon RDS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_CreateDataSourceFromRDS.html
     */
    toCreateDataSourceFromRDS() {
        this.to('machinelearning:CreateDataSourceFromRDS');
        return this;
    }
    /**
     * Creates a DataSource from a database hosted on an Amazon Redshift cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_CreateDataSourceFromRedshift.html
     */
    toCreateDataSourceFromRedshift() {
        this.to('machinelearning:CreateDataSourceFromRedshift');
        return this;
    }
    /**
     * Creates a DataSource object from S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_CreateDataSourceFromS3.html
     */
    toCreateDataSourceFromS3() {
        this.to('machinelearning:CreateDataSourceFromS3');
        return this;
    }
    /**
     * Creates a new Evaluation of an MLModel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_CreateEvaluation.html
     */
    toCreateEvaluation() {
        this.to('machinelearning:CreateEvaluation');
        return this;
    }
    /**
     * Creates a new MLModel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_CreateMLModel.html
     */
    toCreateMLModel() {
        this.to('machinelearning:CreateMLModel');
        return this;
    }
    /**
     * Creates a real-time endpoint for the MLModel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_CreateRealtimeEndpoint.html
     */
    toCreateRealtimeEndpoint() {
        this.to('machinelearning:CreateRealtimeEndpoint');
        return this;
    }
    /**
     * Assigns the DELETED status to a BatchPrediction, rendering it unusable
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DeleteBatchPrediction.html
     */
    toDeleteBatchPrediction() {
        this.to('machinelearning:DeleteBatchPrediction');
        return this;
    }
    /**
     * Assigns the DELETED status to a DataSource, rendering it unusable
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        this.to('machinelearning:DeleteDataSource');
        return this;
    }
    /**
     * Assigns the DELETED status to an Evaluation, rendering it unusable
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DeleteEvaluation.html
     */
    toDeleteEvaluation() {
        this.to('machinelearning:DeleteEvaluation');
        return this;
    }
    /**
     * Assigns the DELETED status to an MLModel, rendering it unusable
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DeleteMLModel.html
     */
    toDeleteMLModel() {
        this.to('machinelearning:DeleteMLModel');
        return this;
    }
    /**
     * Deletes a real time endpoint of an MLModel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DeleteRealtimeEndpoint.html
     */
    toDeleteRealtimeEndpoint() {
        this.to('machinelearning:DeleteRealtimeEndpoint');
        return this;
    }
    /**
     * Deletes the specified tags associated with an ML object. After this operation is complete, you can't recover deleted tags
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        this.to('machinelearning:DeleteTags');
        return this;
    }
    /**
     * Returns a list of BatchPrediction operations that match the search criteria in the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DescribeBatchPredictions.html
     */
    toDescribeBatchPredictions() {
        this.to('machinelearning:DescribeBatchPredictions');
        return this;
    }
    /**
     * Returns a list of DataSource that match the search criteria in the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DescribeDataSources.html
     */
    toDescribeDataSources() {
        this.to('machinelearning:DescribeDataSources');
        return this;
    }
    /**
     * Returns a list of DescribeEvaluations that match the search criteria in the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DescribeEvaluations.html
     */
    toDescribeEvaluations() {
        this.to('machinelearning:DescribeEvaluations');
        return this;
    }
    /**
     * Returns a list of MLModel that match the search criteria in the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DescribeMLModels.html
     */
    toDescribeMLModels() {
        this.to('machinelearning:DescribeMLModels');
        return this;
    }
    /**
     * Describes one or more of the tags for your Amazon ML object
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.to('machinelearning:DescribeTags');
        return this;
    }
    /**
     * Returns a BatchPrediction that includes detailed metadata, status, and data file information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_GetBatchPrediction.html
     */
    toGetBatchPrediction() {
        this.to('machinelearning:GetBatchPrediction');
        return this;
    }
    /**
     * Returns a DataSource that includes metadata and data file information, as well as the current status of the DataSource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_GetDataSource.html
     */
    toGetDataSource() {
        this.to('machinelearning:GetDataSource');
        return this;
    }
    /**
     * Returns an Evaluation that includes metadata as well as the current status of the Evaluation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_GetEvaluation.html
     */
    toGetEvaluation() {
        this.to('machinelearning:GetEvaluation');
        return this;
    }
    /**
     * Returns an MLModel that includes detailed metadata, and data source information as well as the current status of the MLModel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_GetMLModel.html
     */
    toGetMLModel() {
        this.to('machinelearning:GetMLModel');
        return this;
    }
    /**
     * Generates a prediction for the observation using the specified ML Model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_Predict.html
     */
    toPredict() {
        this.to('machinelearning:Predict');
        return this;
    }
    /**
     * Updates the BatchPredictionName of a BatchPrediction
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_UpdateBatchPrediction.html
     */
    toUpdateBatchPrediction() {
        this.to('machinelearning:UpdateBatchPrediction');
        return this;
    }
    /**
     * Updates the DataSourceName of a DataSource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        this.to('machinelearning:UpdateDataSource');
        return this;
    }
    /**
     * Updates the EvaluationName of an Evaluation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_UpdateEvaluation.html
     */
    toUpdateEvaluation() {
        this.to('machinelearning:UpdateEvaluation');
        return this;
    }
    /**
     * Updates the MLModelName and the ScoreThreshold of an MLModel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/machine-learning/latest/APIReference/API_UpdateMLModel.html
     */
    toUpdateMLModel() {
        this.to('machinelearning:UpdateMLModel');
        return this;
    }
    /**
     * Adds a resource of type batchprediction to the statement
     *
     * https://docs.aws.amazon.com/machine-learning/latest/dg/amazon-machine-learning-key-concepts.html#batch-predictions
     *
     * @param batchPredictionId - Identifier for the batchPredictionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBatchprediction(batchPredictionId, account, region, partition) {
        var arn = 'arn:${Partition}:machinelearning:${Region}:${Account}:batchprediction/${BatchPredictionId}';
        arn = arn.replace('${BatchPredictionId}', batchPredictionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/machine-learning/latest/dg/amazon-machine-learning-key-concepts.html#datasources
     *
     * @param datasourceId - Identifier for the datasourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatasource(datasourceId, account, region, partition) {
        var arn = 'arn:${Partition}:machinelearning:${Region}:${Account}:datasource/${DatasourceId}';
        arn = arn.replace('${DatasourceId}', datasourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type evaluation to the statement
     *
     * https://docs.aws.amazon.com/machine-learning/latest/dg/amazon-machine-learning-key-concepts.html#evaluations
     *
     * @param evaluationId - Identifier for the evaluationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvaluation(evaluationId, account, region, partition) {
        var arn = 'arn:${Partition}:machinelearning:${Region}:${Account}:evaluation/${EvaluationId}';
        arn = arn.replace('${EvaluationId}', evaluationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mlmodel to the statement
     *
     * https://docs.aws.amazon.com/machine-learning/latest/dg/amazon-machine-learning-key-concepts.html#ml-models
     *
     * @param mlModelId - Identifier for the mlModelId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMlmodel(mlModelId, account, region, partition) {
        var arn = 'arn:${Partition}:machinelearning:${Region}:${Account}:mlmodel/${MlModelId}';
        arn = arn.replace('${MlModelId}', mlModelId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Machinelearning = Machinelearning;
//# sourceMappingURL=data:application/json;base64,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