"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Managedblockchain = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [managedblockchain](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedblockchain.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Managedblockchain extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [managedblockchain](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedblockchain.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'managedblockchain';
        this.accessLevelList = {
            "Write": [
                "CreateMember",
                "CreateNetwork",
                "CreateNode",
                "CreateProposal",
                "DeleteMember",
                "DeleteNode",
                "RejectInvitation",
                "UpdateMember",
                "UpdateNode",
                "VoteOnProposal"
            ],
            "Read": [
                "GetMember",
                "GetNetwork",
                "GetNode",
                "GetProposal"
            ],
            "List": [
                "ListInvitations",
                "ListMembers",
                "ListNetworks",
                "ListNodes",
                "ListProposalVotes",
                "ListProposals"
            ]
        };
    }
    /**
     * Grants permission to create a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateMember.html
     */
    toCreateMember() {
        this.to('managedblockchain:CreateMember');
        return this;
    }
    /**
     * Grants permission to create an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNetwork.html
     */
    toCreateNetwork() {
        this.to('managedblockchain:CreateNetwork');
        return this;
    }
    /**
     * Grants permission to create a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNode.html
     */
    toCreateNode() {
        this.to('managedblockchain:CreateNode');
        return this;
    }
    /**
     * Grants permission to create a proposal that other blockchain network members can vote on to add or remove a member in an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateProposal.html
     */
    toCreateProposal() {
        this.to('managedblockchain:CreateProposal');
        return this;
    }
    /**
     * Grants permission to delete a member and all associated resources from an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteMember.html
     */
    toDeleteMember() {
        this.to('managedblockchain:DeleteMember');
        return this;
    }
    /**
     * Grants permission to delete a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteNode.html
     */
    toDeleteNode() {
        this.to('managedblockchain:DeleteNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetMember.html
     */
    toGetMember() {
        this.to('managedblockchain:GetMember');
        return this;
    }
    /**
     * Grants permission to return detailed information about an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNetwork.html
     */
    toGetNetwork() {
        this.to('managedblockchain:GetNetwork');
        return this;
    }
    /**
     * Grants permission to return detailed information about a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNode.html
     */
    toGetNode() {
        this.to('managedblockchain:GetNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a proposal of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetProposal.html
     */
    toGetProposal() {
        this.to('managedblockchain:GetProposal');
        return this;
    }
    /**
     * Grants permission to list the invitations extended to the active AWS account from any Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        this.to('managedblockchain:ListInvitations');
        return this;
    }
    /**
     * Grants permission to list the members of an Amazon Managed Blockchain network and the properties of their memberships.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        this.to('managedblockchain:ListMembers');
        return this;
    }
    /**
     * Grants permission to return information about the Amazon Managed Blockchain networks in which the current AWS account has members.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNetworks.html
     */
    toListNetworks() {
        this.to('managedblockchain:ListNetworks');
        return this;
    }
    /**
     * Grants permission to list the nodes within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNodes.html
     */
    toListNodes() {
        this.to('managedblockchain:ListNodes');
        return this;
    }
    /**
     * Grants permission to list all votes for a proposal, including the value of the vote and the unique identifier of the member that cast the vote for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposalVotes.html
     */
    toListProposalVotes() {
        this.to('managedblockchain:ListProposalVotes');
        return this;
    }
    /**
     * Grants permission to list proposals for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposals.html
     */
    toListProposals() {
        this.to('managedblockchain:ListProposals');
        return this;
    }
    /**
     * Grants permission to reject the invitation to join the blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_RejectInvitation.html
     */
    toRejectInvitation() {
        this.to('managedblockchain:RejectInvitation');
        return this;
    }
    /**
     * Grants permission to update a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateMember.html
     */
    toUpdateMember() {
        this.to('managedblockchain:UpdateMember');
        return this;
    }
    /**
     * Grants permission to update a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateNode.html
     */
    toUpdateNode() {
        this.to('managedblockchain:UpdateNode');
        return this;
    }
    /**
     * Grants permission to cast a vote for a proposal on behalf of the blockchain network member specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_VoteOnProposal.html
     */
    toVoteOnProposal() {
        this.to('managedblockchain:VoteOnProposal');
        return this;
    }
    /**
     * Adds a resource of type network to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Network.html
     *
     * @param networkId - Identifier for the networkId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNetwork(networkId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::networks/${NetworkId}';
        arn = arn.replace('${NetworkId}', networkId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type member to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Member.html
     *
     * @param memberId - Identifier for the memberId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMember(memberId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:members/${MemberId}';
        arn = arn.replace('${MemberId}', memberId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type node to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Node.html
     *
     * @param nodeId - Identifier for the nodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNode(nodeId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:nodes/${NodeId}';
        arn = arn.replace('${NodeId}', nodeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type proposal to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Proposal.html
     *
     * @param proposalId - Identifier for the proposalId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProposal(proposalId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::proposals/${ProposalId}';
        arn = arn.replace('${ProposalId}', proposalId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type invitation to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Invitation.html
     *
     * @param invitationId - Identifier for the invitationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInvitation(invitationId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:invitations/${InvitationId}';
        arn = arn.replace('${InvitationId}', invitationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Managedblockchain = Managedblockchain;
//# sourceMappingURL=data:application/json;base64,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