"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Quicksight = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [quicksight](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonquicksight.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Quicksight extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [quicksight](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonquicksight.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'quicksight';
        this.accessLevelList = {
            "Write": [
                "CancelIngestion",
                "CreateAccountCustomization",
                "CreateAdmin",
                "CreateAnalysis",
                "CreateCustomPermissions",
                "CreateDashboard",
                "CreateDataSet",
                "CreateDataSource",
                "CreateGroup",
                "CreateGroupMembership",
                "CreateIAMPolicyAssignment",
                "CreateIngestion",
                "CreateNamespace",
                "CreateReader",
                "CreateTemplate",
                "CreateTemplateAlias",
                "CreateTheme",
                "CreateThemeAlias",
                "CreateUser",
                "DeleteAccountCustomization",
                "DeleteAnalysis",
                "DeleteDashboard",
                "DeleteDataSet",
                "DeleteDataSource",
                "DeleteGroup",
                "DeleteGroupMembership",
                "DeleteIAMPolicyAssignment",
                "DeleteNamespace",
                "DeleteTemplate",
                "DeleteTemplateAlias",
                "DeleteTheme",
                "DeleteThemeAlias",
                "DeleteUser",
                "DeleteUserByPrincipalId",
                "DescribeCustomPermissions",
                "ListCustomPermissions",
                "ListNamespaces",
                "RegisterUser",
                "RestoreAnalysis",
                "SearchDirectoryGroups",
                "SetGroupMapping",
                "Subscribe",
                "Unsubscribe",
                "UpdateAccountCustomization",
                "UpdateAccountSettings",
                "UpdateAnalysis",
                "UpdateAnalysisPermissions",
                "UpdateCustomPermissions",
                "UpdateDashboard",
                "UpdateDashboardPermissions",
                "UpdateDashboardPublishedVersion",
                "UpdateDataSet",
                "UpdateDataSource",
                "UpdateGroup",
                "UpdateIAMPolicyAssignment",
                "UpdateTemplate",
                "UpdateTemplateAlias",
                "UpdateTemplatePermissions",
                "UpdateTheme",
                "UpdateThemeAlias",
                "UpdateThemePermissions",
                "UpdateUser"
            ],
            "Read": [
                "DescribeAccountCustomization",
                "DescribeAccountSettings",
                "DescribeAnalysis",
                "DescribeAnalysisPermissions",
                "DescribeDashboard",
                "DescribeDashboardPermissions",
                "DescribeDataSet",
                "DescribeDataSource",
                "DescribeGroup",
                "DescribeIAMPolicyAssignment",
                "DescribeIngestion",
                "DescribeNamespace",
                "DescribeTemplate",
                "DescribeTemplateAlias",
                "DescribeTemplatePermissions",
                "DescribeTheme",
                "DescribeThemeAlias",
                "DescribeThemePermissions",
                "DescribeUser",
                "GetAuthCode",
                "GetDashboardEmbedUrl",
                "GetGroupMapping",
                "GetSessionEmbedUrl",
                "ListIngestions",
                "PassDataSet",
                "PassDataSource"
            ],
            "Permissions management": [
                "DescribeDataSetPermissions",
                "DescribeDataSourcePermissions",
                "UpdateDataSetPermissions",
                "UpdateDataSourcePermissions"
            ],
            "List": [
                "ListAnalyses",
                "ListDashboardVersions",
                "ListDashboards",
                "ListDataSets",
                "ListDataSources",
                "ListGroupMemberships",
                "ListGroups",
                "ListIAMPolicyAssignments",
                "ListIAMPolicyAssignmentsForUser",
                "ListTagsForResource",
                "ListTemplateAliases",
                "ListTemplateVersions",
                "ListTemplates",
                "ListThemeAliases",
                "ListThemeVersions",
                "ListThemes",
                "ListUserGroups",
                "ListUsers",
                "SearchAnalyses",
                "SearchDashboards"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to cancel a SPICE ingestions on a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CancelIngestion.html
     */
    toCancelIngestion() {
        this.to('quicksight:CancelIngestion');
        return this;
    }
    /**
     * Grants permission to create an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateAccountCustomization.html
     */
    toCreateAccountCustomization() {
        this.to('quicksight:CreateAccountCustomization');
        return this;
    }
    /**
     * CreateAdmin enables the user to provision Amazon QuickSight administrators, authors, and readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateAdmin() {
        this.to('quicksight:CreateAdmin');
        return this;
    }
    /**
     * Creates an analysis from a template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateAnalysis.html
     */
    toCreateAnalysis() {
        this.to('quicksight:CreateAnalysis');
        return this;
    }
    /**
     * Grants permission to create a custom permissions resource for restricting user access
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateCustomPermissions() {
        this.to('quicksight:CreateCustomPermissions');
        return this;
    }
    /**
     * Creates a dashboard from a template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDashboard.html
     */
    toCreateDashboard() {
        this.to('quicksight:CreateDashboard');
        return this;
    }
    /**
     * Grants permission to create a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDataSet.html
     */
    toCreateDataSet() {
        this.to('quicksight:CreateDataSet');
        return this;
    }
    /**
     * Grants permission to create a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDataSource.html
     */
    toCreateDataSource() {
        this.to('quicksight:CreateDataSource');
        return this;
    }
    /**
     * Create a QuickSight group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        this.to('quicksight:CreateGroup');
        return this;
    }
    /**
     * Add a QuickSight user to a QuickSight group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroupMembership.html
     */
    toCreateGroupMembership() {
        this.to('quicksight:CreateGroupMembership');
        return this;
    }
    /**
     * Creates an assignment with one specified IAM Policy ARN that will be assigned to specified groups or users of QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIAMPolicyAssignment.html
     */
    toCreateIAMPolicyAssignment() {
        this.to('quicksight:CreateIAMPolicyAssignment');
        return this;
    }
    /**
     * Grants permission to start a SPICE ingestion on a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIngestion.html
     */
    toCreateIngestion() {
        this.to('quicksight:CreateIngestion');
        return this;
    }
    /**
     * Grants permission to create an QuickSight namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateNamespace.html
     */
    toCreateNamespace() {
        this.to('quicksight:CreateNamespace');
        return this;
    }
    /**
     * CreateReader enables the user to provision Amazon QuickSight readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateReader() {
        this.to('quicksight:CreateReader');
        return this;
    }
    /**
     * Creates a template from an existing QuickSight analysis or template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate() {
        this.to('quicksight:CreateTemplate');
        return this;
    }
    /**
     * Creates a template alias for a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplateAlias.html
     */
    toCreateTemplateAlias() {
        this.to('quicksight:CreateTemplateAlias');
        return this;
    }
    /**
     * Creates a QuickSight theme
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTheme.html
     */
    toCreateTheme() {
        this.to('quicksight:CreateTheme');
        return this;
    }
    /**
     * Creates a theme alias for a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateThemeAlias.html
     */
    toCreateThemeAlias() {
        this.to('quicksight:CreateThemeAlias');
        return this;
    }
    /**
     * CreateUser enables the user to provision Amazon QuickSight authors and readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateUser() {
        this.to('quicksight:CreateUser');
        return this;
    }
    /**
     * Grants permission to delete an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAccountCustomization.html
     */
    toDeleteAccountCustomization() {
        this.to('quicksight:DeleteAccountCustomization');
        return this;
    }
    /**
     * Deletes an analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAnalysis.html
     */
    toDeleteAnalysis() {
        this.to('quicksight:DeleteAnalysis');
        return this;
    }
    /**
     * Deletes a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDashboard.html
     */
    toDeleteDashboard() {
        this.to('quicksight:DeleteDashboard');
        return this;
    }
    /**
     * Grants permission to delete a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDataSet.html
     */
    toDeleteDataSet() {
        this.to('quicksight:DeleteDataSet');
        return this;
    }
    /**
     * Grants permission to delete a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        this.to('quicksight:DeleteDataSource');
        return this;
    }
    /**
     * Remove a user group from QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.to('quicksight:DeleteGroup');
        return this;
    }
    /**
     * Remove a user from a group so that he/she is no longer a member of the group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroupMembership.html
     */
    toDeleteGroupMembership() {
        this.to('quicksight:DeleteGroupMembership');
        return this;
    }
    /**
     * Update an existing assignment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIAMPolicyAssignment.html
     */
    toDeleteIAMPolicyAssignment() {
        this.to('quicksight:DeleteIAMPolicyAssignment');
        return this;
    }
    /**
     * Grants permission to delete a QuickSight namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteNamespace.html
     */
    toDeleteNamespace() {
        this.to('quicksight:DeleteNamespace');
        return this;
    }
    /**
     * Deletes a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate() {
        this.to('quicksight:DeleteTemplate');
        return this;
    }
    /**
     * Deletes the item that the specified template alias points to
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplateAlias.html
     */
    toDeleteTemplateAlias() {
        this.to('quicksight:DeleteTemplateAlias');
        return this;
    }
    /**
     * Deletes a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTheme.html
     */
    toDeleteTheme() {
        this.to('quicksight:DeleteTheme');
        return this;
    }
    /**
     * Deletes the item that the specified theme alias points to
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteThemeAlias.html
     */
    toDeleteThemeAlias() {
        this.to('quicksight:DeleteThemeAlias');
        return this;
    }
    /**
     * Delete the QuickSight user that is associated with the identity of the IAM user/role making the call. The IAM user is not deleted as a result of this call.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('quicksight:DeleteUser');
        return this;
    }
    /**
     * Deletes a user identified by its principal ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserByPrincipalId.html
     */
    toDeleteUserByPrincipalId() {
        this.to('quicksight:DeleteUserByPrincipalId');
        return this;
    }
    /**
     * Grants permission to describe an account customization for QuickSight account or namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountCustomization.html
     */
    toDescribeAccountCustomization() {
        this.to('quicksight:DescribeAccountCustomization');
        return this;
    }
    /**
     * Grants permission to describe the administrative account settings for QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountSettings.html
     */
    toDescribeAccountSettings() {
        this.to('quicksight:DescribeAccountSettings');
        return this;
    }
    /**
     * Provides a summary for an analysis
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAnalysis.html
     */
    toDescribeAnalysis() {
        this.to('quicksight:DescribeAnalysis');
        return this;
    }
    /**
     * Describes read and write permissions for an analysis
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAnalysisPermissions.html
     */
    toDescribeAnalysisPermissions() {
        this.to('quicksight:DescribeAnalysisPermissions');
        return this;
    }
    /**
     * Grants permission to describe a custom permissions resource in a QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toDescribeCustomPermissions() {
        this.to('quicksight:DescribeCustomPermissions');
        return this;
    }
    /**
     * Provides a summary for a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboard.html
     */
    toDescribeDashboard() {
        this.to('quicksight:DescribeDashboard');
        return this;
    }
    /**
     * Describes read and write permissions for a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardPermissions.html
     */
    toDescribeDashboardPermissions() {
        this.to('quicksight:DescribeDashboardPermissions');
        return this;
    }
    /**
     * Grants permission to describe a dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSet.html
     */
    toDescribeDataSet() {
        this.to('quicksight:DescribeDataSet');
        return this;
    }
    /**
     * Grants permission to describe the resource policy of a dataset
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSetPermissions.html
     */
    toDescribeDataSetPermissions() {
        this.to('quicksight:DescribeDataSetPermissions');
        return this;
    }
    /**
     * Grants permission to describe a data source
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSource.html
     */
    toDescribeDataSource() {
        this.to('quicksight:DescribeDataSource');
        return this;
    }
    /**
     * Grants permission to describe the resource policy of a data source
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSourcePermissions.html
     */
    toDescribeDataSourcePermissions() {
        this.to('quicksight:DescribeDataSourcePermissions');
        return this;
    }
    /**
     * Return a QuickSight group’s description and ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroup.html
     */
    toDescribeGroup() {
        this.to('quicksight:DescribeGroup');
        return this;
    }
    /**
     * Describe an existing assignment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIAMPolicyAssignment.html
     */
    toDescribeIAMPolicyAssignment() {
        this.to('quicksight:DescribeIAMPolicyAssignment');
        return this;
    }
    /**
     * Grants permission to describe a SPICE ingestion on a dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIngestion.html
     */
    toDescribeIngestion() {
        this.to('quicksight:DescribeIngestion');
        return this;
    }
    /**
     * Grants permission to describe a QuickSight namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeNamespace.html
     */
    toDescribeNamespace() {
        this.to('quicksight:DescribeNamespace');
        return this;
    }
    /**
     * Describes a template's metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplate.html
     */
    toDescribeTemplate() {
        this.to('quicksight:DescribeTemplate');
        return this;
    }
    /**
     * Describes the template alias for a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplateAlias.html
     */
    toDescribeTemplateAlias() {
        this.to('quicksight:DescribeTemplateAlias');
        return this;
    }
    /**
     * Describes read and write permissions on a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplatePermissions.html
     */
    toDescribeTemplatePermissions() {
        this.to('quicksight:DescribeTemplatePermissions');
        return this;
    }
    /**
     * Describes a theme's metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTheme.html
     */
    toDescribeTheme() {
        this.to('quicksight:DescribeTheme');
        return this;
    }
    /**
     * Describes the theme alias for a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemeAlias.html
     */
    toDescribeThemeAlias() {
        this.to('quicksight:DescribeThemeAlias');
        return this;
    }
    /**
     * Describes read and write permissions on a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemePermissions.html
     */
    toDescribeThemePermissions() {
        this.to('quicksight:DescribeThemePermissions');
        return this;
    }
    /**
     * Return information about a user, given the user name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        this.to('quicksight:DescribeUser');
        return this;
    }
    /**
     * Return an auth code representing a QuickSight user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toGetAuthCode() {
        this.to('quicksight:GetAuthCode');
        return this;
    }
    /**
     * Return a QuickSight dashboard embedding URL.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html
     */
    toGetDashboardEmbedUrl() {
        this.to('quicksight:GetDashboardEmbedUrl');
        return this;
    }
    /**
     * GetGroupMapping is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to identify and display the Microsoft Active Directory (Microsoft Active Directory) directory groups that are mapped to roles in Amazon QuickSight.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toGetGroupMapping() {
        this.to('quicksight:GetGroupMapping');
        return this;
    }
    /**
     * Grants permission to get a URL to embed QuickSight console experience.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetSessionEmbedUrl.html
     */
    toGetSessionEmbedUrl() {
        this.to('quicksight:GetSessionEmbedUrl');
        return this;
    }
    /**
     * Lists analyses in an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListAnalyses.html
     */
    toListAnalyses() {
        this.to('quicksight:ListAnalyses');
        return this;
    }
    /**
     * Grants permission to list custom permissions resources in QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toListCustomPermissions() {
        this.to('quicksight:ListCustomPermissions');
        return this;
    }
    /**
     * Lists all the versions of the dashboards in the QuickSight subscription
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboardVersions.html
     */
    toListDashboardVersions() {
        this.to('quicksight:ListDashboardVersions');
        return this;
    }
    /**
     * Lists dashboards in an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        this.to('quicksight:ListDashboards');
        return this;
    }
    /**
     * Grants permission to list all datasets
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDataSets.html
     */
    toListDataSets() {
        this.to('quicksight:ListDataSets');
        return this;
    }
    /**
     * Grants permission to list all data sources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources() {
        this.to('quicksight:ListDataSources');
        return this;
    }
    /**
     * Return a list of member users in a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroupMemberships.html
     */
    toListGroupMemberships() {
        this.to('quicksight:ListGroupMemberships');
        return this;
    }
    /**
     * Get a list of all user groups in QuickSight.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        this.to('quicksight:ListGroups');
        return this;
    }
    /**
     * List all assignments in the current Amazon QuickSight account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignments.html
     */
    toListIAMPolicyAssignments() {
        this.to('quicksight:ListIAMPolicyAssignments');
        return this;
    }
    /**
     * List all assignments assigned to a user and the groups it belongs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignmentsForUser.html
     */
    toListIAMPolicyAssignmentsForUser() {
        this.to('quicksight:ListIAMPolicyAssignmentsForUser');
        return this;
    }
    /**
     * Grants permission to list all SPICE ingestions on a dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIngestions.html
     */
    toListIngestions() {
        this.to('quicksight:ListIngestions');
        return this;
    }
    /**
     * Grants permission to lists all namespaces in a QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListNamespaces.html
     */
    toListNamespaces() {
        this.to('quicksight:ListNamespaces');
        return this;
    }
    /**
     * List tags of a QuickSight resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('quicksight:ListTagsForResource');
        return this;
    }
    /**
     * Lists all the aliases of a template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateAliases.html
     */
    toListTemplateAliases() {
        this.to('quicksight:ListTemplateAliases');
        return this;
    }
    /**
     * Lists all the versions of the templates in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateVersions.html
     */
    toListTemplateVersions() {
        this.to('quicksight:ListTemplateVersions');
        return this;
    }
    /**
     * Lists all the templates in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates() {
        this.to('quicksight:ListTemplates');
        return this;
    }
    /**
     * Lists all the aliases of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeAliases.html
     */
    toListThemeAliases() {
        this.to('quicksight:ListThemeAliases');
        return this;
    }
    /**
     * Lists all the versions of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeVersions.html
     */
    toListThemeVersions() {
        this.to('quicksight:ListThemeVersions');
        return this;
    }
    /**
     * Lists all the themes in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemes.html
     */
    toListThemes() {
        this.to('quicksight:ListThemes');
        return this;
    }
    /**
     * Return a list of groups that a given user is a member of.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUserGroups.html
     */
    toListUserGroups() {
        this.to('quicksight:ListUserGroups');
        return this;
    }
    /**
     * Return a list of all of the QuickSight users belonging to this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.to('quicksight:ListUsers');
        return this;
    }
    /**
     * Grants permission to use a dataset for a template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/qs-api-overview.html
     */
    toPassDataSet() {
        this.to('quicksight:PassDataSet');
        return this;
    }
    /**
     * Grants permission to use a data source for a data set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/qs-api-overview.html
     */
    toPassDataSource() {
        this.to('quicksight:PassDataSource');
        return this;
    }
    /**
     * Create a QuickSight user, whose identity is associated with the IAM identity/role specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RegisterUser.html
     */
    toRegisterUser() {
        this.to('quicksight:RegisterUser');
        return this;
    }
    /**
     * Restores a deleted analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RestoreAnalysis.html
     */
    toRestoreAnalysis() {
        this.to('quicksight:RestoreAnalysis');
        return this;
    }
    /**
     * Searches for analyses that belong to the user specified in the filter
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchAnalyses.html
     */
    toSearchAnalyses() {
        this.to('quicksight:SearchAnalyses');
        return this;
    }
    /**
     * Searches for dashboards that belong to a user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchDashboards.html
     */
    toSearchDashboards() {
        this.to('quicksight:SearchDashboards');
        return this;
    }
    /**
     * SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSearchDirectoryGroups() {
        this.to('quicksight:SearchDirectoryGroups');
        return this;
    }
    /**
     * SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSetGroupMapping() {
        this.to('quicksight:SetGroupMapping');
        return this;
    }
    /**
     * Subscribe enables the user to subscribe to Amazon QuickSight. Enabling this action also allows the user to upgrade the subscription to Enterprise edition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSubscribe() {
        this.to('quicksight:Subscribe');
        return this;
    }
    /**
     * Add tags to a QuickSight resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('quicksight:TagResource');
        return this;
    }
    /**
     * Unsubscribe enables the user to unsubscribe from Amazon QuickSight, which permanently deletes all users and their resources from Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toUnsubscribe() {
        this.to('quicksight:Unsubscribe');
        return this;
    }
    /**
     * Remove tags from a QuickSight resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('quicksight:UntagResource');
        return this;
    }
    /**
     * Grants permission to update an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAccountCustomization.html
     */
    toUpdateAccountCustomization() {
        this.to('quicksight:UpdateAccountCustomization');
        return this;
    }
    /**
     * Grants permission to update the administrative account settings for QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAccountSettings.html
     */
    toUpdateAccountSettings() {
        this.to('quicksight:UpdateAccountSettings');
        return this;
    }
    /**
     * Updates an analysis in an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAnalysis.html
     */
    toUpdateAnalysis() {
        this.to('quicksight:UpdateAnalysis');
        return this;
    }
    /**
     * Updates read and write permissions on an analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAnalysisPermissions.html
     */
    toUpdateAnalysisPermissions() {
        this.to('quicksight:UpdateAnalysisPermissions');
        return this;
    }
    /**
     * Grants permission to update a custom permissions resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toUpdateCustomPermissions() {
        this.to('quicksight:UpdateCustomPermissions');
        return this;
    }
    /**
     * Updates a dashboard in an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboard.html
     */
    toUpdateDashboard() {
        this.to('quicksight:UpdateDashboard');
        return this;
    }
    /**
     * Updates read and write permissions on a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPermissions.html
     */
    toUpdateDashboardPermissions() {
        this.to('quicksight:UpdateDashboardPermissions');
        return this;
    }
    /**
     * Updates the published version of a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPublishedVersion.html
     */
    toUpdateDashboardPublishedVersion() {
        this.to('quicksight:UpdateDashboardPublishedVersion');
        return this;
    }
    /**
     * Grants permission to update a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSet.html
     */
    toUpdateDataSet() {
        this.to('quicksight:UpdateDataSet');
        return this;
    }
    /**
     * Grants permission to update the resource policy of a dataset
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSetPermissions.html
     */
    toUpdateDataSetPermissions() {
        this.to('quicksight:UpdateDataSetPermissions');
        return this;
    }
    /**
     * Grants permission to update a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        this.to('quicksight:UpdateDataSource');
        return this;
    }
    /**
     * Grants permission to update the resource policy of a data source
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSourcePermissions.html
     */
    toUpdateDataSourcePermissions() {
        this.to('quicksight:UpdateDataSourcePermissions');
        return this;
    }
    /**
     * Change group description.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        this.to('quicksight:UpdateGroup');
        return this;
    }
    /**
     * Update an existing assignment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIAMPolicyAssignment.html
     */
    toUpdateIAMPolicyAssignment() {
        this.to('quicksight:UpdateIAMPolicyAssignment');
        return this;
    }
    /**
     * Updates a template from an existing Amazon QuickSight analysis or another template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate() {
        this.to('quicksight:UpdateTemplate');
        return this;
    }
    /**
     * Updates the template alias of a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplateAlias.html
     */
    toUpdateTemplateAlias() {
        this.to('quicksight:UpdateTemplateAlias');
        return this;
    }
    /**
     * Updates the resource permissions for a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplatePermissions.html
     */
    toUpdateTemplatePermissions() {
        this.to('quicksight:UpdateTemplatePermissions');
        return this;
    }
    /**
     * Updates a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTheme.html
     */
    toUpdateTheme() {
        this.to('quicksight:UpdateTheme');
        return this;
    }
    /**
     * Updates the theme alias of a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemeAlias.html
     */
    toUpdateThemeAlias() {
        this.to('quicksight:UpdateThemeAlias');
        return this;
    }
    /**
     * Updates the resource permissions for a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemePermissions.html
     */
    toUpdateThemePermissions() {
        this.to('quicksight:UpdateThemePermissions');
        return this;
    }
    /**
     * Updates an Amazon QuickSight user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        this.to('quicksight:UpdateUser');
        return this;
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_User.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUser(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:user/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Group.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroup(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:group/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type analysis to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Analysis.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnalysis(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:analysis/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Dashboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboard(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:dashboard/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Template.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:template/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DataSource.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasource(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:datasource/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DataSet.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:dataset/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ingestion to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Ingestion.html
     *
     * @param datasetId - Identifier for the datasetId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIngestion(datasetId, resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:dataset/${DatasetId}/ingestion/${ResourceId}';
        arn = arn.replace('${DatasetId}', datasetId);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type theme to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Theme.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTheme(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:theme/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type assignment to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_IAMPolicyAssignment.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssignment(resourceId, account, partition) {
        var arn = 'arn:${Partition}:quicksight::${Account}:assignment/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type customization to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_AccountCustomization.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomization(resourceId, account, partition) {
        var arn = 'arn:${Partition}:quicksight::${Account}:customization/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Namespace.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNamespace(resourceId, account, partition) {
        var arn = 'arn:${Partition}:quicksight::${Account}:namespace/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * IAM user ARN or role ARN.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toRegisterUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIamArn(value, operator) {
        return this.if(`quicksight:IamArn`, value, operator || 'StringLike');
    }
    /**
     * The session name.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toRegisterUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionName(value, operator) {
        return this.if(`quicksight:SessionName`, value, operator || 'StringLike');
    }
    /**
     * The user name.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toCreateGroupMembership()
     * - .toDeleteGroupMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserName(value, operator) {
        return this.if(`quicksight:UserName`, value, operator || 'StringLike');
    }
}
exports.Quicksight = Quicksight;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9ucXVpY2tzaWdodC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFtYXpvbnF1aWNrc2lnaHQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esc0NBQXNEO0FBRXREOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSx3QkFBZTtJQUc3Qzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQVJOLGtCQUFhLEdBQUcsWUFBWSxDQUFDO1FBODhDMUIsb0JBQWUsR0FBb0I7WUFDM0MsT0FBTyxFQUFFO2dCQUNQLGlCQUFpQjtnQkFDakIsNEJBQTRCO2dCQUM1QixhQUFhO2dCQUNiLGdCQUFnQjtnQkFDaEIseUJBQXlCO2dCQUN6QixpQkFBaUI7Z0JBQ2pCLGVBQWU7Z0JBQ2Ysa0JBQWtCO2dCQUNsQixhQUFhO2dCQUNiLHVCQUF1QjtnQkFDdkIsMkJBQTJCO2dCQUMzQixpQkFBaUI7Z0JBQ2pCLGlCQUFpQjtnQkFDakIsY0FBYztnQkFDZCxnQkFBZ0I7Z0JBQ2hCLHFCQUFxQjtnQkFDckIsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFlBQVk7Z0JBQ1osNEJBQTRCO2dCQUM1QixnQkFBZ0I7Z0JBQ2hCLGlCQUFpQjtnQkFDakIsZUFBZTtnQkFDZixrQkFBa0I7Z0JBQ2xCLGFBQWE7Z0JBQ2IsdUJBQXVCO2dCQUN2QiwyQkFBMkI7Z0JBQzNCLGlCQUFpQjtnQkFDakIsZ0JBQWdCO2dCQUNoQixxQkFBcUI7Z0JBQ3JCLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixZQUFZO2dCQUNaLHlCQUF5QjtnQkFDekIsMkJBQTJCO2dCQUMzQix1QkFBdUI7Z0JBQ3ZCLGdCQUFnQjtnQkFDaEIsY0FBYztnQkFDZCxpQkFBaUI7Z0JBQ2pCLHVCQUF1QjtnQkFDdkIsaUJBQWlCO2dCQUNqQixXQUFXO2dCQUNYLGFBQWE7Z0JBQ2IsNEJBQTRCO2dCQUM1Qix1QkFBdUI7Z0JBQ3ZCLGdCQUFnQjtnQkFDaEIsMkJBQTJCO2dCQUMzQix5QkFBeUI7Z0JBQ3pCLGlCQUFpQjtnQkFDakIsNEJBQTRCO2dCQUM1QixpQ0FBaUM7Z0JBQ2pDLGVBQWU7Z0JBQ2Ysa0JBQWtCO2dCQUNsQixhQUFhO2dCQUNiLDJCQUEyQjtnQkFDM0IsZ0JBQWdCO2dCQUNoQixxQkFBcUI7Z0JBQ3JCLDJCQUEyQjtnQkFDM0IsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLHdCQUF3QjtnQkFDeEIsWUFBWTthQUNiO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLDhCQUE4QjtnQkFDOUIseUJBQXlCO2dCQUN6QixrQkFBa0I7Z0JBQ2xCLDZCQUE2QjtnQkFDN0IsbUJBQW1CO2dCQUNuQiw4QkFBOEI7Z0JBQzlCLGlCQUFpQjtnQkFDakIsb0JBQW9CO2dCQUNwQixlQUFlO2dCQUNmLDZCQUE2QjtnQkFDN0IsbUJBQW1CO2dCQUNuQixtQkFBbUI7Z0JBQ25CLGtCQUFrQjtnQkFDbEIsdUJBQXVCO2dCQUN2Qiw2QkFBNkI7Z0JBQzdCLGVBQWU7Z0JBQ2Ysb0JBQW9CO2dCQUNwQiwwQkFBMEI7Z0JBQzFCLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixzQkFBc0I7Z0JBQ3RCLGlCQUFpQjtnQkFDakIsb0JBQW9CO2dCQUNwQixnQkFBZ0I7Z0JBQ2hCLGFBQWE7Z0JBQ2IsZ0JBQWdCO2FBQ2pCO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLDRCQUE0QjtnQkFDNUIsK0JBQStCO2dCQUMvQiwwQkFBMEI7Z0JBQzFCLDZCQUE2QjthQUM5QjtZQUNELE1BQU0sRUFBRTtnQkFDTixjQUFjO2dCQUNkLHVCQUF1QjtnQkFDdkIsZ0JBQWdCO2dCQUNoQixjQUFjO2dCQUNkLGlCQUFpQjtnQkFDakIsc0JBQXNCO2dCQUN0QixZQUFZO2dCQUNaLDBCQUEwQjtnQkFDMUIsaUNBQWlDO2dCQUNqQyxxQkFBcUI7Z0JBQ3JCLHFCQUFxQjtnQkFDckIsc0JBQXNCO2dCQUN0QixlQUFlO2dCQUNmLGtCQUFrQjtnQkFDbEIsbUJBQW1CO2dCQUNuQixZQUFZO2dCQUNaLGdCQUFnQjtnQkFDaEIsV0FBVztnQkFDWCxnQkFBZ0I7Z0JBQ2hCLGtCQUFrQjthQUNuQjtZQUNELFNBQVMsRUFBRTtnQkFDVCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBbGtERixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7UUFDakQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1FBQ25ELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1FBQ25ELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwrQkFBK0I7UUFDcEMsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxJQUFJLENBQUMsRUFBRSxDQUFDLHdDQUF3QyxDQUFDLENBQUM7UUFDbEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxFQUFFLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNqRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxFQUFFLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNqRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksNkJBQTZCO1FBQ2xDLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFpSUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksTUFBTSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckYsSUFBSSxHQUFHLEdBQUcscUVBQXFFLENBQUM7UUFDaEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksT0FBTyxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEYsSUFBSSxHQUFHLEdBQUcsc0VBQXNFLENBQUM7UUFDakYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekYsSUFBSSxHQUFHLEdBQUcseUVBQXlFLENBQUM7UUFDcEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksV0FBVyxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDMUYsSUFBSSxHQUFHLEdBQUcsMEVBQTBFLENBQUM7UUFDckYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekYsSUFBSSxHQUFHLEdBQUcseUVBQXlFLENBQUM7UUFDcEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksWUFBWSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0YsSUFBSSxHQUFHLEdBQUcsMkVBQTJFLENBQUM7UUFDdEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksU0FBUyxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDeEYsSUFBSSxHQUFHLEdBQUcsd0VBQXdFLENBQUM7UUFDbkYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFdBQVcsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0csSUFBSSxHQUFHLEdBQUcsK0ZBQStGLENBQUM7UUFDMUcsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLE9BQU8sQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RGLElBQUksR0FBRyxHQUFHLHNFQUFzRSxDQUFDO1FBQ2pGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksWUFBWSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxTQUFrQjtRQUMxRSxJQUFJLEdBQUcsR0FBRyxrRUFBa0UsQ0FBQztRQUM3RSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxlQUFlLENBQUMsVUFBa0IsRUFBRSxPQUFnQixFQUFFLFNBQWtCO1FBQzdFLElBQUksR0FBRyxHQUFHLHFFQUFxRSxDQUFDO1FBQ2hGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLFdBQVcsQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDekUsSUFBSSxHQUFHLEdBQUcsaUVBQWlFLENBQUM7UUFDNUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxRQUFRLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNwRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN2RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWEsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFVBQVUsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3RFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7Q0FDRjtBQW4zREQsZ0NBbTNEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gXCIuLi9zaGFyZWQvYWNjZXNzLWxldmVsXCI7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbcXVpY2tzaWdodF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9ucXVpY2tzaWdodC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFF1aWNrc2lnaHQgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdxdWlja3NpZ2h0JztcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtxdWlja3NpZ2h0XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25xdWlja3NpZ2h0Lmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNhbmNlbCBhIFNQSUNFIGluZ2VzdGlvbnMgb24gYSBkYXRhc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2FuY2VsSW5nZXN0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbEluZ2VzdGlvbigpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNhbmNlbEluZ2VzdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBhY2NvdW50IGN1c3RvbWl6YXRpb24gZm9yIFF1aWNrU2lnaHQgYWNjb3VudCBvciBuYW1lc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBY2NvdW50Q3VzdG9taXphdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBY2NvdW50Q3VzdG9taXphdGlvbigpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZUFjY291bnRDdXN0b21pemF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlQWRtaW4gZW5hYmxlcyB0aGUgdXNlciB0byBwcm92aXNpb24gQW1hem9uIFF1aWNrU2lnaHQgYWRtaW5pc3RyYXRvcnMsIGF1dGhvcnMsIGFuZCByZWFkZXJzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC91c2VyL2lhbS1hY3Rpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFkbWluKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6Q3JlYXRlQWRtaW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIGFuYWx5c2lzIGZyb20gYSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUFuYWx5c2lzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFuYWx5c2lzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6Q3JlYXRlQW5hbHlzaXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBjdXN0b20gcGVybWlzc2lvbnMgcmVzb3VyY2UgZm9yIHJlc3RyaWN0aW5nIHVzZXIgYWNjZXNzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L3VzZXIvaWFtLWFjdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ3VzdG9tUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpDcmVhdGVDdXN0b21QZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBkYXNoYm9hcmQgZnJvbSBhIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRGFzaGJvYXJkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURhc2hib2FyZCgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZURhc2hib2FyZCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGRhdGFzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEYXRhU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURhdGFTZXQoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpDcmVhdGVEYXRhU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgZGF0YSBzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEYXRhU291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURhdGFTb3VyY2UoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpDcmVhdGVEYXRhU291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgUXVpY2tTaWdodCBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVHcm91cCgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZUdyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgUXVpY2tTaWdodCB1c2VyIHRvIGEgUXVpY2tTaWdodCBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVHcm91cE1lbWJlcnNoaXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlR3JvdXBNZW1iZXJzaGlwKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6Q3JlYXRlR3JvdXBNZW1iZXJzaGlwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBhc3NpZ25tZW50IHdpdGggb25lIHNwZWNpZmllZCBJQU0gUG9saWN5IEFSTiB0aGF0IHdpbGwgYmUgYXNzaWduZWQgdG8gc3BlY2lmaWVkIGdyb3VwcyBvciB1c2VycyBvZiBRdWlja1NpZ2h0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUlBTVBvbGljeUFzc2lnbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSUFNUG9saWN5QXNzaWdubWVudCgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZUlBTVBvbGljeUFzc2lnbm1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhIFNQSUNFIGluZ2VzdGlvbiBvbiBhIGRhdGFzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVJbmdlc3Rpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW5nZXN0aW9uKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6Q3JlYXRlSW5nZXN0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIFF1aWNrU2lnaHQgbmFtZXNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlTmFtZXNwYWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU5hbWVzcGFjZSgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZU5hbWVzcGFjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZVJlYWRlciBlbmFibGVzIHRoZSB1c2VyIHRvIHByb3Zpc2lvbiBBbWF6b24gUXVpY2tTaWdodCByZWFkZXJzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC91c2VyL2lhbS1hY3Rpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJlYWRlcigpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZVJlYWRlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSB0ZW1wbGF0ZSBmcm9tIGFuIGV4aXN0aW5nIFF1aWNrU2lnaHQgYW5hbHlzaXMgb3IgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVUZW1wbGF0ZSgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZVRlbXBsYXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIHRlbXBsYXRlIGFsaWFzIGZvciBhIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlVGVtcGxhdGVBbGlhcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVUZW1wbGF0ZUFsaWFzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6Q3JlYXRlVGVtcGxhdGVBbGlhcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBRdWlja1NpZ2h0IHRoZW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlVGhlbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVGhlbWUoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpDcmVhdGVUaGVtZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSB0aGVtZSBhbGlhcyBmb3IgYSB0aGVtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVRoZW1lQWxpYXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVGhlbWVBbGlhcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZVRoZW1lQWxpYXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVVc2VyIGVuYWJsZXMgdGhlIHVzZXIgdG8gcHJvdmlzaW9uIEFtYXpvbiBRdWlja1NpZ2h0IGF1dGhvcnMgYW5kIHJlYWRlcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L3VzZXIvaWFtLWFjdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVXNlcigpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkNyZWF0ZVVzZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gYWNjb3VudCBjdXN0b21pemF0aW9uIGZvciBRdWlja1NpZ2h0IGFjY291bnQgb3IgbmFtZXNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQWNjb3VudEN1c3RvbWl6YXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQWNjb3VudEN1c3RvbWl6YXRpb24oKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZWxldGVBY2NvdW50Q3VzdG9taXphdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYW4gYW5hbHlzaXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVBbmFseXNpcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBbmFseXNpcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlbGV0ZUFuYWx5c2lzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIGRhc2hib2FyZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURhc2hib2FyZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEYXNoYm9hcmQoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZWxldGVEYXNoYm9hcmQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBkYXRhc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRGF0YVNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEYXRhU2V0KCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVsZXRlRGF0YVNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGRhdGEgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRGF0YVNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEYXRhU291cmNlKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVsZXRlRGF0YVNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZSBhIHVzZXIgZ3JvdXAgZnJvbSBRdWlja1NpZ2h0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUdyb3VwKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVsZXRlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmUgYSB1c2VyIGZyb20gYSBncm91cCBzbyB0aGF0IGhlL3NoZSBpcyBubyBsb25nZXIgYSBtZW1iZXIgb2YgdGhlIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUdyb3VwTWVtYmVyc2hpcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVHcm91cE1lbWJlcnNoaXAoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZWxldGVHcm91cE1lbWJlcnNoaXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgYW4gZXhpc3RpbmcgYXNzaWdubWVudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVJQU1Qb2xpY3lBc3NpZ25tZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUlBTVBvbGljeUFzc2lnbm1lbnQoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZWxldGVJQU1Qb2xpY3lBc3NpZ25tZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgUXVpY2tTaWdodCBuYW1lc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVOYW1lc3BhY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTmFtZXNwYWNlKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVsZXRlTmFtZXNwYWNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlVGVtcGxhdGUoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZWxldGVUZW1wbGF0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIGl0ZW0gdGhhdCB0aGUgc3BlY2lmaWVkIHRlbXBsYXRlIGFsaWFzIHBvaW50cyB0b1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVRlbXBsYXRlQWxpYXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlVGVtcGxhdGVBbGlhcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlbGV0ZVRlbXBsYXRlQWxpYXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgdGhlbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVUaGVtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVUaGVtZSgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlbGV0ZVRoZW1lJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgaXRlbSB0aGF0IHRoZSBzcGVjaWZpZWQgdGhlbWUgYWxpYXMgcG9pbnRzIHRvXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVGhlbWVBbGlhcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVUaGVtZUFsaWFzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVsZXRlVGhlbWVBbGlhcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZSB0aGUgUXVpY2tTaWdodCB1c2VyIHRoYXQgaXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBpZGVudGl0eSBvZiB0aGUgSUFNIHVzZXIvcm9sZSBtYWtpbmcgdGhlIGNhbGwuIFRoZSBJQU0gdXNlciBpcyBub3QgZGVsZXRlZCBhcyBhIHJlc3VsdCBvZiB0aGlzIGNhbGwuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVVc2VyKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVsZXRlVXNlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSB1c2VyIGlkZW50aWZpZWQgYnkgaXRzIHByaW5jaXBhbCBJRC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVVc2VyQnlQcmluY2lwYWxJZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVVc2VyQnlQcmluY2lwYWxJZCgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlbGV0ZVVzZXJCeVByaW5jaXBhbElkJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYW4gYWNjb3VudCBjdXN0b21pemF0aW9uIGZvciBRdWlja1NpZ2h0IGFjY291bnQgb3IgbmFtZXNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFjY291bnRDdXN0b21pemF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQWNjb3VudEN1c3RvbWl6YXRpb24oKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZXNjcmliZUFjY291bnRDdXN0b21pemF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgdGhlIGFkbWluaXN0cmF0aXZlIGFjY291bnQgc2V0dGluZ3MgZm9yIFF1aWNrU2lnaHQgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVBY2NvdW50U2V0dGluZ3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBY2NvdW50U2V0dGluZ3MoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZXNjcmliZUFjY291bnRTZXR0aW5ncycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3ZpZGVzIGEgc3VtbWFyeSBmb3IgYW4gYW5hbHlzaXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQW5hbHlzaXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBbmFseXNpcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlc2NyaWJlQW5hbHlzaXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgcmVhZCBhbmQgd3JpdGUgcGVybWlzc2lvbnMgZm9yIGFuIGFuYWx5c2lzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFuYWx5c2lzUGVybWlzc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBbmFseXNpc1Blcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVBbmFseXNpc1Blcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBjdXN0b20gcGVybWlzc2lvbnMgcmVzb3VyY2UgaW4gYSBRdWlja1NpZ2h0IGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvdXNlci9pYW0tYWN0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUN1c3RvbVBlcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVDdXN0b21QZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3ZpZGVzIGEgc3VtbWFyeSBmb3IgYSBkYXNoYm9hcmRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRGFzaGJvYXJkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRGFzaGJvYXJkKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVEYXNoYm9hcmQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgcmVhZCBhbmQgd3JpdGUgcGVybWlzc2lvbnMgZm9yIGEgZGFzaGJvYXJkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURhc2hib2FyZFBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRGFzaGJvYXJkUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZXNjcmliZURhc2hib2FyZFBlcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBkYXRhc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURhdGFTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEYXRhU2V0KCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVEYXRhU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgdGhlIHJlc291cmNlIHBvbGljeSBvZiBhIGRhdGFzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRGF0YVNldFBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRGF0YVNldFBlcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVEYXRhU2V0UGVybWlzc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhIGRhdGEgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURhdGFTb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEYXRhU291cmNlKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVEYXRhU291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgdGhlIHJlc291cmNlIHBvbGljeSBvZiBhIGRhdGEgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURhdGFTb3VyY2VQZXJtaXNzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURhdGFTb3VyY2VQZXJtaXNzaW9ucygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlc2NyaWJlRGF0YVNvdXJjZVBlcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgUXVpY2tTaWdodCBncm91cOKAmXMgZGVzY3JpcHRpb24gYW5kIEFSTi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVHcm91cCgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlc2NyaWJlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZSBhbiBleGlzdGluZyBhc3NpZ25tZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVJQU1Qb2xpY3lBc3NpZ25tZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlSUFNUG9saWN5QXNzaWdubWVudCgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlc2NyaWJlSUFNUG9saWN5QXNzaWdubWVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgU1BJQ0UgaW5nZXN0aW9uIG9uIGEgZGF0YXNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVJbmdlc3Rpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVJbmdlc3Rpb24oKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZXNjcmliZUluZ2VzdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgUXVpY2tTaWdodCBuYW1lc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTmFtZXNwYWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTmFtZXNwYWNlKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVOYW1lc3BhY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYSB0ZW1wbGF0ZSdzIG1ldGFkYXRhXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlVGVtcGxhdGUoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZXNjcmliZVRlbXBsYXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSB0ZW1wbGF0ZSBhbGlhcyBmb3IgYSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUZW1wbGF0ZUFsaWFzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlVGVtcGxhdGVBbGlhcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlc2NyaWJlVGVtcGxhdGVBbGlhcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyByZWFkIGFuZCB3cml0ZSBwZXJtaXNzaW9ucyBvbiBhIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRlbXBsYXRlUGVybWlzc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVUZW1wbGF0ZVBlcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVUZW1wbGF0ZVBlcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIGEgdGhlbWUncyBtZXRhZGF0YVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUaGVtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVRoZW1lKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVUaGVtZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyB0aGUgdGhlbWUgYWxpYXMgZm9yIGEgdGhlbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGhlbWVBbGlhcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVRoZW1lQWxpYXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpEZXNjcmliZVRoZW1lQWxpYXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgcmVhZCBhbmQgd3JpdGUgcGVybWlzc2lvbnMgb24gYSB0aGVtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUaGVtZVBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlVGhlbWVQZXJtaXNzaW9ucygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkRlc2NyaWJlVGhlbWVQZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBpbmZvcm1hdGlvbiBhYm91dCBhIHVzZXIsIGdpdmVuIHRoZSB1c2VyIG5hbWUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVVc2VyKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6RGVzY3JpYmVVc2VyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGFuIGF1dGggY29kZSByZXByZXNlbnRpbmcgYSBRdWlja1NpZ2h0IHVzZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvdXNlci9pYW0tYWN0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRBdXRoQ29kZSgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OkdldEF1dGhDb2RlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgUXVpY2tTaWdodCBkYXNoYm9hcmQgZW1iZWRkaW5nIFVSTC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldERhc2hib2FyZEVtYmVkVXJsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERhc2hib2FyZEVtYmVkVXJsKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6R2V0RGFzaGJvYXJkRW1iZWRVcmwnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRHcm91cE1hcHBpbmcgaXMgdXNlZCBvbmx5IGluIEFtYXpvbiBRdWlja1NpZ2h0IEVudGVycHJpc2UgZWRpdGlvbiBhY2NvdW50cy4gSXQgZW5hYmxlcyB0aGUgdXNlciB0byB1c2UgQW1hem9uIFF1aWNrU2lnaHQgdG8gaWRlbnRpZnkgYW5kIGRpc3BsYXkgdGhlIE1pY3Jvc29mdCBBY3RpdmUgRGlyZWN0b3J5IChNaWNyb3NvZnQgQWN0aXZlIERpcmVjdG9yeSkgZGlyZWN0b3J5IGdyb3VwcyB0aGF0IGFyZSBtYXBwZWQgdG8gcm9sZXMgaW4gQW1hem9uIFF1aWNrU2lnaHQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvdXNlci9pYW0tYWN0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRHcm91cE1hcHBpbmcoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpHZXRHcm91cE1hcHBpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBVUkwgdG8gZW1iZWQgUXVpY2tTaWdodCBjb25zb2xlIGV4cGVyaWVuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTZXNzaW9uRW1iZWRVcmwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2Vzc2lvbkVtYmVkVXJsKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6R2V0U2Vzc2lvbkVtYmVkVXJsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYW5hbHlzZXMgaW4gYW4gQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBbmFseXNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QW5hbHlzZXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0QW5hbHlzZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGN1c3RvbSBwZXJtaXNzaW9ucyByZXNvdXJjZXMgaW4gUXVpY2tTaWdodCBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L3VzZXIvaWFtLWFjdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEN1c3RvbVBlcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6TGlzdEN1c3RvbVBlcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYWxsIHRoZSB2ZXJzaW9ucyBvZiB0aGUgZGFzaGJvYXJkcyBpbiB0aGUgUXVpY2tTaWdodCBzdWJzY3JpcHRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REYXNoYm9hcmRWZXJzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RGFzaGJvYXJkVmVyc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0RGFzaGJvYXJkVmVyc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyBkYXNoYm9hcmRzIGluIGFuIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RGFzaGJvYXJkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RGFzaGJvYXJkcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0Okxpc3REYXNoYm9hcmRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgZGF0YXNldHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REYXRhU2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RGF0YVNldHMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0RGF0YVNldHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBkYXRhIHNvdXJjZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REYXRhU291cmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RGF0YVNvdXJjZXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0RGF0YVNvdXJjZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSBsaXN0IG9mIG1lbWJlciB1c2VycyBpbiBhIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEdyb3VwTWVtYmVyc2hpcHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEdyb3VwTWVtYmVyc2hpcHMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0R3JvdXBNZW1iZXJzaGlwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldCBhIGxpc3Qgb2YgYWxsIHVzZXIgZ3JvdXBzIGluIFF1aWNrU2lnaHQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0R3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RHcm91cHMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0R3JvdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBhbGwgYXNzaWdubWVudHMgaW4gdGhlIGN1cnJlbnQgQW1hem9uIFF1aWNrU2lnaHQgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RJQU1Qb2xpY3lBc3NpZ25tZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SUFNUG9saWN5QXNzaWdubWVudHMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0SUFNUG9saWN5QXNzaWdubWVudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IGFsbCBhc3NpZ25tZW50cyBhc3NpZ25lZCB0byBhIHVzZXIgYW5kIHRoZSBncm91cHMgaXQgYmVsb25nc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdElBTVBvbGljeUFzc2lnbm1lbnRzRm9yVXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SUFNUG9saWN5QXNzaWdubWVudHNGb3JVc2VyKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6TGlzdElBTVBvbGljeUFzc2lnbm1lbnRzRm9yVXNlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIFNQSUNFIGluZ2VzdGlvbnMgb24gYSBkYXRhc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0SW5nZXN0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SW5nZXN0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0Okxpc3RJbmdlc3Rpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdHMgYWxsIG5hbWVzcGFjZXMgaW4gYSBRdWlja1NpZ2h0IGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0TmFtZXNwYWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TmFtZXNwYWNlcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0Okxpc3ROYW1lc3BhY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCB0YWdzIG9mIGEgUXVpY2tTaWdodCByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0Okxpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyBhbGwgdGhlIGFsaWFzZXMgb2YgYSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRlbXBsYXRlQWxpYXNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGVtcGxhdGVBbGlhc2VzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6TGlzdFRlbXBsYXRlQWxpYXNlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGFsbCB0aGUgdmVyc2lvbnMgb2YgdGhlIHRlbXBsYXRlcyBpbiB0aGUgY3VycmVudCBBbWF6b24gUXVpY2tTaWdodCBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGVtcGxhdGVWZXJzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGVtcGxhdGVWZXJzaW9ucygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0Okxpc3RUZW1wbGF0ZVZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYWxsIHRoZSB0ZW1wbGF0ZXMgaW4gdGhlIGN1cnJlbnQgQW1hem9uIFF1aWNrU2lnaHQgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRlbXBsYXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGVtcGxhdGVzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6TGlzdFRlbXBsYXRlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGFsbCB0aGUgYWxpYXNlcyBvZiBhIHRoZW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGhlbWVBbGlhc2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUaGVtZUFsaWFzZXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0VGhlbWVBbGlhc2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYWxsIHRoZSB2ZXJzaW9ucyBvZiBhIHRoZW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGhlbWVWZXJzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGhlbWVWZXJzaW9ucygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0Okxpc3RUaGVtZVZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYWxsIHRoZSB0aGVtZXMgaW4gdGhlIGN1cnJlbnQgQW1hem9uIFF1aWNrU2lnaHQgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRoZW1lcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGhlbWVzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6TGlzdFRoZW1lcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIGxpc3Qgb2YgZ3JvdXBzIHRoYXQgYSBnaXZlbiB1c2VyIGlzIGEgbWVtYmVyIG9mLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFVzZXJHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFVzZXJHcm91cHMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpMaXN0VXNlckdyb3VwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIGxpc3Qgb2YgYWxsIG9mIHRoZSBRdWlja1NpZ2h0IHVzZXJzIGJlbG9uZ2luZyB0byB0aGlzIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VXNlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFVzZXJzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6TGlzdFVzZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXNlIGEgZGF0YXNldCBmb3IgYSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9xcy1hcGktb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUGFzc0RhdGFTZXQoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpQYXNzRGF0YVNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVzZSBhIGRhdGEgc291cmNlIGZvciBhIGRhdGEgc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL3FzLWFwaS1vdmVydmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QYXNzRGF0YVNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlBhc3NEYXRhU291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgUXVpY2tTaWdodCB1c2VyLCB3aG9zZSBpZGVudGl0eSBpcyBhc3NvY2lhdGVkIHdpdGggdGhlIElBTSBpZGVudGl0eS9yb2xlIHNwZWNpZmllZCBpbiB0aGUgcmVxdWVzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlclVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVnaXN0ZXJVc2VyKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6UmVnaXN0ZXJVc2VyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVzdG9yZXMgYSBkZWxldGVkIGFuYWx5c2lzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzdG9yZUFuYWx5c2lzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc3RvcmVBbmFseXNpcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlJlc3RvcmVBbmFseXNpcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNlYXJjaGVzIGZvciBhbmFseXNlcyB0aGF0IGJlbG9uZyB0byB0aGUgdXNlciBzcGVjaWZpZWQgaW4gdGhlIGZpbHRlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2VhcmNoQW5hbHlzZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2VhcmNoQW5hbHlzZXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpTZWFyY2hBbmFseXNlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNlYXJjaGVzIGZvciBkYXNoYm9hcmRzIHRoYXQgYmVsb25nIHRvIGEgdXNlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2VhcmNoRGFzaGJvYXJkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZWFyY2hEYXNoYm9hcmRzKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6U2VhcmNoRGFzaGJvYXJkcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNlYXJjaERpcmVjdG9yeUdyb3VwcyBpcyB1c2VkIG9ubHkgaW4gQW1hem9uIFF1aWNrU2lnaHQgRW50ZXJwcmlzZSBlZGl0aW9uIGFjY291bnRzLiBJdCBlbmFibGVzIHRoZSB1c2VyIHRvIHVzZSBBbWF6b24gUXVpY2tTaWdodCB0byBkaXNwbGF5IHlvdXIgTWljcm9zb2Z0IEFjdGl2ZSBEaXJlY3RvcnkgZGlyZWN0b3J5IGdyb3VwcyBzbyB0aGF0IHlvdSBjYW4gY2hvb3NlIHdoaWNoIG9uZXMgdG8gbWFwIHRvIHJvbGVzIGluIEFtYXpvbiBRdWlja1NpZ2h0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC91c2VyL2lhbS1hY3Rpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NlYXJjaERpcmVjdG9yeUdyb3VwcygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlNlYXJjaERpcmVjdG9yeUdyb3VwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNlYXJjaERpcmVjdG9yeUdyb3VwcyBpcyB1c2VkIG9ubHkgaW4gQW1hem9uIFF1aWNrU2lnaHQgRW50ZXJwcmlzZSBlZGl0aW9uIGFjY291bnRzLiBJdCBlbmFibGVzIHRoZSB1c2VyIHRvIHVzZSBBbWF6b24gUXVpY2tTaWdodCB0byBkaXNwbGF5IHlvdXIgTWljcm9zb2Z0IEFjdGl2ZSBEaXJlY3RvcnkgZGlyZWN0b3J5IGdyb3VwcyBzbyB0aGF0IHlvdSBjYW4gY2hvb3NlIHdoaWNoIG9uZXMgdG8gbWFwIHRvIHJvbGVzIGluIEFtYXpvbiBRdWlja1NpZ2h0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC91c2VyL2lhbS1hY3Rpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NldEdyb3VwTWFwcGluZygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlNldEdyb3VwTWFwcGluZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN1YnNjcmliZSBlbmFibGVzIHRoZSB1c2VyIHRvIHN1YnNjcmliZSB0byBBbWF6b24gUXVpY2tTaWdodC4gRW5hYmxpbmcgdGhpcyBhY3Rpb24gYWxzbyBhbGxvd3MgdGhlIHVzZXIgdG8gdXBncmFkZSB0aGUgc3Vic2NyaXB0aW9uIHRvIEVudGVycHJpc2UgZWRpdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvdXNlci9pYW0tYWN0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdWJzY3JpYmUoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpTdWJzY3JpYmUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgdGFncyB0byBhIFF1aWNrU2lnaHQgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6VGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVbnN1YnNjcmliZSBlbmFibGVzIHRoZSB1c2VyIHRvIHVuc3Vic2NyaWJlIGZyb20gQW1hem9uIFF1aWNrU2lnaHQsIHdoaWNoIHBlcm1hbmVudGx5IGRlbGV0ZXMgYWxsIHVzZXJzIGFuZCB0aGVpciByZXNvdXJjZXMgZnJvbSBBbWF6b24gUXVpY2tTaWdodC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvdXNlci9pYW0tYWN0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnN1YnNjcmliZSgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlVuc3Vic2NyaWJlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlIHRhZ3MgZnJvbSBhIFF1aWNrU2lnaHQgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIGFjY291bnQgY3VzdG9taXphdGlvbiBmb3IgUXVpY2tTaWdodCBhY2NvdW50IG9yIG5hbWVzcGFjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUFjY291bnRDdXN0b21pemF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUFjY291bnRDdXN0b21pemF0aW9uKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6VXBkYXRlQWNjb3VudEN1c3RvbWl6YXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGFkbWluaXN0cmF0aXZlIGFjY291bnQgc2V0dGluZ3MgZm9yIFF1aWNrU2lnaHQgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUFjY291bnRTZXR0aW5ncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVBY2NvdW50U2V0dGluZ3MoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVBY2NvdW50U2V0dGluZ3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGFuIGFuYWx5c2lzIGluIGFuIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlQW5hbHlzaXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQW5hbHlzaXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVBbmFseXNpcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgcmVhZCBhbmQgd3JpdGUgcGVybWlzc2lvbnMgb24gYW4gYW5hbHlzaXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVBbmFseXNpc1Blcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUFuYWx5c2lzUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVBbmFseXNpc1Blcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY3VzdG9tIHBlcm1pc3Npb25zIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L3VzZXIvaWFtLWFjdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ3VzdG9tUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVDdXN0b21QZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBkYXNoYm9hcmQgaW4gYW4gQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVEYXNoYm9hcmQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRGFzaGJvYXJkKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6VXBkYXRlRGFzaGJvYXJkJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyByZWFkIGFuZCB3cml0ZSBwZXJtaXNzaW9ucyBvbiBhIGRhc2hib2FyZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURhc2hib2FyZFBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURhc2hib2FyZFBlcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6VXBkYXRlRGFzaGJvYXJkUGVybWlzc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBwdWJsaXNoZWQgdmVyc2lvbiBvZiBhIGRhc2hib2FyZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURhc2hib2FyZFB1Ymxpc2hlZFZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRGFzaGJvYXJkUHVibGlzaGVkVmVyc2lvbigpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlVwZGF0ZURhc2hib2FyZFB1Ymxpc2hlZFZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBkYXRhc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRGF0YVNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEYXRhU2V0KCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6VXBkYXRlRGF0YVNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgcmVzb3VyY2UgcG9saWN5IG9mIGEgZGF0YXNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRGF0YVNldFBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURhdGFTZXRQZXJtaXNzaW9ucygpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlVwZGF0ZURhdGFTZXRQZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGRhdGEgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRGF0YVNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEYXRhU291cmNlKCkge1xuICAgIHRoaXMudG8oJ3F1aWNrc2lnaHQ6VXBkYXRlRGF0YVNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgcmVzb3VyY2UgcG9saWN5IG9mIGEgZGF0YSBzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURhdGFTb3VyY2VQZXJtaXNzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEYXRhU291cmNlUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVEYXRhU291cmNlUGVybWlzc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGFuZ2UgZ3JvdXAgZGVzY3JpcHRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlR3JvdXAoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBhbiBleGlzdGluZyBhc3NpZ25tZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUlBTVBvbGljeUFzc2lnbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSUFNUG9saWN5QXNzaWdubWVudCgpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlVwZGF0ZUlBTVBvbGljeUFzc2lnbm1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGEgdGVtcGxhdGUgZnJvbSBhbiBleGlzdGluZyBBbWF6b24gUXVpY2tTaWdodCBhbmFseXNpcyBvciBhbm90aGVyIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVGVtcGxhdGUoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVUZW1wbGF0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIHRlbXBsYXRlIGFsaWFzIG9mIGEgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVUZW1wbGF0ZUFsaWFzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVRlbXBsYXRlQWxpYXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVUZW1wbGF0ZUFsaWFzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGUgcmVzb3VyY2UgcGVybWlzc2lvbnMgZm9yIGEgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVUZW1wbGF0ZVBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVRlbXBsYXRlUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVUZW1wbGF0ZVBlcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHRoZW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGhlbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVGhlbWUoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVUaGVtZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIHRoZW1lIGFsaWFzIG9mIGEgdGhlbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVUaGVtZUFsaWFzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVRoZW1lQWxpYXMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVUaGVtZUFsaWFzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGUgcmVzb3VyY2UgcGVybWlzc2lvbnMgZm9yIGEgdGhlbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVUaGVtZVBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVRoZW1lUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygncXVpY2tzaWdodDpVcGRhdGVUaGVtZVBlcm1pc3Npb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhbiBBbWF6b24gUXVpY2tTaWdodCB1c2VyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVXNlcigpIHtcbiAgICB0aGlzLnRvKCdxdWlja3NpZ2h0OlVwZGF0ZVVzZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBcIldyaXRlXCI6IFtcbiAgICAgIFwiQ2FuY2VsSW5nZXN0aW9uXCIsXG4gICAgICBcIkNyZWF0ZUFjY291bnRDdXN0b21pemF0aW9uXCIsXG4gICAgICBcIkNyZWF0ZUFkbWluXCIsXG4gICAgICBcIkNyZWF0ZUFuYWx5c2lzXCIsXG4gICAgICBcIkNyZWF0ZUN1c3RvbVBlcm1pc3Npb25zXCIsXG4gICAgICBcIkNyZWF0ZURhc2hib2FyZFwiLFxuICAgICAgXCJDcmVhdGVEYXRhU2V0XCIsXG4gICAgICBcIkNyZWF0ZURhdGFTb3VyY2VcIixcbiAgICAgIFwiQ3JlYXRlR3JvdXBcIixcbiAgICAgIFwiQ3JlYXRlR3JvdXBNZW1iZXJzaGlwXCIsXG4gICAgICBcIkNyZWF0ZUlBTVBvbGljeUFzc2lnbm1lbnRcIixcbiAgICAgIFwiQ3JlYXRlSW5nZXN0aW9uXCIsXG4gICAgICBcIkNyZWF0ZU5hbWVzcGFjZVwiLFxuICAgICAgXCJDcmVhdGVSZWFkZXJcIixcbiAgICAgIFwiQ3JlYXRlVGVtcGxhdGVcIixcbiAgICAgIFwiQ3JlYXRlVGVtcGxhdGVBbGlhc1wiLFxuICAgICAgXCJDcmVhdGVUaGVtZVwiLFxuICAgICAgXCJDcmVhdGVUaGVtZUFsaWFzXCIsXG4gICAgICBcIkNyZWF0ZVVzZXJcIixcbiAgICAgIFwiRGVsZXRlQWNjb3VudEN1c3RvbWl6YXRpb25cIixcbiAgICAgIFwiRGVsZXRlQW5hbHlzaXNcIixcbiAgICAgIFwiRGVsZXRlRGFzaGJvYXJkXCIsXG4gICAgICBcIkRlbGV0ZURhdGFTZXRcIixcbiAgICAgIFwiRGVsZXRlRGF0YVNvdXJjZVwiLFxuICAgICAgXCJEZWxldGVHcm91cFwiLFxuICAgICAgXCJEZWxldGVHcm91cE1lbWJlcnNoaXBcIixcbiAgICAgIFwiRGVsZXRlSUFNUG9saWN5QXNzaWdubWVudFwiLFxuICAgICAgXCJEZWxldGVOYW1lc3BhY2VcIixcbiAgICAgIFwiRGVsZXRlVGVtcGxhdGVcIixcbiAgICAgIFwiRGVsZXRlVGVtcGxhdGVBbGlhc1wiLFxuICAgICAgXCJEZWxldGVUaGVtZVwiLFxuICAgICAgXCJEZWxldGVUaGVtZUFsaWFzXCIsXG4gICAgICBcIkRlbGV0ZVVzZXJcIixcbiAgICAgIFwiRGVsZXRlVXNlckJ5UHJpbmNpcGFsSWRcIixcbiAgICAgIFwiRGVzY3JpYmVDdXN0b21QZXJtaXNzaW9uc1wiLFxuICAgICAgXCJMaXN0Q3VzdG9tUGVybWlzc2lvbnNcIixcbiAgICAgIFwiTGlzdE5hbWVzcGFjZXNcIixcbiAgICAgIFwiUmVnaXN0ZXJVc2VyXCIsXG4gICAgICBcIlJlc3RvcmVBbmFseXNpc1wiLFxuICAgICAgXCJTZWFyY2hEaXJlY3RvcnlHcm91cHNcIixcbiAgICAgIFwiU2V0R3JvdXBNYXBwaW5nXCIsXG4gICAgICBcIlN1YnNjcmliZVwiLFxuICAgICAgXCJVbnN1YnNjcmliZVwiLFxuICAgICAgXCJVcGRhdGVBY2NvdW50Q3VzdG9taXphdGlvblwiLFxuICAgICAgXCJVcGRhdGVBY2NvdW50U2V0dGluZ3NcIixcbiAgICAgIFwiVXBkYXRlQW5hbHlzaXNcIixcbiAgICAgIFwiVXBkYXRlQW5hbHlzaXNQZXJtaXNzaW9uc1wiLFxuICAgICAgXCJVcGRhdGVDdXN0b21QZXJtaXNzaW9uc1wiLFxuICAgICAgXCJVcGRhdGVEYXNoYm9hcmRcIixcbiAgICAgIFwiVXBkYXRlRGFzaGJvYXJkUGVybWlzc2lvbnNcIixcbiAgICAgIFwiVXBkYXRlRGFzaGJvYXJkUHVibGlzaGVkVmVyc2lvblwiLFxuICAgICAgXCJVcGRhdGVEYXRhU2V0XCIsXG4gICAgICBcIlVwZGF0ZURhdGFTb3VyY2VcIixcbiAgICAgIFwiVXBkYXRlR3JvdXBcIixcbiAgICAgIFwiVXBkYXRlSUFNUG9saWN5QXNzaWdubWVudFwiLFxuICAgICAgXCJVcGRhdGVUZW1wbGF0ZVwiLFxuICAgICAgXCJVcGRhdGVUZW1wbGF0ZUFsaWFzXCIsXG4gICAgICBcIlVwZGF0ZVRlbXBsYXRlUGVybWlzc2lvbnNcIixcbiAgICAgIFwiVXBkYXRlVGhlbWVcIixcbiAgICAgIFwiVXBkYXRlVGhlbWVBbGlhc1wiLFxuICAgICAgXCJVcGRhdGVUaGVtZVBlcm1pc3Npb25zXCIsXG4gICAgICBcIlVwZGF0ZVVzZXJcIlxuICAgIF0sXG4gICAgXCJSZWFkXCI6IFtcbiAgICAgIFwiRGVzY3JpYmVBY2NvdW50Q3VzdG9taXphdGlvblwiLFxuICAgICAgXCJEZXNjcmliZUFjY291bnRTZXR0aW5nc1wiLFxuICAgICAgXCJEZXNjcmliZUFuYWx5c2lzXCIsXG4gICAgICBcIkRlc2NyaWJlQW5hbHlzaXNQZXJtaXNzaW9uc1wiLFxuICAgICAgXCJEZXNjcmliZURhc2hib2FyZFwiLFxuICAgICAgXCJEZXNjcmliZURhc2hib2FyZFBlcm1pc3Npb25zXCIsXG4gICAgICBcIkRlc2NyaWJlRGF0YVNldFwiLFxuICAgICAgXCJEZXNjcmliZURhdGFTb3VyY2VcIixcbiAgICAgIFwiRGVzY3JpYmVHcm91cFwiLFxuICAgICAgXCJEZXNjcmliZUlBTVBvbGljeUFzc2lnbm1lbnRcIixcbiAgICAgIFwiRGVzY3JpYmVJbmdlc3Rpb25cIixcbiAgICAgIFwiRGVzY3JpYmVOYW1lc3BhY2VcIixcbiAgICAgIFwiRGVzY3JpYmVUZW1wbGF0ZVwiLFxuICAgICAgXCJEZXNjcmliZVRlbXBsYXRlQWxpYXNcIixcbiAgICAgIFwiRGVzY3JpYmVUZW1wbGF0ZVBlcm1pc3Npb25zXCIsXG4gICAgICBcIkRlc2NyaWJlVGhlbWVcIixcbiAgICAgIFwiRGVzY3JpYmVUaGVtZUFsaWFzXCIsXG4gICAgICBcIkRlc2NyaWJlVGhlbWVQZXJtaXNzaW9uc1wiLFxuICAgICAgXCJEZXNjcmliZVVzZXJcIixcbiAgICAgIFwiR2V0QXV0aENvZGVcIixcbiAgICAgIFwiR2V0RGFzaGJvYXJkRW1iZWRVcmxcIixcbiAgICAgIFwiR2V0R3JvdXBNYXBwaW5nXCIsXG4gICAgICBcIkdldFNlc3Npb25FbWJlZFVybFwiLFxuICAgICAgXCJMaXN0SW5nZXN0aW9uc1wiLFxuICAgICAgXCJQYXNzRGF0YVNldFwiLFxuICAgICAgXCJQYXNzRGF0YVNvdXJjZVwiXG4gICAgXSxcbiAgICBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIjogW1xuICAgICAgXCJEZXNjcmliZURhdGFTZXRQZXJtaXNzaW9uc1wiLFxuICAgICAgXCJEZXNjcmliZURhdGFTb3VyY2VQZXJtaXNzaW9uc1wiLFxuICAgICAgXCJVcGRhdGVEYXRhU2V0UGVybWlzc2lvbnNcIixcbiAgICAgIFwiVXBkYXRlRGF0YVNvdXJjZVBlcm1pc3Npb25zXCJcbiAgICBdLFxuICAgIFwiTGlzdFwiOiBbXG4gICAgICBcIkxpc3RBbmFseXNlc1wiLFxuICAgICAgXCJMaXN0RGFzaGJvYXJkVmVyc2lvbnNcIixcbiAgICAgIFwiTGlzdERhc2hib2FyZHNcIixcbiAgICAgIFwiTGlzdERhdGFTZXRzXCIsXG4gICAgICBcIkxpc3REYXRhU291cmNlc1wiLFxuICAgICAgXCJMaXN0R3JvdXBNZW1iZXJzaGlwc1wiLFxuICAgICAgXCJMaXN0R3JvdXBzXCIsXG4gICAgICBcIkxpc3RJQU1Qb2xpY3lBc3NpZ25tZW50c1wiLFxuICAgICAgXCJMaXN0SUFNUG9saWN5QXNzaWdubWVudHNGb3JVc2VyXCIsXG4gICAgICBcIkxpc3RUYWdzRm9yUmVzb3VyY2VcIixcbiAgICAgIFwiTGlzdFRlbXBsYXRlQWxpYXNlc1wiLFxuICAgICAgXCJMaXN0VGVtcGxhdGVWZXJzaW9uc1wiLFxuICAgICAgXCJMaXN0VGVtcGxhdGVzXCIsXG4gICAgICBcIkxpc3RUaGVtZUFsaWFzZXNcIixcbiAgICAgIFwiTGlzdFRoZW1lVmVyc2lvbnNcIixcbiAgICAgIFwiTGlzdFRoZW1lc1wiLFxuICAgICAgXCJMaXN0VXNlckdyb3Vwc1wiLFxuICAgICAgXCJMaXN0VXNlcnNcIixcbiAgICAgIFwiU2VhcmNoQW5hbHlzZXNcIixcbiAgICAgIFwiU2VhcmNoRGFzaGJvYXJkc1wiXG4gICAgXSxcbiAgICBcIlRhZ2dpbmdcIjogW1xuICAgICAgXCJUYWdSZXNvdXJjZVwiLFxuICAgICAgXCJVbnRhZ1Jlc291cmNlXCJcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHVzZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Vc2VyLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvblVzZXIocmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpxdWlja3NpZ2h0OiR7UmVnaW9ufToke0FjY291bnR9OnVzZXIvJHtSZXNvdXJjZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VJZH0nLCByZXNvdXJjZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBncm91cCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dyb3VwLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkdyb3VwKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06cXVpY2tzaWdodDoke1JlZ2lvbn06JHtBY2NvdW50fTpncm91cC8ke1Jlc291cmNlSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZUlkfScsIHJlc291cmNlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFuYWx5c2lzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQW5hbHlzaXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkFuYWx5c2lzKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06cXVpY2tzaWdodDoke1JlZ2lvbn06JHtBY2NvdW50fTphbmFseXNpcy8ke1Jlc291cmNlSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZUlkfScsIHJlc291cmNlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRhc2hib2FyZCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rhc2hib2FyZC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGFzaGJvYXJkKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06cXVpY2tzaWdodDoke1JlZ2lvbn06JHtBY2NvdW50fTpkYXNoYm9hcmQvJHtSZXNvdXJjZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VJZH0nLCByZXNvdXJjZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0ZW1wbGF0ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RlbXBsYXRlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25UZW1wbGF0ZShyZXNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnF1aWNrc2lnaHQ6JHtSZWdpb259OiR7QWNjb3VudH06dGVtcGxhdGUvJHtSZXNvdXJjZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VJZH0nLCByZXNvdXJjZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkYXRhc291cmNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGF0YVNvdXJjZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGF0YXNvdXJjZShyZXNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnF1aWNrc2lnaHQ6JHtSZWdpb259OiR7QWNjb3VudH06ZGF0YXNvdXJjZS8ke1Jlc291cmNlSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZUlkfScsIHJlc291cmNlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRhdGFzZXQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EYXRhU2V0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25EYXRhc2V0KHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06cXVpY2tzaWdodDoke1JlZ2lvbn06JHtBY2NvdW50fTpkYXRhc2V0LyR7UmVzb3VyY2VJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlSWR9JywgcmVzb3VyY2VJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgaW5nZXN0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfSW5nZXN0aW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRhdGFzZXRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkYXRhc2V0SWQuXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSW5nZXN0aW9uKGRhdGFzZXRJZDogc3RyaW5nLCByZXNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnF1aWNrc2lnaHQ6JHtSZWdpb259OiR7QWNjb3VudH06ZGF0YXNldC8ke0RhdGFzZXRJZH0vaW5nZXN0aW9uLyR7UmVzb3VyY2VJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0RhdGFzZXRJZH0nLCBkYXRhc2V0SWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlSWR9JywgcmVzb3VyY2VJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgdGhlbWUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UaGVtZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uVGhlbWUocmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpxdWlja3NpZ2h0OiR7UmVnaW9ufToke0FjY291bnR9OnRoZW1lLyR7UmVzb3VyY2VJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlSWR9JywgcmVzb3VyY2VJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXNzaWdubWVudCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0lBTVBvbGljeUFzc2lnbm1lbnQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25Bc3NpZ25tZW50KHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnF1aWNrc2lnaHQ6OiR7QWNjb3VudH06YXNzaWdubWVudC8ke1Jlc291cmNlSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZUlkfScsIHJlc291cmNlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY3VzdG9taXphdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FjY291bnRDdXN0b21pemF0aW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkN1c3RvbWl6YXRpb24ocmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06cXVpY2tzaWdodDo6JHtBY2NvdW50fTpjdXN0b21pemF0aW9uLyR7UmVzb3VyY2VJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlSWR9JywgcmVzb3VyY2VJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBuYW1lc3BhY2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcXVpY2tzaWdodC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9OYW1lc3BhY2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25OYW1lc3BhY2UocmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06cXVpY2tzaWdodDo6JHtBY2NvdW50fTpuYW1lc3BhY2UvJHtSZXNvdXJjZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VJZH0nLCByZXNvdXJjZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIElBTSB1c2VyIEFSTiBvciByb2xlIEFSTi5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L3VzZXIvaWFtLWFjdGlvbnMuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJVc2VyKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZJYW1Bcm4odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHF1aWNrc2lnaHQ6SWFtQXJuYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHNlc3Npb24gbmFtZS5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3F1aWNrc2lnaHQvbGF0ZXN0L3VzZXIvaWFtLWFjdGlvbnMuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJVc2VyKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZTZXNzaW9uTmFtZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgcXVpY2tzaWdodDpTZXNzaW9uTmFtZWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB1c2VyIG5hbWUuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9xdWlja3NpZ2h0L2xhdGVzdC91c2VyL2lhbS1hY3Rpb25zLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUdyb3VwTWVtYmVyc2hpcCgpXG4gICAqIC0gLnRvRGVsZXRlR3JvdXBNZW1iZXJzaGlwKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZVc2VyTmFtZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgcXVpY2tzaWdodDpVc2VyTmFtZWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG59XG4iXX0=