"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Redshift = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [redshift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonredshift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Redshift extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [redshift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonredshift.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'redshift';
        this.accessLevelList = {
            "Write": [
                "AcceptReservedNodeExchange",
                "BatchDeleteClusterSnapshots",
                "BatchModifyClusterSnapshots",
                "CancelQuery",
                "CancelQuerySession",
                "CancelResize",
                "CopyClusterSnapshot",
                "CreateCluster",
                "CreateClusterParameterGroup",
                "CreateClusterSecurityGroup",
                "CreateClusterSnapshot",
                "CreateClusterSubnetGroup",
                "CreateEventSubscription",
                "CreateHsmClientCertificate",
                "CreateHsmConfiguration",
                "CreateSavedQuery",
                "CreateScheduledAction",
                "CreateSnapshotSchedule",
                "DeleteCluster",
                "DeleteClusterParameterGroup",
                "DeleteClusterSecurityGroup",
                "DeleteClusterSnapshot",
                "DeleteClusterSubnetGroup",
                "DeleteEventSubscription",
                "DeleteHsmClientCertificate",
                "DeleteHsmConfiguration",
                "DeleteSavedQueries",
                "DeleteScheduledAction",
                "DeleteSnapshotCopyGrant",
                "DeleteSnapshotSchedule",
                "DisableLogging",
                "DisableSnapshotCopy",
                "EnableLogging",
                "EnableSnapshotCopy",
                "ExecuteQuery",
                "GetClusterCredentials",
                "ModifyCluster",
                "ModifyClusterDbRevision",
                "ModifyClusterMaintenance",
                "ModifyClusterParameterGroup",
                "ModifyClusterSnapshot",
                "ModifyClusterSnapshotSchedule",
                "ModifyClusterSubnetGroup",
                "ModifyEventSubscription",
                "ModifySavedQuery",
                "ModifyScheduledAction",
                "ModifySnapshotCopyRetentionPeriod",
                "ModifySnapshotSchedule",
                "PauseCluster",
                "PurchaseReservedNodeOffering",
                "RebootCluster",
                "ResetClusterParameterGroup",
                "ResizeCluster",
                "RestoreFromClusterSnapshot",
                "RestoreTableFromClusterSnapshot",
                "ResumeCluster"
            ],
            "Permissions management": [
                "AuthorizeClusterSecurityGroupIngress",
                "AuthorizeSnapshotAccess",
                "CreateClusterUser",
                "CreateSnapshotCopyGrant",
                "JoinGroup",
                "ModifyClusterIamRoles",
                "RevokeClusterSecurityGroupIngress",
                "RevokeSnapshotAccess",
                "RotateEncryptionKey"
            ],
            "Tagging": [
                "CreateTags",
                "DeleteTags"
            ],
            "Read": [
                "DescribeAccountAttributes",
                "DescribeClusterParameterGroups",
                "DescribeClusterParameters",
                "DescribeClusterSecurityGroups",
                "DescribeClusterSnapshots",
                "DescribeClusterSubnetGroups",
                "DescribeClusterVersions",
                "DescribeDefaultClusterParameters",
                "DescribeEventCategories",
                "DescribeEventSubscriptions",
                "DescribeHsmClientCertificates",
                "DescribeHsmConfigurations",
                "DescribeLoggingStatus",
                "DescribeOrderableClusterOptions",
                "DescribeQuery",
                "DescribeReservedNodeOfferings",
                "DescribeReservedNodes",
                "DescribeResize",
                "DescribeSavedQueries",
                "DescribeScheduledActions",
                "DescribeSnapshotCopyGrants",
                "DescribeSnapshotSchedules",
                "DescribeStorage",
                "DescribeTable",
                "DescribeTableRestoreStatus",
                "DescribeTags",
                "FetchResults",
                "GetReservedNodeExchangeOfferings"
            ],
            "List": [
                "DescribeClusterDbRevisions",
                "DescribeClusterTracks",
                "DescribeClusters",
                "DescribeEvents",
                "DescribeNodeConfigurationOptions",
                "ListDatabases",
                "ListSavedQueries",
                "ListSchemas",
                "ListTables",
                "ViewQueriesFromConsole",
                "ViewQueriesInConsole"
            ]
        };
    }
    /**
     * Grants permission to exchange a DC1 reserved node for a DC2 reserved node with no changes to the configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AcceptReservedNodeExchange.html
     */
    toAcceptReservedNodeExchange() {
        this.to('redshift:AcceptReservedNodeExchange');
        return this;
    }
    /**
     * Grants permission to add an inbound (ingress) rule to an Amazon Redshift security group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeClusterSecurityGroupIngress.html
     */
    toAuthorizeClusterSecurityGroupIngress() {
        this.to('redshift:AuthorizeClusterSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to the specified AWS account to restore a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeSnapshotAccess.html
     */
    toAuthorizeSnapshotAccess() {
        this.to('redshift:AuthorizeSnapshotAccess');
        return this;
    }
    /**
     * Grants permission to delete snapshots in a batch of size upto 100
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchDeleteClusterSnapshots.html
     */
    toBatchDeleteClusterSnapshots() {
        this.to('redshift:BatchDeleteClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to modify settings for a list of snapshots
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchModifyClusterSnapshots.html
     */
    toBatchModifyClusterSnapshots() {
        this.to('redshift:BatchModifyClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to cancel a query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toCancelQuery() {
        this.to('redshift:CancelQuery');
        return this;
    }
    /**
     * Grants permission to see queries in the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toCancelQuerySession() {
        this.to('redshift:CancelQuerySession');
        return this;
    }
    /**
     * Grants permission to cancel a resize operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CancelResize.html
     */
    toCancelResize() {
        this.to('redshift:CancelResize');
        return this;
    }
    /**
     * Grants permission to copy a cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CopyClusterSnapshot.html
     */
    toCopyClusterSnapshot() {
        this.to('redshift:CopyClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster() {
        this.to('redshift:CreateCluster');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterParameterGroup.html
     */
    toCreateClusterParameterGroup() {
        this.to('redshift:CreateClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSecurityGroup.html
     */
    toCreateClusterSecurityGroup() {
        this.to('redshift:CreateClusterSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a manual snapshot of the specified cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSnapshot.html
     */
    toCreateClusterSnapshot() {
        this.to('redshift:CreateClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSubnetGroup.html
     */
    toCreateClusterSubnetGroup() {
        this.to('redshift:CreateClusterSubnetGroup');
        return this;
    }
    /**
     * Grants permission to automatically create the specified Amazon Redshift user if it does not exist
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDbUser()
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/generating-iam-credentials-role-permissions.html
     */
    toCreateClusterUser() {
        this.to('redshift:CreateClusterUser');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription() {
        this.to('redshift:CreateEventSubscription');
        return this;
    }
    /**
     * Grants permission to create an HSM client certificate that a cluster uses to connect to an HSM
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmClientCertificate.html
     */
    toCreateHsmClientCertificate() {
        this.to('redshift:CreateHsmClientCertificate');
        return this;
    }
    /**
     * Grants permission to create an HSM configuration that contains information required by a cluster to store and use database encryption keys in a hardware security module (HSM)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmConfiguration.html
     */
    toCreateHsmConfiguration() {
        this.to('redshift:CreateHsmConfiguration');
        return this;
    }
    /**
     * Grants permission to create saved SQL queries through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toCreateSavedQuery() {
        this.to('redshift:CreateSavedQuery');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateScheduledAction.html
     */
    toCreateScheduledAction() {
        this.to('redshift:CreateScheduledAction');
        return this;
    }
    /**
     * Grants permission to create a snapshot copy grant and encrypt copied snapshots in a destination AWS Region
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotCopyGrant.html
     */
    toCreateSnapshotCopyGrant() {
        this.to('redshift:CreateSnapshotCopyGrant');
        return this;
    }
    /**
     * Grants permission to create a snapshot schedule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotSchedule.html
     */
    toCreateSnapshotSchedule() {
        this.to('redshift:CreateSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateTags.html
     */
    toCreateTags() {
        this.to('redshift:CreateTags');
        return this;
    }
    /**
     * Grants permission to delete a previously provisioned cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster() {
        this.to('redshift:DeleteCluster');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterParameterGroup.html
     */
    toDeleteClusterParameterGroup() {
        this.to('redshift:DeleteClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSecurityGroup.html
     */
    toDeleteClusterSecurityGroup() {
        this.to('redshift:DeleteClusterSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a manual snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSnapshot.html
     */
    toDeleteClusterSnapshot() {
        this.to('redshift:DeleteClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a cluster subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSubnetGroup.html
     */
    toDeleteClusterSubnetGroup() {
        this.to('redshift:DeleteClusterSubnetGroup');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription() {
        this.to('redshift:DeleteEventSubscription');
        return this;
    }
    /**
     * Grants permission to delete an HSM client certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmClientCertificate.html
     */
    toDeleteHsmClientCertificate() {
        this.to('redshift:DeleteHsmClientCertificate');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift HSM configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmConfiguration.html
     */
    toDeleteHsmConfiguration() {
        this.to('redshift:DeleteHsmConfiguration');
        return this;
    }
    /**
     * Grants permission to delete saved SQL queries through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toDeleteSavedQueries() {
        this.to('redshift:DeleteSavedQueries');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift scheduled action
     *
     * Access Level: Write
     */
    toDeleteScheduledAction() {
        this.to('redshift:DeleteScheduledAction');
        return this;
    }
    /**
     * Grants permission to delete a snapshot copy grant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotCopyGrant.html
     */
    toDeleteSnapshotCopyGrant() {
        this.to('redshift:DeleteSnapshotCopyGrant');
        return this;
    }
    /**
     * Grants permission to delete a snapshot schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotSchedule.html
     */
    toDeleteSnapshotSchedule() {
        this.to('redshift:DeleteSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to delete a tag or tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        this.to('redshift:DeleteTags');
        return this;
    }
    /**
     * Grants permission to describe attributes attached to the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        this.to('redshift:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to describe database revisions for a cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterDbRevisions.html
     */
    toDescribeClusterDbRevisions() {
        this.to('redshift:DescribeClusterDbRevisions');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift parameter groups, including parameter groups you created and the default parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameterGroups.html
     */
    toDescribeClusterParameterGroups() {
        this.to('redshift:DescribeClusterParameterGroups');
        return this;
    }
    /**
     * Grants permission to describe parameters contained within an Amazon Redshift parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameters.html
     */
    toDescribeClusterParameters() {
        this.to('redshift:DescribeClusterParameters');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift security groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSecurityGroups.html
     */
    toDescribeClusterSecurityGroups() {
        this.to('redshift:DescribeClusterSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe one or more snapshot objects, which contain metadata about your cluster snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSnapshots.html
     */
    toDescribeClusterSnapshots() {
        this.to('redshift:DescribeClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to describe one or more cluster subnet group objects, which contain metadata about your cluster subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSubnetGroups.html
     */
    toDescribeClusterSubnetGroups() {
        this.to('redshift:DescribeClusterSubnetGroups');
        return this;
    }
    /**
     * Grants permission to describe available maintenance tracks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterTracks.html
     */
    toDescribeClusterTracks() {
        this.to('redshift:DescribeClusterTracks');
        return this;
    }
    /**
     * Grants permission to describe available Amazon Redshift cluster versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterVersions.html
     */
    toDescribeClusterVersions() {
        this.to('redshift:DescribeClusterVersions');
        return this;
    }
    /**
     * Grants permission to describe properties of provisioned clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusters.html
     */
    toDescribeClusters() {
        this.to('redshift:DescribeClusters');
        return this;
    }
    /**
     * Grants permission to describe parameter settings for a parameter group family
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeDefaultClusterParameters.html
     */
    toDescribeDefaultClusterParameters() {
        this.to('redshift:DescribeDefaultClusterParameters');
        return this;
    }
    /**
     * Grants permission to describe event categories for all event source types, or for a specified source type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories() {
        this.to('redshift:DescribeEventCategories');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift event notification subscriptions for the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions() {
        this.to('redshift:DescribeEventSubscriptions');
        return this;
    }
    /**
     * Grants permission to describe events related to clusters, security groups, snapshots, and parameter groups for the past 14 days
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('redshift:DescribeEvents');
        return this;
    }
    /**
     * Grants permission to describe HSM client certificates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmClientCertificates.html
     */
    toDescribeHsmClientCertificates() {
        this.to('redshift:DescribeHsmClientCertificates');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift HSM configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmConfigurations.html
     */
    toDescribeHsmConfigurations() {
        this.to('redshift:DescribeHsmConfigurations');
        return this;
    }
    /**
     * Grants permission to describe whether information, such as queries and connection attempts, is being logged for a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeLoggingStatus.html
     */
    toDescribeLoggingStatus() {
        this.to('redshift:DescribeLoggingStatus');
        return this;
    }
    /**
     * Grants permission to describe properties of possible node configurations such as node type, number of nodes, and disk usage for the specified action type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeNodeConfigurationOptions.html
     */
    toDescribeNodeConfigurationOptions() {
        this.to('redshift:DescribeNodeConfigurationOptions');
        return this;
    }
    /**
     * Grants permission to describe orderable cluster options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeOrderableClusterOptions.html
     */
    toDescribeOrderableClusterOptions() {
        this.to('redshift:DescribeOrderableClusterOptions');
        return this;
    }
    /**
     * Grants permission to describe a query through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toDescribeQuery() {
        this.to('redshift:DescribeQuery');
        return this;
    }
    /**
     * Grants permission to describe available reserved node offerings by Amazon Redshift
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodeOfferings.html
     */
    toDescribeReservedNodeOfferings() {
        this.to('redshift:DescribeReservedNodeOfferings');
        return this;
    }
    /**
     * Grants permission to describe the reserved nodes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodes.html
     */
    toDescribeReservedNodes() {
        this.to('redshift:DescribeReservedNodes');
        return this;
    }
    /**
     * Grants permission to describe the last resize operation for a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeResize.html
     */
    toDescribeResize() {
        this.to('redshift:DescribeResize');
        return this;
    }
    /**
     * Grants permission to describe saved queries through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toDescribeSavedQueries() {
        this.to('redshift:DescribeSavedQueries');
        return this;
    }
    /**
     * Grants permission to describe created Amazon Redshift scheduled actions
     *
     * Access Level: Read
     */
    toDescribeScheduledActions() {
        this.to('redshift:DescribeScheduledActions');
        return this;
    }
    /**
     * Grants permission to describe snapshot copy grants owned by the specified AWS account in the destination AWS Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotCopyGrants.html
     */
    toDescribeSnapshotCopyGrants() {
        this.to('redshift:DescribeSnapshotCopyGrants');
        return this;
    }
    /**
     * Grants permission to describe snapshot schedules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotSchedules.html
     */
    toDescribeSnapshotSchedules() {
        this.to('redshift:DescribeSnapshotSchedules');
        return this;
    }
    /**
     * Grants permission to describe account level backups storage size and provisional storage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeStorage.html
     */
    toDescribeStorage() {
        this.to('redshift:DescribeStorage');
        return this;
    }
    /**
     * Grants permission to describe a table through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toDescribeTable() {
        this.to('redshift:DescribeTable');
        return this;
    }
    /**
     * Grants permission to describe status of one or more table restore requests made using the RestoreTableFromClusterSnapshot API action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTableRestoreStatus.html
     */
    toDescribeTableRestoreStatus() {
        this.to('redshift:DescribeTableRestoreStatus');
        return this;
    }
    /**
     * Grants permission to describe tags
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.to('redshift:DescribeTags');
        return this;
    }
    /**
     * Grants permission to disable logging information, such as queries and connection attempts, for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableLogging.html
     */
    toDisableLogging() {
        this.to('redshift:DisableLogging');
        return this;
    }
    /**
     * Grants permission to disable the automatic copy of snapshots for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableSnapshotCopy.html
     */
    toDisableSnapshotCopy() {
        this.to('redshift:DisableSnapshotCopy');
        return this;
    }
    /**
     * Grants permission to enable logging information, such as queries and connection attempts, for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableLogging.html
     */
    toEnableLogging() {
        this.to('redshift:EnableLogging');
        return this;
    }
    /**
     * Grants permission to enable the automatic copy of snapshots for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableSnapshotCopy.html
     */
    toEnableSnapshotCopy() {
        this.to('redshift:EnableSnapshotCopy');
        return this;
    }
    /**
     * Grants permission to execute a query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toExecuteQuery() {
        this.to('redshift:ExecuteQuery');
        return this;
    }
    /**
     * Grants permission to fetch query results through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toFetchResults() {
        this.to('redshift:FetchResults');
        return this;
    }
    /**
     * Grants permission to get temporary credentials to access an Amazon Redshift database by the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDbName()
     * - .ifDbUser()
     * - .ifDurationSeconds()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html
     */
    toGetClusterCredentials() {
        this.to('redshift:GetClusterCredentials');
        return this;
    }
    /**
     * Grants permission to get an array of DC2 ReservedNodeOfferings that matches the payment type, term, and usage price of the given DC1 reserved node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetReservedNodeExchangeOfferings.html
     */
    toGetReservedNodeExchangeOfferings() {
        this.to('redshift:GetReservedNodeExchangeOfferings');
        return this;
    }
    /**
     * Grants permission to join the specified Amazon Redshift group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html
     */
    toJoinGroup() {
        this.to('redshift:JoinGroup');
        return this;
    }
    /**
     * Grants permission to list databases through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toListDatabases() {
        this.to('redshift:ListDatabases');
        return this;
    }
    /**
     * Grants permission to list saved queries through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toListSavedQueries() {
        this.to('redshift:ListSavedQueries');
        return this;
    }
    /**
     * Grants permission to list schemas through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toListSchemas() {
        this.to('redshift:ListSchemas');
        return this;
    }
    /**
     * Grants permission to list tables through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toListTables() {
        this.to('redshift:ListTables');
        return this;
    }
    /**
     * Grants permission to modify the settings of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyCluster.html
     */
    toModifyCluster() {
        this.to('redshift:ModifyCluster');
        return this;
    }
    /**
     * Grants permission to modify the database revision of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterDbRevision.html
     */
    toModifyClusterDbRevision() {
        this.to('redshift:ModifyClusterDbRevision');
        return this;
    }
    /**
     * Grants permission to modify the list of AWS Identity and Access Management (IAM) roles that can be used by a cluster to access other AWS services
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterIamRoles.html
     */
    toModifyClusterIamRoles() {
        this.to('redshift:ModifyClusterIamRoles');
        return this;
    }
    /**
     * Grants permission to modify the maintenance settings of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterMaintenance.html
     */
    toModifyClusterMaintenance() {
        this.to('redshift:ModifyClusterMaintenance');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterParameterGroup.html
     */
    toModifyClusterParameterGroup() {
        this.to('redshift:ModifyClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to modify the settings of a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshot.html
     */
    toModifyClusterSnapshot() {
        this.to('redshift:ModifyClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to modify a snapshot schedule for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshotSchedule.html
     */
    toModifyClusterSnapshotSchedule() {
        this.to('redshift:ModifyClusterSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to modify a cluster subnet group to include the specified list of VPC subnets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSubnetGroup.html
     */
    toModifyClusterSubnetGroup() {
        this.to('redshift:ModifyClusterSubnetGroup');
        return this;
    }
    /**
     * Grants permission to modify an existing Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription() {
        this.to('redshift:ModifyEventSubscription');
        return this;
    }
    /**
     * Grants permission to modify an existing saved query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toModifySavedQuery() {
        this.to('redshift:ModifySavedQuery');
        return this;
    }
    /**
     * Grants permission to modify an existing Amazon Redshift scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyScheduledAction.html
     */
    toModifyScheduledAction() {
        this.to('redshift:ModifyScheduledAction');
        return this;
    }
    /**
     * Grants permission to modify the number of days to retain snapshots in the destination AWS Region after they are copied from the source AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotCopyRetentionPeriod.html
     */
    toModifySnapshotCopyRetentionPeriod() {
        this.to('redshift:ModifySnapshotCopyRetentionPeriod');
        return this;
    }
    /**
     * Grants permission to modify a snapshot schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotSchedule.html
     */
    toModifySnapshotSchedule() {
        this.to('redshift:ModifySnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to pause a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_PauseCluster.html
     */
    toPauseCluster() {
        this.to('redshift:PauseCluster');
        return this;
    }
    /**
     * Grants permission to purchase a reserved node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_PurchaseReservedNodeOffering.html
     */
    toPurchaseReservedNodeOffering() {
        this.to('redshift:PurchaseReservedNodeOffering');
        return this;
    }
    /**
     * Grants permission to reboot a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RebootCluster.html
     */
    toRebootCluster() {
        this.to('redshift:RebootCluster');
        return this;
    }
    /**
     * Grants permission to set one or more parameters of a parameter group to their default values and set the source values of the parameters to "engine-default"
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResetClusterParameterGroup.html
     */
    toResetClusterParameterGroup() {
        this.to('redshift:ResetClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to change the size of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResizeCluster.html
     */
    toResizeCluster() {
        this.to('redshift:ResizeCluster');
        return this;
    }
    /**
     * Grants permission to create a cluster from a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreFromClusterSnapshot.html
     */
    toRestoreFromClusterSnapshot() {
        this.to('redshift:RestoreFromClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create a table from a table in an Amazon Redshift cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreTableFromClusterSnapshot.html
     */
    toRestoreTableFromClusterSnapshot() {
        this.to('redshift:RestoreTableFromClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to resume a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResumeCluster.html
     */
    toResumeCluster() {
        this.to('redshift:ResumeCluster');
        return this;
    }
    /**
     * Grants permission to revoke an ingress rule in an Amazon Redshift security group for a previously authorized IP range or Amazon EC2 security group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeClusterSecurityGroupIngress.html
     */
    toRevokeClusterSecurityGroupIngress() {
        this.to('redshift:RevokeClusterSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to revoke access from the specified AWS account to restore a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeSnapshotAccess.html
     */
    toRevokeSnapshotAccess() {
        this.to('redshift:RevokeSnapshotAccess');
        return this;
    }
    /**
     * Grants permission to rotate an encryption key for a cluster
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RotateEncryptionKey.html
     */
    toRotateEncryptionKey() {
        this.to('redshift:RotateEncryptionKey');
        return this;
    }
    /**
     * Grants permission to view query results through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toViewQueriesFromConsole() {
        this.to('redshift:ViewQueriesFromConsole');
        return this;
    }
    /**
     * Grants permission to terminate running queries and loads through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    toViewQueriesInConsole() {
        this.to('redshift:ViewQueriesInConsole');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:cluster:${ClusterName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dbgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_GROUP.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbGroup - Identifier for the dbGroup.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDbgroup(clusterName, dbGroup, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:dbgroup:${ClusterName}/${DbGroup}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${DbGroup}', dbGroup);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dbname to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/t_creating_database.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbName - Identifier for the dbName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDbname(clusterName, dbName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:dbname:${ClusterName}/${DbName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${DbName}', dbName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dbuser to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/r_Users.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbUser - Identifier for the dbUser.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDbuser(clusterName, dbUser, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:dbuser:${ClusterName}/${DbUser}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${DbUser}', dbUser);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type eventsubscription to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-events.html
     *
     * @param eventSubscriptionName - Identifier for the eventSubscriptionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventsubscription(eventSubscriptionName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:eventsubscription:${EventSubscriptionName}';
        arn = arn.replace('${EventSubscriptionName}', eventSubscriptionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hsmclientcertificate to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM
     *
     * @param hSMClientCertificateId - Identifier for the hSMClientCertificateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHsmclientcertificate(hSMClientCertificateId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:hsmclientcertificate:${HSMClientCertificateId}';
        arn = arn.replace('${HSMClientCertificateId}', hSMClientCertificateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hsmconfiguration to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM
     *
     * @param hSMConfigurationId - Identifier for the hSMConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHsmconfiguration(hSMConfigurationId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:hsmconfiguration:${HSMConfigurationId}';
        arn = arn.replace('${HSMConfigurationId}', hSMConfigurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-parameter-groups.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onParametergroup(parameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:parametergroup:${ParameterGroupName}';
        arn = arn.replace('${ParameterGroupName}', parameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param owner - Identifier for the owner.
     * @param ec2SecurityGroupId - Identifier for the ec2SecurityGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecuritygroup(securityGroupName, owner, ec2SecurityGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:securitygroup:${SecurityGroupName}/ec2securitygroup/${Owner}/${Ec2SecurityGroupId}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${Owner}', owner);
        arn = arn.replace('${Ec2SecurityGroupId}', ec2SecurityGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroupingress-cidr to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param ipRange - Identifier for the ipRange.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecuritygroupingressCidr(securityGroupName, ipRange, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:securitygroupingress:${SecurityGroupName}/cidrip/${IpRange}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${IpRange}', ipRange);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroupingress-ec2securitygroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param owner - Identifier for the owner.
     * @param ece2SecuritygroupId - Identifier for the ece2SecuritygroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecuritygroupingressEc2securitygroup(securityGroupName, owner, ece2SecuritygroupId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:securitygroupingress:${SecurityGroupName}/ec2securitygroup/${Owner}/${Ece2SecuritygroupId}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${Owner}', owner);
        arn = arn.replace('${Ece2SecuritygroupId}', ece2SecuritygroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(clusterName, snapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:snapshot:${ClusterName}/${SnapshotName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${SnapshotName}', snapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshotcopygrant to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#configure-snapshot-copy-grant
     *
     * @param snapshotCopyGrantName - Identifier for the snapshotCopyGrantName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshotcopygrant(snapshotCopyGrantName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:snapshotcopygrant:${SnapshotCopyGrantName}';
        arn = arn.replace('${SnapshotCopyGrantName}', snapshotCopyGrantName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshotschedule to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshotschedule(parameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:snapshotschedule:${ParameterGroupName}';
        arn = arn.replace('${ParameterGroupName}', parameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-cluster-subnet-groups.html
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubnetgroup(subnetGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:subnetgroup:${SubnetGroupName}';
        arn = arn.replace('${SubnetGroupName}', subnetGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the database name
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .toGetClusterCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbName(value, operator) {
        return this.if(`redshift:DbName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the database user name
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .toCreateClusterUser()
     * - .toGetClusterCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbUser(value, operator) {
        return this.if(`redshift:DbUser`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the number of seconds until a temporary credential set expires
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .toGetClusterCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDurationSeconds(value, operator) {
        return this.if(`redshift:DurationSeconds`, value, operator || 'StringLike');
    }
}
exports.Redshift = Redshift;
//# sourceMappingURL=data:application/json;base64,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