"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Outposts = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [s3-outposts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3onoutposts.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3Outposts extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [s3-outposts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3onoutposts.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 's3-outposts';
        this.accessLevelList = {
            "Write": [
                "AbortMultipartUpload",
                "CreateAccessPoint",
                "CreateBucket",
                "CreateEndpoint",
                "DeleteAccessPoint",
                "DeleteBucket",
                "DeleteEndpoint",
                "DeleteObject",
                "PutLifecycleConfiguration",
                "PutObject"
            ],
            "Permissions management": [
                "DeleteAccessPointPolicy",
                "DeleteBucketPolicy",
                "PutAccessPointPolicy",
                "PutBucketPolicy",
                "PutObjectAcl"
            ],
            "Tagging": [
                "DeleteObjectTagging",
                "PutBucketTagging",
                "PutObjectTagging"
            ],
            "Read": [
                "GetAccessPoint",
                "GetAccessPointPolicy",
                "GetBucket",
                "GetBucketPolicy",
                "GetBucketTagging",
                "GetLifecycleConfiguration",
                "GetObject",
                "GetObjectTagging"
            ],
            "List": [
                "ListAccessPoints",
                "ListBucket",
                "ListBucketMultipartUploads",
                "ListEndpoints",
                "ListMultipartUploadParts",
                "ListRegionalBuckets"
            ]
        };
    }
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    toAbortMultipartUpload() {
        this.to('s3-outposts:AbortMultipartUpload');
        return this;
    }
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        this.to('s3-outposts:CreateAccessPoint');
        return this;
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateBucket.html
     */
    toCreateBucket() {
        this.to('s3-outposts:CreateBucket');
        return this;
    }
    /**
     * Grants permission to create a new endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html
     */
    toCreateEndpoint() {
        this.to('s3-outposts:CreateEndpoint');
        return this;
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        this.to('s3-outposts:DeleteAccessPoint');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy() {
        this.to('s3-outposts:DeleteAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucket.html
     */
    toDeleteBucket() {
        this.to('s3-outposts:DeleteBucket');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy() {
        this.to('s3-outposts:DeleteBucketPolicy');
        return this;
    }
    /**
     * Grants permission to delete the endpoint named in the URI
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.to('s3-outposts:DeleteEndpoint');
        return this;
    }
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObject() {
        this.to('s3-outposts:DeleteObject');
        return this;
    }
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectTagging() {
        this.to('s3-outposts:DeleteObjectTagging');
        return this;
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint() {
        this.to('s3-outposts:GetAccessPoint');
        return this;
    }
    /**
     * Grants permission to returns the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy() {
        this.to('s3-outposts:GetAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to return the bucket configuration associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucket.html
     */
    toGetBucket() {
        this.to('s3-outposts:GetBucket');
        return this;
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketPolicy.html
     */
    toGetBucketPolicy() {
        this.to('s3-outposts:GetBucketPolicy');
        return this;
    }
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketTagging.html
     */
    toGetBucketTagging() {
        this.to('s3-outposts:GetBucketTagging');
        return this;
    }
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration() {
        this.to('s3-outposts:GetLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObject() {
        this.to('s3-outposts:GetObject');
        return this;
    }
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    toGetObjectTagging() {
        this.to('s3-outposts:GetObjectTagging');
        return this;
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    toListAccessPoints() {
        this.to('s3-outposts:ListAccessPoints');
        return this;
    }
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    toListBucket() {
        this.to('s3-outposts:ListBucket');
        return this;
    }
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    toListBucketMultipartUploads() {
        this.to('s3-outposts:ListBucketMultipartUploads');
        return this;
    }
    /**
     * Grants permission to list endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html
     */
    toListEndpoints() {
        this.to('s3-outposts:ListEndpoints');
        return this;
    }
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    toListMultipartUploadParts() {
        this.to('s3-outposts:ListMultipartUploadParts');
        return this;
    }
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListRegionalBuckets.html
     */
    toListRegionalBuckets() {
        this.to('s3-outposts:ListRegionalBuckets');
        return this;
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy() {
        this.to('s3-outposts:PutAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketPolicy.html
     */
    toPutBucketPolicy() {
        this.to('s3-outposts:PutBucketPolicy');
        return this;
    }
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketTagging.html
     */
    toPutBucketTagging() {
        this.to('s3-outposts:PutBucketTagging');
        return this;
    }
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        this.to('s3-outposts:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    toPutObject() {
        this.to('s3-outposts:PutObject');
        return this;
    }
    /**
     * Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectAcl() {
        this.to('s3-outposts:PutObjectAcl');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectTagging() {
        this.to('s3-outposts:PutObjectTagging');
        return this;
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html
     *
     * @param outpostId - Identifier for the outpostId.
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccesspoint(outpostId, accessPointName, account, region, partition) {
        var arn = 'arn:${Partition}:s3-outposts:${Region}:${Account}:outpost/${OutpostId}/accesspoint/${AccessPointName}';
        arn = arn.replace('${OutpostId}', outpostId);
        arn = arn.replace('${AccessPointName}', accessPointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html
     *
     * @param outpostId - Identifier for the outpostId.
     * @param bucketName - Identifier for the bucketName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBucket(outpostId, bucketName, account, region, partition) {
        var arn = 'arn:${Partition}:s3-outposts:${Region}:${Account}:outpost/${OutpostId}/bucket/${BucketName}';
        arn = arn.replace('${OutpostId}', outpostId);
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/outposts-endpoints.html
     *
     * @param outpostId - Identifier for the outpostId.
     * @param endpointId - Identifier for the endpointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEndpoint(outpostId, endpointId, account, region, partition) {
        var arn = 'arn:${Partition}:s3-outposts:${Region}:${Account}:outpost/${OutpostId}/endpoint/${EndpointId}';
        arn = arn.replace('${OutpostId}', outpostId);
        arn = arn.replace('${EndpointId}', endpointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html
     *
     * @param outpostId - Identifier for the outpostId.
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onObject(outpostId, bucketName, objectName, account, region, partition) {
        var arn = 'arn:${Partition}:s3-outposts:${Region}:${Account}:outpost/${OutpostId}/bucket/${BucketName}/object/${ObjectName}';
        arn = arn.replace('${OutpostId}', outpostId);
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${ObjectName}', objectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`s3-outposts:AccessPointNetworkOrigin`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`s3-outposts:DataAccessPointAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`s3-outposts:DataAccessPointArn`, value, operator || 'StringLike');
    }
    /**
     * Filters access by requiring that an existing object tag has a specific tag key and value
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toDeleteObjectTagging()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key, value, operator) {
        return this.if(`s3-outposts:ExistingObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the tag keys and values allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key, value, operator) {
        return this.if(`s3-outposts:RequestObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the tag keys allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value, operator) {
        return this.if(`s3-outposts:RequestObjectTagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting incoming requests to a specific authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetBucket()
     * - .toGetBucketPolicy()
     * - .toGetBucketTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toListAccessPoints()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListMultipartUploadParts()
     * - .toListRegionalBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutBucketPolicy()
     * - .toPutBucketTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`s3-outposts:authType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by requiring the delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/walkthrough1.html
     *
     * Applies to actions:
     * - .toListBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value, operator) {
        return this.if(`s3-outposts:delimiter`, value, operator || 'StringLike');
    }
    /**
     * Filters access by limiting the maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * Applies to actions:
     * - .toListBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value, operator) {
        return this.if(`s3-outposts:max-keys`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * Applies to actions:
     * - .toListBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value, operator) {
        return this.if(`s3-outposts:prefix`, value, operator || 'StringLike');
    }
    /**
     * Filters access by identifying the length of time, in milliseconds, that a signature is valid in an authenticated request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetBucket()
     * - .toGetBucketPolicy()
     * - .toGetBucketTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toListAccessPoints()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListMultipartUploadParts()
     * - .toListRegionalBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutBucketPolicy()
     * - .toPutBucketTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`s3-outposts:signatureAge`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by identifying the version of AWS Signature that is supported for authenticated requests
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetBucket()
     * - .toGetBucketPolicy()
     * - .toGetBucketTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toListAccessPoints()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListMultipartUploadParts()
     * - .toListRegionalBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutBucketPolicy()
     * - .toPutBucketTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`s3-outposts:signatureversion`, value, operator || 'StringLike');
    }
    /**
     * Filters access by requiring the x-amz-acl header with a specific canned ACL in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value, operator) {
        return this.if(`s3-outposts:x-amz-acl`, value, operator || 'StringLike');
    }
    /**
     * Filters access by disallowing unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetBucket()
     * - .toGetBucketPolicy()
     * - .toGetBucketTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetObject()
     * - .toGetObjectTagging()
     * - .toListAccessPoints()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListMultipartUploadParts()
     * - .toListRegionalBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutBucketPolicy()
     * - .toPutBucketTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`s3-outposts:x-amz-content-sha256`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the copy source to a specific bucket, prefix, or object
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * Applies to actions:
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value, operator) {
        return this.if(`s3-outposts:x-amz-copy-source`, value, operator || 'StringLike');
    }
    /**
     * Filters access by enabling enforcement of object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * Applies to actions:
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value, operator) {
        return this.if(`s3-outposts:x-amz-metadata-directive`, value, operator || 'StringLike');
    }
    /**
     * Filters access by requiring server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     *
     * Applies to actions:
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`s3-outposts:x-amz-server-side-encryption`, value, operator || 'StringLike');
    }
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-class-intro.html#sc-howtoset
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value, operator) {
        return this.if(`s3-outposts:x-amz-storage-class`, value, operator || 'StringLike');
    }
}
exports.S3Outposts = S3Outposts;
//# sourceMappingURL=data:application/json;base64,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