"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ses = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ses](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonses.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ses extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ses](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonses.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ses';
        this.accessLevelList = {
            "Write": [
                "CloneReceiptRuleSet",
                "CreateConfigurationSet",
                "CreateConfigurationSetEventDestination",
                "CreateConfigurationSetTrackingOptions",
                "CreateCustomVerificationEmailTemplate",
                "CreateReceiptFilter",
                "CreateReceiptRule",
                "CreateReceiptRuleSet",
                "CreateTemplate",
                "DeleteConfigurationSet",
                "DeleteConfigurationSetEventDestination",
                "DeleteConfigurationSetTrackingOptions",
                "DeleteCustomVerificationEmailTemplate",
                "DeleteIdentity",
                "DeleteIdentityPolicy",
                "DeleteReceiptFilter",
                "DeleteReceiptRule",
                "DeleteReceiptRuleSet",
                "DeleteTemplate",
                "DeleteVerifiedEmailAddress",
                "PutIdentityPolicy",
                "ReorderReceiptRuleSet",
                "SendBounce",
                "SendBulkTemplatedEmail",
                "SendCustomVerificationEmail",
                "SendEmail",
                "SendRawEmail",
                "SendTemplatedEmail",
                "SetActiveReceiptRuleSet",
                "SetIdentityDkimEnabled",
                "SetIdentityFeedbackForwardingEnabled",
                "SetIdentityHeadersInNotificationsEnabled",
                "SetIdentityMailFromDomain",
                "SetIdentityNotificationTopic",
                "SetReceiptRulePosition",
                "TestRenderTemplate",
                "UpdateAccountSendingEnabled",
                "UpdateConfigurationSetEventDestination",
                "UpdateConfigurationSetReputationMetricsEnabled",
                "UpdateConfigurationSetSendingEnabled",
                "UpdateConfigurationSetTrackingOptions",
                "UpdateCustomVerificationEmailTemplate",
                "UpdateReceiptRule",
                "UpdateTemplate"
            ],
            "Read": [
                "DescribeActiveReceiptRuleSet",
                "DescribeConfigurationSet",
                "DescribeReceiptRule",
                "DescribeReceiptRuleSet",
                "GetAccountSendingEnabled",
                "GetCustomVerificationEmailTemplate",
                "GetIdentityDkimAttributes",
                "GetIdentityMailFromDomainAttributes",
                "GetIdentityNotificationAttributes",
                "GetIdentityPolicies",
                "GetIdentityVerificationAttributes",
                "GetSendQuota",
                "GetSendStatistics",
                "GetTemplate",
                "VerifyDomainDkim",
                "VerifyDomainIdentity",
                "VerifyEmailAddress",
                "VerifyEmailIdentity"
            ],
            "List": [
                "ListConfigurationSets",
                "ListCustomVerificationEmailTemplates",
                "ListIdentities",
                "ListIdentityPolicies",
                "ListReceiptFilters",
                "ListReceiptRuleSets",
                "ListTemplates",
                "ListVerifiedEmailAddresses"
            ]
        };
    }
    /**
     * Creates a receipt rule set by cloning an existing one
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html
     */
    toCloneReceiptRuleSet() {
        this.to('ses:CloneReceiptRuleSet');
        return this;
    }
    /**
     * Creates a new configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        this.to('ses:CreateConfigurationSet');
        return this;
    }
    /**
     * Creates a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination() {
        this.to('ses:CreateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Creates an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html
     */
    toCreateConfigurationSetTrackingOptions() {
        this.to('ses:CreateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Creates a new custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html
     */
    toCreateCustomVerificationEmailTemplate() {
        this.to('ses:CreateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Creates a new IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html
     */
    toCreateReceiptFilter() {
        this.to('ses:CreateReceiptFilter');
        return this;
    }
    /**
     * Creates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html
     */
    toCreateReceiptRule() {
        this.to('ses:CreateReceiptRule');
        return this;
    }
    /**
     * Creates an empty receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html
     */
    toCreateReceiptRuleSet() {
        this.to('ses:CreateReceiptRuleSet');
        return this;
    }
    /**
     * Creates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate() {
        this.to('ses:CreateTemplate');
        return this;
    }
    /**
     * Deletes the configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        this.to('ses:DeleteConfigurationSet');
        return this;
    }
    /**
     * Deletes a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination() {
        this.to('ses:DeleteConfigurationSetEventDestination');
        return this;
    }
    /**
     * Deletes an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html
     */
    toDeleteConfigurationSetTrackingOptions() {
        this.to('ses:DeleteConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Deletes an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html
     */
    toDeleteCustomVerificationEmailTemplate() {
        this.to('ses:DeleteCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html
     */
    toDeleteIdentity() {
        this.to('ses:DeleteIdentity');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html
     */
    toDeleteIdentityPolicy() {
        this.to('ses:DeleteIdentityPolicy');
        return this;
    }
    /**
     * Deletes the specified IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html
     */
    toDeleteReceiptFilter() {
        this.to('ses:DeleteReceiptFilter');
        return this;
    }
    /**
     * Deletes the specified receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html
     */
    toDeleteReceiptRule() {
        this.to('ses:DeleteReceiptRule');
        return this;
    }
    /**
     * Deletes the specified receipt rule set and all of the receipt rules it contains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html
     */
    toDeleteReceiptRuleSet() {
        this.to('ses:DeleteReceiptRuleSet');
        return this;
    }
    /**
     * Deletes an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate() {
        this.to('ses:DeleteTemplate');
        return this;
    }
    /**
     * Deletes the specified email address from the list of verified addresses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html
     */
    toDeleteVerifiedEmailAddress() {
        this.to('ses:DeleteVerifiedEmailAddress');
        return this;
    }
    /**
     * Returns the metadata and receipt rules for the receipt rule set that is currently active
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html
     */
    toDescribeActiveReceiptRuleSet() {
        this.to('ses:DescribeActiveReceiptRuleSet');
        return this;
    }
    /**
     * Returns the details of the specified configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html
     */
    toDescribeConfigurationSet() {
        this.to('ses:DescribeConfigurationSet');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html
     */
    toDescribeReceiptRule() {
        this.to('ses:DescribeReceiptRule');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html
     */
    toDescribeReceiptRuleSet() {
        this.to('ses:DescribeReceiptRuleSet');
        return this;
    }
    /**
     * Returns the email sending status of the Amazon SES account for the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html
     */
    toGetAccountSendingEnabled() {
        this.to('ses:GetAccountSendingEnabled');
        return this;
    }
    /**
     * Returns the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html
     */
    toGetCustomVerificationEmailTemplate() {
        this.to('ses:GetCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Returns the current status of Easy DKIM signing for an entity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html
     */
    toGetIdentityDkimAttributes() {
        this.to('ses:GetIdentityDkimAttributes');
        return this;
    }
    /**
     * Returns the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html
     */
    toGetIdentityMailFromDomainAttributes() {
        this.to('ses:GetIdentityMailFromDomainAttributes');
        return this;
    }
    /**
     * Given a list of verified identities (email addresses and/or domains), returns a structure describing identity notification attributes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html
     */
    toGetIdentityNotificationAttributes() {
        this.to('ses:GetIdentityNotificationAttributes');
        return this;
    }
    /**
     * Returns the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html
     */
    toGetIdentityPolicies() {
        this.to('ses:GetIdentityPolicies');
        return this;
    }
    /**
     * Given a list of identities (email addresses and/or domains), returns the verification status and (for domain identities) the verification token for each identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html
     */
    toGetIdentityVerificationAttributes() {
        this.to('ses:GetIdentityVerificationAttributes');
        return this;
    }
    /**
     * Returns the user's current sending limits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html
     */
    toGetSendQuota() {
        this.to('ses:GetSendQuota');
        return this;
    }
    /**
     * Returns the user's sending statistics. The result is a list of data points, representing the last two weeks of sending activity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html
     */
    toGetSendStatistics() {
        this.to('ses:GetSendStatistics');
        return this;
    }
    /**
     * Returns the template object (which includes the Subject line, HTML part and text part) for the template you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        this.to('ses:GetTemplate');
        return this;
    }
    /**
     * Returns a list of the configuration sets associated with your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html
     */
    toListConfigurationSets() {
        this.to('ses:ListConfigurationSets');
        return this;
    }
    /**
     * Lists the existing custom verification email templates for your account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html
     */
    toListCustomVerificationEmailTemplates() {
        this.to('ses:ListCustomVerificationEmailTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the identities (email addresses and domains) for your AWS account, regardless of verification status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html
     */
    toListIdentities() {
        this.to('ses:ListIdentities');
        return this;
    }
    /**
     * Returns a list of sending authorization policies that are attached to the given identity (an email address or a domain)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html
     */
    toListIdentityPolicies() {
        this.to('ses:ListIdentityPolicies');
        return this;
    }
    /**
     * Lists the IP address filters associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html
     */
    toListReceiptFilters() {
        this.to('ses:ListReceiptFilters');
        return this;
    }
    /**
     * Lists the receipt rule sets that exist under your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html
     */
    toListReceiptRuleSets() {
        this.to('ses:ListReceiptRuleSets');
        return this;
    }
    /**
     * Lists the email templates present in your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates() {
        this.to('ses:ListTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the email addresses that have been verified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html
     */
    toListVerifiedEmailAddresses() {
        this.to('ses:ListVerifiedEmailAddresses');
        return this;
    }
    /**
     * Adds or updates a sending authorization policy for the specified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html
     */
    toPutIdentityPolicy() {
        this.to('ses:PutIdentityPolicy');
        return this;
    }
    /**
     * Reorders the receipt rules within a receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html
     */
    toReorderReceiptRuleSet() {
        this.to('ses:ReorderReceiptRuleSet');
        return this;
    }
    /**
     * Generates and sends a bounce message to the sender of an email you received through Amazon SES
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromAddress()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html
     */
    toSendBounce() {
        this.to('ses:SendBounce');
        return this;
    }
    /**
     * Composes an email message to multiple destinations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html
     */
    toSendBulkTemplatedEmail() {
        this.to('ses:SendBulkTemplatedEmail');
        return this;
    }
    /**
     * Adds an email address to the list of identities for your Amazon SES account in the current AWS Region and attempts to verify it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html
     */
    toSendCustomVerificationEmail() {
        this.to('ses:SendCustomVerificationEmail');
        return this;
    }
    /**
     * Composes an email message based on input data, and then immediately queues the message for sending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html
     */
    toSendEmail() {
        this.to('ses:SendEmail');
        return this;
    }
    /**
     * Sends an email message, with header and content specified by the client
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html
     */
    toSendRawEmail() {
        this.to('ses:SendRawEmail');
        return this;
    }
    /**
     * Composes an email message using an email template and immediately queues it for sending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html
     */
    toSendTemplatedEmail() {
        this.to('ses:SendTemplatedEmail');
        return this;
    }
    /**
     * Sets the specified receipt rule set as the active receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html
     */
    toSetActiveReceiptRuleSet() {
        this.to('ses:SetActiveReceiptRuleSet');
        return this;
    }
    /**
     * Enables or disables Easy DKIM signing of email sent from an identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html
     */
    toSetIdentityDkimEnabled() {
        this.to('ses:SetIdentityDkimEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), enables or disables whether Amazon SES forwards bounce and complaint notifications as email
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html
     */
    toSetIdentityFeedbackForwardingEnabled() {
        this.to('ses:SetIdentityFeedbackForwardingEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html
     */
    toSetIdentityHeadersInNotificationsEnabled() {
        this.to('ses:SetIdentityHeadersInNotificationsEnabled');
        return this;
    }
    /**
     * Enables or disables the custom MAIL FROM domain setup for a verified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html
     */
    toSetIdentityMailFromDomain() {
        this.to('ses:SetIdentityMailFromDomain');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets the Amazon Simple Notification Service (Amazon SNS) topic to which Amazon SES will publish bounce, complaint, and/or delivery notifications for emails sent with that identity as the Source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html
     */
    toSetIdentityNotificationTopic() {
        this.to('ses:SetIdentityNotificationTopic');
        return this;
    }
    /**
     * Sets the position of the specified receipt rule in the receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html
     */
    toSetReceiptRulePosition() {
        this.to('ses:SetReceiptRulePosition');
        return this;
    }
    /**
     * Creates a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html
     */
    toTestRenderTemplate() {
        this.to('ses:TestRenderTemplate');
        return this;
    }
    /**
     * Enables or disables email sending across your entire Amazon SES account in the current AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html
     */
    toUpdateAccountSendingEnabled() {
        this.to('ses:UpdateAccountSendingEnabled');
        return this;
    }
    /**
     * Updates the event destination of a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination() {
        this.to('ses:UpdateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Enables or disables the publishing of reputation metrics for emails sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html
     */
    toUpdateConfigurationSetReputationMetricsEnabled() {
        this.to('ses:UpdateConfigurationSetReputationMetricsEnabled');
        return this;
    }
    /**
     * Enables or disables email sending for messages sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html
     */
    toUpdateConfigurationSetSendingEnabled() {
        this.to('ses:UpdateConfigurationSetSendingEnabled');
        return this;
    }
    /**
     * Modifies an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html
     */
    toUpdateConfigurationSetTrackingOptions() {
        this.to('ses:UpdateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Updates an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html
     */
    toUpdateCustomVerificationEmailTemplate() {
        this.to('ses:UpdateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Updates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html
     */
    toUpdateReceiptRule() {
        this.to('ses:UpdateReceiptRule');
        return this;
    }
    /**
     * Updates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate() {
        this.to('ses:UpdateTemplate');
        return this;
    }
    /**
     * Returns a set of DKIM tokens for a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html
     */
    toVerifyDomainDkim() {
        this.to('ses:VerifyDomainDkim');
        return this;
    }
    /**
     * Verifies a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html
     */
    toVerifyDomainIdentity() {
        this.to('ses:VerifyDomainIdentity');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html
     */
    toVerifyEmailAddress() {
        this.to('ses:VerifyEmailAddress');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html
     */
    toVerifyEmailIdentity() {
        this.to('ses:VerifyEmailIdentity');
        return this;
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCustomVerificationEmailTemplate(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:custom-verification-email-template/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-destination to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_EventDestination.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param eventDestinationName - Identifier for the eventDestinationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventDestination(configurationSetName, eventDestinationName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${EventDestinationName}', eventDestinationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}';
        arn = arn.replace('${IdentityName}', identityName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-filter to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptFilter.html
     *
     * @param receiptFilterName - Identifier for the receiptFilterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptFilter(receiptFilterName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-filter/${ReceiptFilterName}';
        arn = arn.replace('${ReceiptFilterName}', receiptFilterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRule.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param receiptRuleName - Identifier for the receiptRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRule(receiptRuleSetName, receiptRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}:receipt-rule/${ReceiptRuleName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${ReceiptRuleName}', receiptRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRuleSetMetadata.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRuleSet(receiptRuleSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTemplate(templateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:template/${TemplateName}';
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`ses:FeedbackAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBounce()
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`ses:FromAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address that is used as the display name of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`ses:FromDisplayName`, value, operator || 'StringLike');
    }
    /**
     * The recipient addresses of a message, which include the "To", "CC", and "BCC" addresses.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`ses:Recipients`, value, operator || 'StringLike');
    }
}
exports.Ses = Ses;
//# sourceMappingURL=data:application/json;base64,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