"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Swf = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [swf](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleworkflowservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Swf extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [swf](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleworkflowservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'swf';
        this.accessLevelList = {
            "Write": [
                "CancelTimer",
                "CancelWorkflowExecution",
                "CompleteWorkflowExecution",
                "ContinueAsNewWorkflowExecution",
                "DeprecateActivityType",
                "DeprecateDomain",
                "DeprecateWorkflowType",
                "FailWorkflowExecution",
                "PollForActivityTask",
                "PollForDecisionTask",
                "RecordActivityTaskHeartbeat",
                "RecordMarker",
                "RegisterActivityType",
                "RegisterDomain",
                "RegisterWorkflowType",
                "RequestCancelActivityTask",
                "RequestCancelExternalWorkflowExecution",
                "RequestCancelWorkflowExecution",
                "RespondActivityTaskCanceled",
                "RespondActivityTaskCompleted",
                "RespondActivityTaskFailed",
                "RespondDecisionTaskCompleted",
                "ScheduleActivityTask",
                "SignalExternalWorkflowExecution",
                "SignalWorkflowExecution",
                "StartChildWorkflowExecution",
                "StartTimer",
                "StartWorkflowExecution",
                "TerminateWorkflowExecution"
            ],
            "Read": [
                "CountClosedWorkflowExecutions",
                "CountOpenWorkflowExecutions",
                "CountPendingActivityTasks",
                "CountPendingDecisionTasks",
                "DescribeActivityType",
                "DescribeDomain",
                "DescribeWorkflowExecution",
                "DescribeWorkflowType",
                "GetWorkflowExecutionHistory"
            ],
            "List": [
                "ListActivityTypes",
                "ListClosedWorkflowExecutions",
                "ListDomains",
                "ListOpenWorkflowExecutions",
                "ListTagsForResource",
                "ListWorkflowTypes"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Description for CancelTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelTimer.html
     */
    toCancelTimer() {
        this.to('swf:CancelTimer');
        return this;
    }
    /**
     * Description for CancelWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelWorkflowExecution.html
     */
    toCancelWorkflowExecution() {
        this.to('swf:CancelWorkflowExecution');
        return this;
    }
    /**
     * Description for CompleteWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CompleteWorkflowExecution.html
     */
    toCompleteWorkflowExecution() {
        this.to('swf:CompleteWorkflowExecution');
        return this;
    }
    /**
     * Description for ContinueAsNewWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ContinueAsNewWorkflowExecution.html
     */
    toContinueAsNewWorkflowExecution() {
        this.to('swf:ContinueAsNewWorkflowExecution');
        return this;
    }
    /**
     * Returns the number of closed workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html
     */
    toCountClosedWorkflowExecutions() {
        this.to('swf:CountClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the number of open workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html
     */
    toCountOpenWorkflowExecutions() {
        this.to('swf:CountOpenWorkflowExecutions');
        return this;
    }
    /**
     * Returns the estimated number of activity tasks in the specified task list.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html
     */
    toCountPendingActivityTasks() {
        this.to('swf:CountPendingActivityTasks');
        return this;
    }
    /**
     * Returns the estimated number of decision tasks in the specified task list.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html
     */
    toCountPendingDecisionTasks() {
        this.to('swf:CountPendingDecisionTasks');
        return this;
    }
    /**
     * Deprecates the specified activity type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html
     */
    toDeprecateActivityType() {
        this.to('swf:DeprecateActivityType');
        return this;
    }
    /**
     * Deprecates the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html
     */
    toDeprecateDomain() {
        this.to('swf:DeprecateDomain');
        return this;
    }
    /**
     * Deprecates the specified workflow type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html
     */
    toDeprecateWorkflowType() {
        this.to('swf:DeprecateWorkflowType');
        return this;
    }
    /**
     * Returns information about the specified activity type.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html
     */
    toDescribeActivityType() {
        this.to('swf:DescribeActivityType');
        return this;
    }
    /**
     * Returns information about the specified domain, including description and status.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html
     */
    toDescribeDomain() {
        this.to('swf:DescribeDomain');
        return this;
    }
    /**
     * Returns information about the specified workflow execution including its type and some statistics.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html
     */
    toDescribeWorkflowExecution() {
        this.to('swf:DescribeWorkflowExecution');
        return this;
    }
    /**
     * Returns information about the specified workflow type.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html
     */
    toDescribeWorkflowType() {
        this.to('swf:DescribeWorkflowType');
        return this;
    }
    /**
     * Description for FailWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_FailWorkflowExecution.html
     */
    toFailWorkflowExecution() {
        this.to('swf:FailWorkflowExecution');
        return this;
    }
    /**
     * Returns the history of the specified workflow execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html
     */
    toGetWorkflowExecutionHistory() {
        this.to('swf:GetWorkflowExecutionHistory');
        return this;
    }
    /**
     * Returns information about all activities registered in the specified domain that match the specified name and registration status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html
     */
    toListActivityTypes() {
        this.to('swf:ListActivityTypes');
        return this;
    }
    /**
     * Returns a list of closed workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html
     */
    toListClosedWorkflowExecutions() {
        this.to('swf:ListClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the list of domains registered in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html
     */
    toListDomains() {
        this.to('swf:ListDomains');
        return this;
    }
    /**
     * Returns a list of open workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html
     */
    toListOpenWorkflowExecutions() {
        this.to('swf:ListOpenWorkflowExecutions');
        return this;
    }
    /**
     * This action lists tags for an AWS SWF resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('swf:ListTagsForResource');
        return this;
    }
    /**
     * Returns information about workflow types in the specified domain.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html
     */
    toListWorkflowTypes() {
        this.to('swf:ListWorkflowTypes');
        return this;
    }
    /**
     * Used by workers to get an ActivityTask from the specified activity taskList.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html
     */
    toPollForActivityTask() {
        this.to('swf:PollForActivityTask');
        return this;
    }
    /**
     * Used by deciders to get a DecisionTask from the specified decision taskList.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html
     */
    toPollForDecisionTask() {
        this.to('swf:PollForDecisionTask');
        return this;
    }
    /**
     * Used by activity workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html
     */
    toRecordActivityTaskHeartbeat() {
        this.to('swf:RecordActivityTaskHeartbeat');
        return this;
    }
    /**
     * Description for RecordMarker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordMarker.html
     */
    toRecordMarker() {
        this.to('swf:RecordMarker');
        return this;
    }
    /**
     * Registers a new activity type along with its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDefaultTaskListName()
     * - .ifName()
     * - .ifVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html
     */
    toRegisterActivityType() {
        this.to('swf:RegisterActivityType');
        return this;
    }
    /**
     * Registers a new domain.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html
     */
    toRegisterDomain() {
        this.to('swf:RegisterDomain');
        return this;
    }
    /**
     * Registers a new workflow type and its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDefaultTaskListName()
     * - .ifName()
     * - .ifVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html
     */
    toRegisterWorkflowType() {
        this.to('swf:RegisterWorkflowType');
        return this;
    }
    /**
     * Description for RequestCancelActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelActivityTask.html
     */
    toRequestCancelActivityTask() {
        this.to('swf:RequestCancelActivityTask');
        return this;
    }
    /**
     * Description for RequestCancelExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelExternalWorkflowExecution.html
     */
    toRequestCancelExternalWorkflowExecution() {
        this.to('swf:RequestCancelExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html
     */
    toRequestCancelWorkflowExecution() {
        this.to('swf:RequestCancelWorkflowExecution');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html
     */
    toRespondActivityTaskCanceled() {
        this.to('swf:RespondActivityTaskCanceled');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     * - .ifTagListMember0()
     * - .ifTagListMember1()
     * - .ifTagListMember2()
     * - .ifTagListMember3()
     * - .ifTagListMember4()
     * - .ifTaskListName()
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html
     */
    toRespondActivityTaskCompleted() {
        this.to('swf:RespondActivityTaskCompleted');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html
     */
    toRespondActivityTaskFailed() {
        this.to('swf:RespondActivityTaskFailed');
        return this;
    }
    /**
     * Used by deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html
     */
    toRespondDecisionTaskCompleted() {
        this.to('swf:RespondDecisionTaskCompleted');
        return this;
    }
    /**
     * Description for ScheduleActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ScheduleActivityTask.html
     */
    toScheduleActivityTask() {
        this.to('swf:ScheduleActivityTask');
        return this;
    }
    /**
     * Description for SignalExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalExternalWorkflowExecution.html
     */
    toSignalExternalWorkflowExecution() {
        this.to('swf:SignalExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionSignaled event in the workflow execution history and creates a decision task for the workflow execution identified by the given domain, workflowId and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html
     */
    toSignalWorkflowExecution() {
        this.to('swf:SignalWorkflowExecution');
        return this;
    }
    /**
     * Description for StartChildWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartChildWorkflowExecution.html
     */
    toStartChildWorkflowExecution() {
        this.to('swf:StartChildWorkflowExecution');
        return this;
    }
    /**
     * Description for StartTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartTimer.html
     */
    toStartTimer() {
        this.to('swf:StartTimer');
        return this;
    }
    /**
     * Starts an execution of the workflow type in the specified domain using the provided workflowId and input data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTagListMember0()
     * - .ifTagListMember1()
     * - .ifTagListMember2()
     * - .ifTagListMember3()
     * - .ifTagListMember4()
     * - .ifTaskListName()
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html
     */
    toStartWorkflowExecution() {
        this.to('swf:StartWorkflowExecution');
        return this;
    }
    /**
     * This action tags an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        this.to('swf:TagResource');
        return this;
    }
    /**
     * Records a WorkflowExecutionTerminated event and forces closure of the workflow execution identified by the given domain, runId, and workflowId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html
     */
    toTerminateWorkflowExecution() {
        this.to('swf:TerminateWorkflowExecution');
        return this;
    }
    /**
     * This action removes a tag from an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('swf:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/swf/latest/developerguide/swf-dev-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(domainName, account, partition) {
        var arn = 'arn:${Partition}:swf::${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Constrains the policy statement to only an activity type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeprecateActivityType()
     * - .toDescribeActivityType()
     * - .toRespondActivityTaskCompleted()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeName(value, operator) {
        return this.if(`swf:activityType.name`, value, operator || 'StringLike');
    }
    /**
     * Contstrains the policy statement to only an activity type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeprecateActivityType()
     * - .toDescribeActivityType()
     * - .toRespondActivityTaskCompleted()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeVersion(value, operator) {
        return this.if(`swf:activityType.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching defaultTaskList name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRegisterActivityType()
     * - .toRegisterWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDefaultTaskListName(value, operator) {
        return this.if(`swf:defaultTaskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRegisterActivityType()
     * - .toRegisterWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`swf:name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching tagFilter.tag value.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountClosedWorkflowExecutions()
     * - .toCountOpenWorkflowExecutions()
     * - .toListClosedWorkflowExecutions()
     * - .toListOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagFilterTag(value, operator) {
        return this.if(`swf:tagFilter.tag`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember0(value, operator) {
        return this.if(`swf:tagList.member.0`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember1(value, operator) {
        return this.if(`swf:tagList.member.1`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember2(value, operator) {
        return this.if(`swf:tagList.member.2`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember3(value, operator) {
        return this.if(`swf:tagList.member.3`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember4(value, operator) {
        return this.if(`swf:tagList.member.4`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a tasklist with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountPendingActivityTasks()
     * - .toCountPendingDecisionTasks()
     * - .toPollForActivityTask()
     * - .toPollForDecisionTask()
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskListName(value, operator) {
        return this.if(`swf:taskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountClosedWorkflowExecutions()
     * - .toCountOpenWorkflowExecutions()
     * - .toListClosedWorkflowExecutions()
     * - .toListOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterName(value, operator) {
        return this.if(`swf:typeFilter.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountClosedWorkflowExecutions()
     * - .toCountOpenWorkflowExecutions()
     * - .toListClosedWorkflowExecutions()
     * - .toListOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterVersion(value, operator) {
        return this.if(`swf:typeFilter.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRegisterActivityType()
     * - .toRegisterWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersion(value, operator) {
        return this.if(`swf:version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeprecateWorkflowType()
     * - .toDescribeWorkflowType()
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeName(value, operator) {
        return this.if(`swf:workflowType.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeprecateWorkflowType()
     * - .toDescribeWorkflowType()
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeVersion(value, operator) {
        return this.if(`swf:workflowType.version`, value, operator || 'StringLike');
    }
}
exports.Swf = Swf;
//# sourceMappingURL=data:application/json;base64,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