"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sns = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsns.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sns extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsns.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sns';
        this.accessLevelList = {
            "Permissions management": [
                "AddPermission",
                "RemovePermission"
            ],
            "Read": [
                "CheckIfPhoneNumberIsOptedOut",
                "GetEndpointAttributes",
                "GetPlatformApplicationAttributes",
                "GetSMSAttributes",
                "GetSubscriptionAttributes",
                "GetTopicAttributes",
                "ListPhoneNumbersOptedOut",
                "ListTagsForResource"
            ],
            "Write": [
                "ConfirmSubscription",
                "CreatePlatformApplication",
                "CreatePlatformEndpoint",
                "CreateTopic",
                "DeleteEndpoint",
                "DeletePlatformApplication",
                "DeleteTopic",
                "OptInPhoneNumber",
                "Publish",
                "SetEndpointAttributes",
                "SetPlatformApplicationAttributes",
                "SetSMSAttributes",
                "SetSubscriptionAttributes",
                "SetTopicAttributes",
                "Subscribe",
                "Unsubscribe"
            ],
            "List": [
                "ListEndpointsByPlatformApplication",
                "ListPlatformApplications",
                "ListSubscriptions",
                "ListSubscriptionsByTopic",
                "ListTopics"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html
     */
    toAddPermission() {
        this.to('sns:AddPermission');
        return this;
    }
    /**
     * Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CheckIfPhoneNumberIsOptedOut.html
     */
    toCheckIfPhoneNumberIsOptedOut() {
        this.to('sns:CheckIfPhoneNumberIsOptedOut');
        return this;
    }
    /**
     * Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html
     */
    toConfirmSubscription() {
        this.to('sns:ConfirmSubscription');
        return this;
    }
    /**
     * Creates a platform application object for one of the supported push notification services, such as APNS and GCM, to which devices and mobile apps may register.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformApplication.html
     */
    toCreatePlatformApplication() {
        this.to('sns:CreatePlatformApplication');
        return this;
    }
    /**
     * Creates an endpoint for a device and mobile app on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformEndpoint.html
     */
    toCreatePlatformEndpoint() {
        this.to('sns:CreatePlatformEndpoint');
        return this;
    }
    /**
     * Creates a topic to which notifications can be published.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html
     */
    toCreateTopic() {
        this.to('sns:CreateTopic');
        return this;
    }
    /**
     * Deletes the endpoint for a device and mobile app from Amazon SNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.to('sns:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes a platform application object for one of the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeletePlatformApplication.html
     */
    toDeletePlatformApplication() {
        this.to('sns:DeletePlatformApplication');
        return this;
    }
    /**
     * Deletes a topic and all its subscriptions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html
     */
    toDeleteTopic() {
        this.to('sns:DeleteTopic');
        return this;
    }
    /**
     * Retrieves the endpoint attributes for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetEndpointAttributes.html
     */
    toGetEndpointAttributes() {
        this.to('sns:GetEndpointAttributes');
        return this;
    }
    /**
     * Retrieves the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetPlatformApplicationAttributes.html
     */
    toGetPlatformApplicationAttributes() {
        this.to('sns:GetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Returns the settings for sending SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSMSAttributes.html
     */
    toGetSMSAttributes() {
        this.to('sns:GetSMSAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a subscription.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html
     */
    toGetSubscriptionAttributes() {
        this.to('sns:GetSubscriptionAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of the user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html
     */
    toGetTopicAttributes() {
        this.to('sns:GetTopicAttributes');
        return this;
    }
    /**
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM and APNS.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListEndpointsByPlatformApplication.html
     */
    toListEndpointsByPlatformApplication() {
        this.to('sns:ListEndpointsByPlatformApplication');
        return this;
    }
    /**
     * Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPhoneNumbersOptedOut.html
     */
    toListPhoneNumbersOptedOut() {
        this.to('sns:ListPhoneNumbersOptedOut');
        return this;
    }
    /**
     * Lists the platform application objects for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPlatformApplications.html
     */
    toListPlatformApplications() {
        this.to('sns:ListPlatformApplications');
        return this;
    }
    /**
     * Returns a list of the requester's subscriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html
     */
    toListSubscriptions() {
        this.to('sns:ListSubscriptions');
        return this;
    }
    /**
     * Returns a list of the subscriptions to a specific topic.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html
     */
    toListSubscriptionsByTopic() {
        this.to('sns:ListSubscriptionsByTopic');
        return this;
    }
    /**
     * List all tags added to the specified Amazon SNS topic.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('sns:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html
     */
    toListTopics() {
        this.to('sns:ListTopics');
        return this;
    }
    /**
     * Opts in a phone number that is currently opted out, which enables you to resume sending SMS messages to the number.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_OptInPhoneNumber.html
     */
    toOptInPhoneNumber() {
        this.to('sns:OptInPhoneNumber');
        return this;
    }
    /**
     * Sends a message to all of a topic's subscribed endpoints.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Publish.html
     */
    toPublish() {
        this.to('sns:Publish');
        return this;
    }
    /**
     * Removes a statement from a topic's access control policy.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html
     */
    toRemovePermission() {
        this.to('sns:RemovePermission');
        return this;
    }
    /**
     * Sets the attributes for an endpoint for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html
     */
    toSetEndpointAttributes() {
        this.to('sns:SetEndpointAttributes');
        return this;
    }
    /**
     * Sets the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetPlatformApplicationAttributes.html
     */
    toSetPlatformApplicationAttributes() {
        this.to('sns:SetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Set the default settings for sending SMS messages and receiving daily SMS usage reports.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSMSAttributes.html
     */
    toSetSMSAttributes() {
        this.to('sns:SetSMSAttributes');
        return this;
    }
    /**
     * Allows a subscription owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html
     */
    toSetSubscriptionAttributes() {
        this.to('sns:SetSubscriptionAttributes');
        return this;
    }
    /**
     * Allows a topic owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html
     */
    toSetTopicAttributes() {
        this.to('sns:SetTopicAttributes');
        return this;
    }
    /**
     * Prepares to subscribe an endpoint by sending the endpoint a confirmation message.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpoint()
     * - .ifProtocol()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html
     */
    toSubscribe() {
        this.to('sns:Subscribe');
        return this;
    }
    /**
     * Add tags to the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('sns:TagResource');
        return this;
    }
    /**
     * Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the subscription or the topic's owner can unsubscribe, and an AWS signature is required.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html
     */
    toUnsubscribe() {
        this.to('sns:Unsubscribe');
        return this;
    }
    /**
     * Remove tags from the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('sns:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopic(topicName, account, region, partition) {
        var arn = 'arn:${Partition}:sns:${Region}:${Account}:${TopicName}';
        arn = arn.replace('${TopicName}', topicName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The URL, email address, or ARN from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpoint(value, operator) {
        return this.if(`sns:Endpoint`, value, operator || 'StringLike');
    }
    /**
     * The protocol value from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProtocol(value, operator) {
        return this.if(`sns:Protocol`, value, operator || 'StringLike');
    }
}
exports.Sns = Sns;
//# sourceMappingURL=data:application/json;base64,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