"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sqs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sqs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsqs.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sqs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sqs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsqs.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sqs';
        this.accessLevelList = {
            "Permissions management": [
                "AddPermission",
                "RemovePermission"
            ],
            "Write": [
                "ChangeMessageVisibility",
                "ChangeMessageVisibilityBatch",
                "CreateQueue",
                "DeleteMessage",
                "DeleteMessageBatch",
                "DeleteQueue",
                "PurgeQueue",
                "SendMessage",
                "SendMessageBatch",
                "SetQueueAttributes"
            ],
            "Read": [
                "GetQueueAttributes",
                "GetQueueUrl",
                "ListDeadLetterSourceQueues",
                "ListQueueTags",
                "ReceiveMessage"
            ],
            "List": [
                "ListQueues"
            ],
            "Tagging": [
                "TagQueue",
                "UntagQueue"
            ]
        };
    }
    /**
     * Adds a permission to a queue for a specific principal.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_AddPermission.html
     */
    toAddPermission() {
        this.to('sqs:AddPermission');
        return this;
    }
    /**
     * Changes the visibility timeout of a specified message in a queue to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibility.html
     */
    toChangeMessageVisibility() {
        this.to('sqs:ChangeMessageVisibility');
        return this;
    }
    /**
     * Changes the visibility timeout of multiple messages.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibilityBatch.html
     */
    toChangeMessageVisibilityBatch() {
        this.to('sqs:ChangeMessageVisibilityBatch');
        return this;
    }
    /**
     * Creates a new queue, or returns the URL of an existing one.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_CreateQueue.html
     */
    toCreateQueue() {
        this.to('sqs:CreateQueue');
        return this;
    }
    /**
     * Deletes the specified message from the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessage.html
     */
    toDeleteMessage() {
        this.to('sqs:DeleteMessage');
        return this;
    }
    /**
     * Deletes up to ten messages from the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessageBatch.html
     */
    toDeleteMessageBatch() {
        this.to('sqs:DeleteMessageBatch');
        return this;
    }
    /**
     * Deletes the queue specified by the queue URL, regardless of whether the queue is empty.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteQueue.html
     */
    toDeleteQueue() {
        this.to('sqs:DeleteQueue');
        return this;
    }
    /**
     * Gets attributes for the specified queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_GetQueueAttributes.html
     */
    toGetQueueAttributes() {
        this.to('sqs:GetQueueAttributes');
        return this;
    }
    /**
     * Returns the URL of an existing queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_GetQueueUrl.html
     */
    toGetQueueUrl() {
        this.to('sqs:GetQueueUrl');
        return this;
    }
    /**
     * Returns a list of your queues that have the RedrivePolicy queue attribute configured with a dead letter queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListDeadLetterSourceQueues.html
     */
    toListDeadLetterSourceQueues() {
        this.to('sqs:ListDeadLetterSourceQueues');
        return this;
    }
    /**
     * Lists tags added to an SQS queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListQueueTags.html
     */
    toListQueueTags() {
        this.to('sqs:ListQueueTags');
        return this;
    }
    /**
     * Returns a list of your queues.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListQueues.html
     */
    toListQueues() {
        this.to('sqs:ListQueues');
        return this;
    }
    /**
     * Deletes the messages in a queue specified by the queue URL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_PurgeQueue.html
     */
    toPurgeQueue() {
        this.to('sqs:PurgeQueue');
        return this;
    }
    /**
     * Retrieves one or more messages, with a maximum limit of 10 messages, from the specified queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ReceiveMessage.html
     */
    toReceiveMessage() {
        this.to('sqs:ReceiveMessage');
        return this;
    }
    /**
     * Revokes any permissions in the queue policy that matches the specified Label parameter.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_RemovePermission.html
     */
    toRemovePermission() {
        this.to('sqs:RemovePermission');
        return this;
    }
    /**
     * Delivers a message to the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html
     */
    toSendMessage() {
        this.to('sqs:SendMessage');
        return this;
    }
    /**
     * Delivers up to ten messages to the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessageBatch.html
     */
    toSendMessageBatch() {
        this.to('sqs:SendMessageBatch');
        return this;
    }
    /**
     * Sets the value of one or more queue attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SetQueueAttributes.html
     */
    toSetQueueAttributes() {
        this.to('sqs:SetQueueAttributes');
        return this;
    }
    /**
     * Add tags to the specified SQS queue.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_TagQueue.html
     */
    toTagQueue() {
        this.to('sqs:TagQueue');
        return this;
    }
    /**
     * Remove tags from the specified SQS queue.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_UntagQueue.html
     */
    toUntagQueue() {
        this.to('sqs:UntagQueue');
        return this;
    }
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-how-it-works.html
     *
     * @param queueName - Identifier for the queueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQueue(queueName, account, region, partition) {
        var arn = 'arn:${Partition}:sqs:${Region}:${Account}:${QueueName}';
        arn = arn.replace('${QueueName}', queueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Sqs = Sqs;
//# sourceMappingURL=data:application/json;base64,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