"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sumerian = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sumerian](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsumerian.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sumerian extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sumerian](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsumerian.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sumerian';
        this.accessLevelList = {
            "Write": [
                "Login"
            ],
            "Read": [
                "ViewRelease"
            ]
        };
    }
    /**
     * Grant login access to the Sumerian console.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sumerian/latest/userguide/sumerian-permissions.html
     */
    toLogin() {
        this.to('sumerian:Login');
        return this;
    }
    /**
     * Grant access to view a project release.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sumerian/latest/userguide/sumerian-permissions.html
     */
    toViewRelease() {
        this.to('sumerian:ViewRelease');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:sumerian:${Region}:${Account}:project:${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Sumerian = Sumerian;
//# sourceMappingURL=data:application/json;base64,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