"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspaces = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [workspaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workspaces extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [workspaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspaces.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'workspaces';
        this.accessLevelList = {
            "Write": [
                "AssociateIpGroups",
                "AuthorizeIpRules",
                "CreateIpGroup",
                "CreateWorkspaces",
                "DeleteIpGroup",
                "DeleteTags",
                "DeleteWorkspaceImage",
                "DisassociateIpGroups",
                "ImportWorkspaceImage",
                "ModifyAccount",
                "ModifyClientProperties",
                "ModifyWorkspaceProperties",
                "ModifyWorkspaceState",
                "RebootWorkspaces",
                "RebuildWorkspaces",
                "RevokeIpRules",
                "StartWorkspaces",
                "StopWorkspaces",
                "TerminateWorkspaces",
                "UpdateRulesOfIpGroup"
            ],
            "Tagging": [
                "CreateTags"
            ],
            "List": [
                "DescribeAccount",
                "DescribeAccountModifications",
                "DescribeClientProperties",
                "DescribeIpGroups",
                "DescribeTags",
                "DescribeWorkspaceBundles",
                "DescribeWorkspaceDirectories",
                "DescribeWorkspaceImages",
                "DescribeWorkspaces",
                "ListAvailableManagementCidrRanges"
            ],
            "Read": [
                "DescribeWorkspacesConnectionStatus"
            ]
        };
    }
    /**
     * Associates the specified IP access control group with the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateIpGroups.html
     */
    toAssociateIpGroups() {
        this.to('workspaces:AssociateIpGroups');
        return this;
    }
    /**
     * Adds one or more rules to the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AuthorizeIpRules.html
     */
    toAuthorizeIpRules() {
        this.to('workspaces:AuthorizeIpRules');
        return this;
    }
    /**
     * Creates an IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateIpGroup.html
     */
    toCreateIpGroup() {
        this.to('workspaces:CreateIpGroup');
        return this;
    }
    /**
     * Creates tags for a WorkSpace.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateTags.html
     */
    toCreateTags() {
        this.to('workspaces:CreateTags');
        return this;
    }
    /**
     * Creates one or more WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaces.html
     */
    toCreateWorkspaces() {
        this.to('workspaces:CreateWorkspaces');
        return this;
    }
    /**
     * Deletes the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteIpGroup.html
     */
    toDeleteIpGroup() {
        this.to('workspaces:DeleteIpGroup');
        return this;
    }
    /**
     * Deletes tags from a Workspace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteTags.html
     */
    toDeleteTags() {
        this.to('workspaces:DeleteTags');
        return this;
    }
    /**
     * Deletes the specified workspace image.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceImage.html
     */
    toDeleteWorkspaceImage() {
        this.to('workspaces:DeleteWorkspaceImage');
        return this;
    }
    /**
     * Retrieves a list that describes the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccount.html
     */
    toDescribeAccount() {
        this.to('workspaces:DescribeAccount');
        return this;
    }
    /**
     * Retrieves a list that describes modifications to the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccountModifications.html
     */
    toDescribeAccountModifications() {
        this.to('workspaces:DescribeAccountModifications');
        return this;
    }
    /**
     * Describe client properties about the specified resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientProperties.html
     */
    toDescribeClientProperties() {
        this.to('workspaces:DescribeClientProperties');
        return this;
    }
    /**
     * Retrieves information about the IP access control groups of your account in the region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeIpGroups.html
     */
    toDescribeIpGroups() {
        this.to('workspaces:DescribeIpGroups');
        return this;
    }
    /**
     * Describes tags for a WorkSpace.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeTags.html
     */
    toDescribeTags() {
        this.to('workspaces:DescribeTags');
        return this;
    }
    /**
     * Obtains information about the WorkSpace bundles that are available to your account in the specified region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceBundles.html
     */
    toDescribeWorkspaceBundles() {
        this.to('workspaces:DescribeWorkspaceBundles');
        return this;
    }
    /**
     * Retrieves information about the AWS Directory Service directories in the region that are registered with Amazon WorkSpaces and are available to your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceDirectories.html
     */
    toDescribeWorkspaceDirectories() {
        this.to('workspaces:DescribeWorkspaceDirectories');
        return this;
    }
    /**
     * Retrieves a list that describes one or more specified images.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImages.html
     */
    toDescribeWorkspaceImages() {
        this.to('workspaces:DescribeWorkspaceImages');
        return this;
    }
    /**
     * Obtains information about the specified WorkSpaces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html
     */
    toDescribeWorkspaces() {
        this.to('workspaces:DescribeWorkspaces');
        return this;
    }
    /**
     * Describes the connection status of a specified WorkSpace.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesConnectionStatus.html
     */
    toDescribeWorkspacesConnectionStatus() {
        this.to('workspaces:DescribeWorkspacesConnectionStatus');
        return this;
    }
    /**
     * Disassociates the specified IP access control group from the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateIpGroups.html
     */
    toDisassociateIpGroups() {
        this.to('workspaces:DisassociateIpGroups');
        return this;
    }
    /**
     * Import a licensed EC2 image to into Amazon WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportWorkspaceImage.html
     */
    toImportWorkspaceImage() {
        this.to('workspaces:ImportWorkspaceImage');
        return this;
    }
    /**
     * List available CIDR ranges for a CIDR range constraint.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListAvailableManagementCidrRanges.html
     */
    toListAvailableManagementCidrRanges() {
        this.to('workspaces:ListAvailableManagementCidrRanges');
        return this;
    }
    /**
     * Modify the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyAccount.html
     */
    toModifyAccount() {
        this.to('workspaces:ModifyAccount');
        return this;
    }
    /**
     * Modify the client properties of a specified resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyClientProperties.html
     */
    toModifyClientProperties() {
        this.to('workspaces:ModifyClientProperties');
        return this;
    }
    /**
     * Modifies the WorkSpace properties, including the running mode and AutoStop time.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceProperties.html
     */
    toModifyWorkspaceProperties() {
        this.to('workspaces:ModifyWorkspaceProperties');
        return this;
    }
    /**
     * Modify the state of specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceState.html
     */
    toModifyWorkspaceState() {
        this.to('workspaces:ModifyWorkspaceState');
        return this;
    }
    /**
     * Reboots the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebootWorkspaces.html
     */
    toRebootWorkspaces() {
        this.to('workspaces:RebootWorkspaces');
        return this;
    }
    /**
     * Rebuilds the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebuildWorkspaces.html
     */
    toRebuildWorkspaces() {
        this.to('workspaces:RebuildWorkspaces');
        return this;
    }
    /**
     * Removes one or more rules from the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RevokeIpRules.html
     */
    toRevokeIpRules() {
        this.to('workspaces:RevokeIpRules');
        return this;
    }
    /**
     * Starts the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspaces.html
     */
    toStartWorkspaces() {
        this.to('workspaces:StartWorkspaces');
        return this;
    }
    /**
     * Stops the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspaces.html
     */
    toStopWorkspaces() {
        this.to('workspaces:StopWorkspaces');
        return this;
    }
    /**
     * Terminates the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspaces.html
     */
    toTerminateWorkspaces() {
        this.to('workspaces:TerminateWorkspaces');
        return this;
    }
    /**
     * Replaces the current rules of the specified IP access control group with the specified rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateRulesOfIpGroup.html
     */
    toUpdateRulesOfIpGroup() {
        this.to('workspaces:UpdateRulesOfIpGroup');
        return this;
    }
    /**
     * Adds a resource of type workspacebundle to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html
     *
     * @param bundleId - Identifier for the bundleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspacebundle(bundleId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspacebundle/${BundleId}';
        arn = arn.replace('${BundleId}', bundleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workspaceipgroup to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-ip-access-control-groups.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspaceipgroup(groupId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspaceipgroup/${GroupId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type directoryid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/manage-workspaces-directory.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDirectoryid(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workspaceid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp_workspace_management.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspaceid(workspaceId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspace/${WorkspaceId}';
        arn = arn.replace('${WorkspaceId}', workspaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Workspaces = Workspaces;
//# sourceMappingURL=data:application/json;base64,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