"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [discovery](https://docs.aws.amazon.com/service-authorization/latest/reference/list_applicationdiscovery.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Discovery extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [discovery](https://docs.aws.amazon.com/service-authorization/latest/reference/list_applicationdiscovery.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'discovery';
        this.accessLevelList = {
            "Write": [
                "AssociateConfigurationItemsToApplication",
                "BatchDeleteImportData",
                "CreateApplication",
                "DeleteApplications",
                "DisassociateConfigurationItemsFromApplication",
                "ExportConfigurations",
                "StartContinuousExport",
                "StartDataCollectionByAgentIds",
                "StartExportTask",
                "StartImportTask",
                "StopContinuousExport",
                "StopDataCollectionByAgentIds",
                "UpdateApplication"
            ],
            "Tagging": [
                "CreateTags",
                "DeleteTags"
            ],
            "Read": [
                "DescribeAgents",
                "DescribeConfigurations",
                "DescribeContinuousExports",
                "DescribeExportConfigurations",
                "DescribeExportTasks",
                "DescribeTags",
                "GetDiscoverySummary"
            ],
            "List": [
                "DescribeImportTasks",
                "ListConfigurations",
                "ListServerNeighbors"
            ]
        };
    }
    /**
     * Associates one or more configuration items with an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_AssociateConfigurationItemsToApplication.html
     */
    toAssociateConfigurationItemsToApplication() {
        this.to('discovery:AssociateConfigurationItemsToApplication');
        return this;
    }
    /**
     * Deletes one or more Migration Hub import tasks, each identified by their import ID. Each import task has a number of records, which can identify servers or applications.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_BatchDeleteImportData.html
     */
    toBatchDeleteImportData() {
        this.to('discovery:BatchDeleteImportData');
        return this;
    }
    /**
     * Creates an application with the given name and description.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        this.to('discovery:CreateApplication');
        return this;
    }
    /**
     * Creates one or more tags for configuration items. Tags are metadata that help you categorize IT assets. This API accepts a list of multiple configuration items.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_CreateTags.html
     */
    toCreateTags() {
        this.to('discovery:CreateTags');
        return this;
    }
    /**
     * Deletes a list of applications and their associations with configuration items.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DeleteApplications.html
     */
    toDeleteApplications() {
        this.to('discovery:DeleteApplications');
        return this;
    }
    /**
     * Deletes the association between configuration items and one or more tags. This API accepts a list of multiple configuration items.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        this.to('discovery:DeleteTags');
        return this;
    }
    /**
     * Lists agents or the Connector by ID or lists all agents/Connectors associated with your user account if you did not specify an ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeAgents.html
     */
    toDescribeAgents() {
        this.to('discovery:DescribeAgents');
        return this;
    }
    /**
     * Retrieves attributes for a list of configuration item IDs. All of the supplied IDs must be for the same asset type (server, application, process, or connection). Output fields are specific to the asset type selected. For example, the output for a server configuration item includes a list of attributes about the server, such as host name, operating system, and number of network cards.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeConfigurations.html
     */
    toDescribeConfigurations() {
        this.to('discovery:DescribeConfigurations');
        return this;
    }
    /**
     * Lists exports as specified by ID. All continuous exports associated with your user account can be listed if you call DescribeContinuousExports as is without passing any parameters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeContinuousExports.html
     */
    toDescribeContinuousExports() {
        this.to('discovery:DescribeContinuousExports');
        return this;
    }
    /**
     * Retrieves the status of a given export process. You can retrieve status from a maximum of 100 processes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeExportConfigurations.html
     */
    toDescribeExportConfigurations() {
        this.to('discovery:DescribeExportConfigurations');
        return this;
    }
    /**
     * Retrieve status of one or more export tasks. You can retrieve the status of up to 100 export tasks.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        this.to('discovery:DescribeExportTasks');
        return this;
    }
    /**
     * Returns an array of import tasks for your account, including status information, times, IDs, the Amazon S3 Object URL for the import file, and more.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeImportTasks.html
     */
    toDescribeImportTasks() {
        this.to('discovery:DescribeImportTasks');
        return this;
    }
    /**
     * Retrieves a list of configuration items that are tagged with a specific tag. Or retrieves a list of all tags assigned to a specific configuration item.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.to('discovery:DescribeTags');
        return this;
    }
    /**
     * Disassociates one or more configuration items from an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DisassociateConfigurationItemsFromApplication.html
     */
    toDisassociateConfigurationItemsFromApplication() {
        this.to('discovery:DisassociateConfigurationItemsFromApplication');
        return this;
    }
    /**
     * Exports all discovered configuration data to an Amazon S3 bucket or an application that enables you to view and evaluate the data. Data includes tags and tag associations, processes, connections, servers, and system performance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_ExportConfigurations.html
     */
    toExportConfigurations() {
        this.to('discovery:ExportConfigurations');
        return this;
    }
    /**
     * Retrieves a short summary of discovered assets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_GetDiscoverySummary.html
     */
    toGetDiscoverySummary() {
        this.to('discovery:GetDiscoverySummary');
        return this;
    }
    /**
     * Retrieves a list of configuration items according to criteria you specify in a filter. The filter criteria identify relationship requirements.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_ListConfigurations.html
     */
    toListConfigurations() {
        this.to('discovery:ListConfigurations');
        return this;
    }
    /**
     * Retrieves a list of servers which are one network hop away from a specified server.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_ListServerNeighbors.html
     */
    toListServerNeighbors() {
        this.to('discovery:ListServerNeighbors');
        return this;
    }
    /**
     * Start the continuous flow of agent's discovered data into Amazon Athena.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartContinuousExport.html
     */
    toStartContinuousExport() {
        this.to('discovery:StartContinuousExport');
        return this;
    }
    /**
     * Instructs the specified agents or Connectors to start collecting data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartDataCollectionByAgentIds.html
     */
    toStartDataCollectionByAgentIds() {
        this.to('discovery:StartDataCollectionByAgentIds');
        return this;
    }
    /**
     * Export the configuration data about discovered configuration items and relationships to an S3 bucket in a specified format.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartExportTask.html
     */
    toStartExportTask() {
        this.to('discovery:StartExportTask');
        return this;
    }
    /**
     * Starts an import task. The Migration Hub import feature allows you to import details of your on-premises environment directly into AWS without having to use the Application Discovery Service (ADS) tools such as the Discovery Connector or Discovery Agent. This gives you the option to perform migration assessment and planning directly from your imported data including the ability to group your devices as applications and track their migration status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartImportTask.html
     */
    toStartImportTask() {
        this.to('discovery:StartImportTask');
        return this;
    }
    /**
     * Stop the continuous flow of agent's discovered data into Amazon Athena.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StopContinuousExport.html
     */
    toStopContinuousExport() {
        this.to('discovery:StopContinuousExport');
        return this;
    }
    /**
     * Instructs the specified agents or Connectors to stop collecting data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StopDataCollectionByAgentIds.html
     */
    toStopDataCollectionByAgentIds() {
        this.to('discovery:StopDataCollectionByAgentIds');
        return this;
    }
    /**
     * Updates metadata about an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        this.to('discovery:UpdateApplication');
        return this;
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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