"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Amplify = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [amplify](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsamplify.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Amplify extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [amplify](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsamplify.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'amplify';
        this.accessLevelList = {
            "Write": [
                "CreateApp",
                "CreateBackendEnvironment",
                "CreateBranch",
                "CreateDeployment",
                "CreateDomainAssociation",
                "CreateWebHook",
                "DeleteApp",
                "DeleteBackendEnvironment",
                "DeleteBranch",
                "DeleteDomainAssociation",
                "DeleteJob",
                "DeleteWebHook",
                "GenerateAccessLogs",
                "StartDeployment",
                "StartJob",
                "StopJob",
                "UpdateApp",
                "UpdateBranch",
                "UpdateDomainAssociation",
                "UpdateWebHook"
            ],
            "Read": [
                "GetApp",
                "GetArtifactUrl",
                "GetBackendEnvironment",
                "GetBranch",
                "GetDomainAssociation",
                "GetJob",
                "GetWebHook"
            ],
            "List": [
                "ListApps",
                "ListArtifacts",
                "ListBackendEnvironments",
                "ListBranches",
                "ListDomainAssociations",
                "ListJobs",
                "ListWebHooks"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates a new Amplify App.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toCreateApp() {
        this.to('amplify:CreateApp');
        return this;
    }
    /**
     * Creates a new backend environment for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toCreateBackendEnvironment() {
        this.to('amplify:CreateBackendEnvironment');
        return this;
    }
    /**
     * Creates a new Branch for an Amplify App.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toCreateBranch() {
        this.to('amplify:CreateBranch');
        return this;
    }
    /**
     * Create a deployment for manual deploy apps. (Apps are not connected to repository)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toCreateDeployment() {
        this.to('amplify:CreateDeployment');
        return this;
    }
    /**
     * Create a new DomainAssociation on an App
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toCreateDomainAssociation() {
        this.to('amplify:CreateDomainAssociation');
        return this;
    }
    /**
     * Create a new webhook on an App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toCreateWebHook() {
        this.to('amplify:CreateWebHook');
        return this;
    }
    /**
     * Delete an existing Amplify App by appId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toDeleteApp() {
        this.to('amplify:DeleteApp');
        return this;
    }
    /**
     * Deletes a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toDeleteBackendEnvironment() {
        this.to('amplify:DeleteBackendEnvironment');
        return this;
    }
    /**
     * Deletes a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toDeleteBranch() {
        this.to('amplify:DeleteBranch');
        return this;
    }
    /**
     * Deletes a DomainAssociation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toDeleteDomainAssociation() {
        this.to('amplify:DeleteDomainAssociation');
        return this;
    }
    /**
     * Delete a job, for an Amplify branch, part of Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toDeleteJob() {
        this.to('amplify:DeleteJob');
        return this;
    }
    /**
     * Delete a webhook by id.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toDeleteWebHook() {
        this.to('amplify:DeleteWebHook');
        return this;
    }
    /**
     * Generate website access logs for a specific time range via a pre-signed URL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGenerateAccessLogs() {
        this.to('amplify:GenerateAccessLogs');
        return this;
    }
    /**
     * Retrieves an existing Amplify App by appId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGetApp() {
        this.to('amplify:GetApp');
        return this;
    }
    /**
     * Retrieves artifact info that corresponds to a artifactId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGetArtifactUrl() {
        this.to('amplify:GetArtifactUrl');
        return this;
    }
    /**
     * Retrieves a backend environment for an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGetBackendEnvironment() {
        this.to('amplify:GetBackendEnvironment');
        return this;
    }
    /**
     * Retrieves a branch for an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGetBranch() {
        this.to('amplify:GetBranch');
        return this;
    }
    /**
     * Retrieves domain info that corresponds to an appId and domainName.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGetDomainAssociation() {
        this.to('amplify:GetDomainAssociation');
        return this;
    }
    /**
     * Get a job for a branch, part of an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGetJob() {
        this.to('amplify:GetJob');
        return this;
    }
    /**
     * Retrieves webhook info that corresponds to a webhookId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toGetWebHook() {
        this.to('amplify:GetWebHook');
        return this;
    }
    /**
     * Lists existing Amplify Apps.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toListApps() {
        this.to('amplify:ListApps');
        return this;
    }
    /**
     * List artifacts with an app, a branch, a job and an artifact type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toListArtifacts() {
        this.to('amplify:ListArtifacts');
        return this;
    }
    /**
     * Lists backend environments for an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toListBackendEnvironments() {
        this.to('amplify:ListBackendEnvironments');
        return this;
    }
    /**
     * Lists branches for an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toListBranches() {
        this.to('amplify:ListBranches');
        return this;
    }
    /**
     * List domains with an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toListDomainAssociations() {
        this.to('amplify:ListDomainAssociations');
        return this;
    }
    /**
     * List Jobs for a branch, part of an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toListJobs() {
        this.to('amplify:ListJobs');
        return this;
    }
    /**
     * List webhooks on an App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toListWebHooks() {
        this.to('amplify:ListWebHooks');
        return this;
    }
    /**
     * Start a deployment for manual deploy apps. (Apps are not connected to repository)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toStartDeployment() {
        this.to('amplify:StartDeployment');
        return this;
    }
    /**
     * Starts a new job for a branch, part of an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toStartJob() {
        this.to('amplify:StartJob');
        return this;
    }
    /**
     * Stop a job that is in progress, for an Amplify branch, part of Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toStopJob() {
        this.to('amplify:StopJob');
        return this;
    }
    /**
     * This action tags an AWS Amplify Console resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toTagResource() {
        this.to('amplify:TagResource');
        return this;
    }
    /**
     * This action removes a tag from an AWS Amplify Console resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toUntagResource() {
        this.to('amplify:UntagResource');
        return this;
    }
    /**
     * Updates an existing Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toUpdateApp() {
        this.to('amplify:UpdateApp');
        return this;
    }
    /**
     * Updates a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toUpdateBranch() {
        this.to('amplify:UpdateBranch');
        return this;
    }
    /**
     * Update a DomainAssociation on an App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toUpdateDomainAssociation() {
        this.to('amplify:UpdateDomainAssociation');
        return this;
    }
    /**
     * Update a webhook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    toUpdateWebHook() {
        this.to('amplify:UpdateWebHook');
        return this;
    }
    /**
     * Adds a resource of type apps to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApps(appId, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type branches to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param branchName - Identifier for the branchName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBranches(appId, branchName, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${BranchName}', branchName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type jobs to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param branchName - Identifier for the branchName.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJobs(appId, branchName, jobId, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}/jobs/${JobId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${BranchName}', branchName);
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type domains to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomains(appId, domainName, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/domains/${DomainName}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Amplify = Amplify;
//# sourceMappingURL=data:application/json;base64,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