"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appsync = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appsync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappsync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appsync extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appsync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappsync.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appsync';
        this.accessLevelList = {
            "Write": [
                "CreateApiKey",
                "CreateDataSource",
                "CreateFunction",
                "CreateResolver",
                "CreateType",
                "DeleteApiKey",
                "DeleteDataSource",
                "DeleteFunction",
                "DeleteGraphqlApi",
                "DeleteResolver",
                "DeleteType",
                "GraphQL",
                "SetWebACL",
                "StartSchemaCreation",
                "UpdateApiKey",
                "UpdateDataSource",
                "UpdateFunction",
                "UpdateGraphqlApi",
                "UpdateResolver",
                "UpdateType"
            ],
            "Tagging": [
                "CreateGraphqlApi",
                "TagResource",
                "UntagResource"
            ],
            "Read": [
                "GetDataSource",
                "GetFunction",
                "GetGraphqlApi",
                "GetIntrospectionSchema",
                "GetResolver",
                "GetSchemaCreationStatus",
                "GetType",
                "ListTagsForResource"
            ],
            "List": [
                "ListApiKeys",
                "ListDataSources",
                "ListFunctions",
                "ListGraphqlApis",
                "ListResolvers",
                "ListResolversByFunction",
                "ListTypes"
            ]
        };
    }
    /**
     * Creates a unique key that you can distribute to clients who are executing your API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiKey.html
     */
    toCreateApiKey() {
        this.to('appsync:CreateApiKey');
        return this;
    }
    /**
     * Creates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDataSource.html
     */
    toCreateDataSource() {
        this.to('appsync:CreateDataSource');
        return this;
    }
    /**
     * Create a new Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateFunction.html
     */
    toCreateFunction() {
        this.to('appsync:CreateFunction');
        return this;
    }
    /**
     * Creates a GraphqlApi object, which is the top level AppSync resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateGraphqlApi.html
     */
    toCreateGraphqlApi() {
        this.to('appsync:CreateGraphqlApi');
        return this;
    }
    /**
     * Creates a Resolver object. A resolver converts incoming requests into a format that a data source can understand, and converts the data source's responses into GraphQL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateResolver.html
     */
    toCreateResolver() {
        this.to('appsync:CreateResolver');
        return this;
    }
    /**
     * Creates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateType.html
     */
    toCreateType() {
        this.to('appsync:CreateType');
        return this;
    }
    /**
     * Deletes an API key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiKey.html
     */
    toDeleteApiKey() {
        this.to('appsync:DeleteApiKey');
        return this;
    }
    /**
     * Deletes a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        this.to('appsync:DeleteDataSource');
        return this;
    }
    /**
     * Deletes a Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteFunction.html
     */
    toDeleteFunction() {
        this.to('appsync:DeleteFunction');
        return this;
    }
    /**
     * Deletes a GraphqlApi object. This will also clean up every AppSync resource below that API.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteGraphqlApi.html
     */
    toDeleteGraphqlApi() {
        this.to('appsync:DeleteGraphqlApi');
        return this;
    }
    /**
     * Deletes a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteResolver.html
     */
    toDeleteResolver() {
        this.to('appsync:DeleteResolver');
        return this;
    }
    /**
     * Deletes a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteType.html
     */
    toDeleteType() {
        this.to('appsync:DeleteType');
        return this;
    }
    /**
     * Retrieves a DataSource object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSource.html
     */
    toGetDataSource() {
        this.to('appsync:GetDataSource');
        return this;
    }
    /**
     * Retrieves a Function object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetFunction.html
     */
    toGetFunction() {
        this.to('appsync:GetFunction');
        return this;
    }
    /**
     * Retrieves a GraphqlApi object.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApi.html
     */
    toGetGraphqlApi() {
        this.to('appsync:GetGraphqlApi');
        return this;
    }
    /**
     * Retrieves the introspection schema for a GraphQL API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetIntrospectionSchema.html
     */
    toGetIntrospectionSchema() {
        this.to('appsync:GetIntrospectionSchema');
        return this;
    }
    /**
     * Retrieves a Resolver object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetResolver.html
     */
    toGetResolver() {
        this.to('appsync:GetResolver');
        return this;
    }
    /**
     * Retrieves the current status of a schema creation operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSchemaCreationStatus.html
     */
    toGetSchemaCreationStatus() {
        this.to('appsync:GetSchemaCreationStatus');
        return this;
    }
    /**
     * Retrieves a Type object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetType.html
     */
    toGetType() {
        this.to('appsync:GetType');
        return this;
    }
    /**
     * Sends a GraphQL query to a GraphQL API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-api.html
     */
    toGraphQL() {
        this.to('appsync:GraphQL');
        return this;
    }
    /**
     * Lists the API keys for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApiKeys.html
     */
    toListApiKeys() {
        this.to('appsync:ListApiKeys');
        return this;
    }
    /**
     * Lists the data sources for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources() {
        this.to('appsync:ListDataSources');
        return this;
    }
    /**
     * Lists the functions for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListFunctions.html
     */
    toListFunctions() {
        this.to('appsync:ListFunctions');
        return this;
    }
    /**
     * Lists your GraphQL APIs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListGraphqlApis.html
     */
    toListGraphqlApis() {
        this.to('appsync:ListGraphqlApis');
        return this;
    }
    /**
     * Lists the resolvers for a given API and type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolvers.html
     */
    toListResolvers() {
        this.to('appsync:ListResolvers');
        return this;
    }
    /**
     * List the resolvers that are associated with a specific function.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolversByFunction.html
     */
    toListResolversByFunction() {
        this.to('appsync:ListResolversByFunction');
        return this;
    }
    /**
     * List the tags for a resource.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('appsync:ListTagsForResource');
        return this;
    }
    /**
     * Lists the types for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypes.html
     */
    toListTypes() {
        this.to('appsync:ListTypes');
        return this;
    }
    /**
     * Gives WebAcl permissions to WAF.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_SetWebACL.html
     */
    toSetWebACL() {
        this.to('appsync:SetWebACL');
        return this;
    }
    /**
     * Adds a new schema to your GraphQL API. This operation is asynchronous - GetSchemaCreationStatus can show when it has completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaCreation.html
     */
    toStartSchemaCreation() {
        this.to('appsync:StartSchemaCreation');
        return this;
    }
    /**
     * Tag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('appsync:TagResource');
        return this;
    }
    /**
     * Untag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('appsync:UntagResource');
        return this;
    }
    /**
     * Updates an API key for a given API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiKey.html
     */
    toUpdateApiKey() {
        this.to('appsync:UpdateApiKey');
        return this;
    }
    /**
     * Updates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        this.to('appsync:UpdateDataSource');
        return this;
    }
    /**
     * Updates an existing Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateFunction.html
     */
    toUpdateFunction() {
        this.to('appsync:UpdateFunction');
        return this;
    }
    /**
     * Updates a GraphqlApi object.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateGraphqlApi.html
     */
    toUpdateGraphqlApi() {
        this.to('appsync:UpdateGraphqlApi');
        return this;
    }
    /**
     * Updates a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateResolver.html
     */
    toUpdateResolver() {
        this.to('appsync:UpdateResolver');
        return this;
    }
    /**
     * Updates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateType.html
     */
    toUpdateType() {
        this.to('appsync:UpdateType');
        return this;
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/attaching-a-data-source.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param datasourceName - Identifier for the datasourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatasource(graphQLAPIId, datasourceName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/datasources/${DatasourceName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${DatasourceName}', datasourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type graphqlapi to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-a-graphql-api.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraphqlapi(graphQLAPIId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/configuring-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param fieldName - Identifier for the fieldName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onField(graphQLAPIId, typeName, fieldName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}/fields/${FieldName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${FieldName}', fieldName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type type to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-your-schema.html#adding-a-root-query-type
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onType(graphQLAPIId, typeName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param functionId - Identifier for the functionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunction(graphQLAPIId, functionId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/functions/${FunctionId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${FunctionId}', functionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appsync = Appsync;
//# sourceMappingURL=data:application/json;base64,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