"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chatbot = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [chatbot](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awschatbot.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chatbot extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [chatbot](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awschatbot.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'chatbot';
        this.accessLevelList = {
            "Write": [
                "CreateChimeWebhookConfiguration",
                "CreateSlackChannelConfiguration",
                "DeleteChimeWebhookConfiguration",
                "DeleteSlackChannelConfiguration",
                "RedeemSlackOauthCode",
                "UpdateChimeWebhookConfiguration",
                "UpdateSlackChannelConfiguration"
            ],
            "Read": [
                "DescribeChimeWebhookConfigurations",
                "DescribeSlackChannelConfigurations",
                "DescribeSlackChannels",
                "DescribeSlackWorkspaces",
                "GetSlackOauthParameters"
            ]
        };
    }
    /**
     * Creates an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    toCreateChimeWebhookConfiguration() {
        this.to('chatbot:CreateChimeWebhookConfiguration');
        return this;
    }
    /**
     * Creates an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    toCreateSlackChannelConfiguration() {
        this.to('chatbot:CreateSlackChannelConfiguration');
        return this;
    }
    /**
     * Deletes an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    toDeleteChimeWebhookConfiguration() {
        this.to('chatbot:DeleteChimeWebhookConfiguration');
        return this;
    }
    /**
     * Deletes an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    toDeleteSlackChannelConfiguration() {
        this.to('chatbot:DeleteSlackChannelConfiguration');
        return this;
    }
    /**
     * Lists all AWS Chatbot Chime Webhook Configurations in an AWS Account.
     *
     * Access Level: Read
     */
    toDescribeChimeWebhookConfigurations() {
        this.to('chatbot:DescribeChimeWebhookConfigurations');
        return this;
    }
    /**
     * Lists all AWS Chatbot Slack Channel Configurations in an AWS account.
     *
     * Access Level: Read
     */
    toDescribeSlackChannelConfigurations() {
        this.to('chatbot:DescribeSlackChannelConfigurations');
        return this;
    }
    /**
     * Lists all public Slack channels in the Slack workspace connected to the AWS Account onboarded with AWS Chatbot service.
     *
     * Access Level: Read
     */
    toDescribeSlackChannels() {
        this.to('chatbot:DescribeSlackChannels');
        return this;
    }
    /**
     * Lists all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot service.
     *
     * Access Level: Read
     */
    toDescribeSlackWorkspaces() {
        this.to('chatbot:DescribeSlackWorkspaces');
        return this;
    }
    /**
     * Generate OAuth parameters to request Slack OAuth code to be used by the AWS Chatbot service.
     *
     * Access Level: Read
     */
    toGetSlackOauthParameters() {
        this.to('chatbot:GetSlackOauthParameters');
        return this;
    }
    /**
     * Redeem previously generated parameters with Slack API, to acquire OAuth tokens to be used by the AWS Chatbot service.
     *
     * Access Level: Write
     */
    toRedeemSlackOauthCode() {
        this.to('chatbot:RedeemSlackOauthCode');
        return this;
    }
    /**
     * Updates an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    toUpdateChimeWebhookConfiguration() {
        this.to('chatbot:UpdateChimeWebhookConfiguration');
        return this;
    }
    /**
     * Updates an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    toUpdateSlackChannelConfiguration() {
        this.to('chatbot:UpdateSlackChannelConfiguration');
        return this;
    }
    /**
     * Adds a resource of type ChatbotConfiguration to the statement
     *
     * @param resourceType - Identifier for the resourceType.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChatbotConfiguration(resourceType, resourceName, account, partition) {
        var arn = 'arn:${Partition}:chatbot::${Account}:${ResourceType}/${ResourceName}';
        arn = arn.replace('${ResourceType}', resourceType);
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Chatbot = Chatbot;
//# sourceMappingURL=data:application/json;base64,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