"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codeartifact = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codeartifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodeartifact.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codeartifact extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codeartifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodeartifact.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codeartifact';
        this.accessLevelList = {
            "Write": [
                "AssociateExternalConnection",
                "AssociateWithDownstreamRepository",
                "CopyPackageVersions",
                "CreateDomain",
                "CreateRepository",
                "DeleteDomain",
                "DeletePackageVersions",
                "DeleteRepository",
                "DisassociateExternalConnection",
                "DisposePackageVersions",
                "PublishPackageVersion",
                "PutDomainPermissionsPolicy",
                "PutPackageMetadata",
                "PutRepositoryPermissionsPolicy",
                "UpdatePackageVersionsStatus",
                "UpdateRepository"
            ],
            "Permissions management": [
                "DeleteDomainPermissionsPolicy",
                "DeleteRepositoryPermissionsPolicy"
            ],
            "Read": [
                "DescribeDomain",
                "DescribePackageVersion",
                "DescribeRepository",
                "GetAuthorizationToken",
                "GetDomainPermissionsPolicy",
                "GetPackageVersionAsset",
                "GetPackageVersionReadme",
                "GetRepositoryEndpoint",
                "GetRepositoryPermissionsPolicy",
                "ReadFromRepository"
            ],
            "List": [
                "ListDomains",
                "ListPackageVersionAssets",
                "ListPackageVersionDependencies",
                "ListPackageVersions",
                "ListPackages",
                "ListRepositories",
                "ListRepositoriesInDomain"
            ]
        };
    }
    /**
     * Grants permission to add an external connection to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssociateExternalConnection.html
     */
    toAssociateExternalConnection() {
        this.to('codeartifact:AssociateExternalConnection');
        return this;
    }
    /**
     * Grants permission to associate an existing repository as an upstream repository to another repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repos-upstream.html
     */
    toAssociateWithDownstreamRepository() {
        this.to('codeartifact:AssociateWithDownstreamRepository');
        return this;
    }
    /**
     * Grants permission to copy package versions from one repository to another repository in the same domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CopyPackageVersions.html
     */
    toCopyPackageVersions() {
        this.to('codeartifact:CopyPackageVersions');
        return this;
    }
    /**
     * Grants permission to create a new domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        this.to('codeartifact:CreateDomain');
        return this;
    }
    /**
     * Grants permission to create a new repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        this.to('codeartifact:CreateRepository');
        return this;
    }
    /**
     * Grants permission to delete a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.to('codeartifact:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete the resource policy set on a domain
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomainPermissionsPolicy.html
     */
    toDeleteDomainPermissionsPolicy() {
        this.to('codeartifact:DeleteDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to delete package versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackageVersions.html
     */
    toDeletePackageVersions() {
        this.to('codeartifact:DeletePackageVersions');
        return this;
    }
    /**
     * Grants permission to delete a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        this.to('codeartifact:DeleteRepository');
        return this;
    }
    /**
     * Grants permission to delete the resource policy set on a repository
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepositoryPermissionsPolicy.html
     */
    toDeleteRepositoryPermissionsPolicy() {
        this.to('codeartifact:DeleteRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return information about a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeDomain.html
     */
    toDescribeDomain() {
        this.to('codeartifact:DescribeDomain');
        return this;
    }
    /**
     * Grants permission to return information about a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackageVersion.html
     */
    toDescribePackageVersion() {
        this.to('codeartifact:DescribePackageVersion');
        return this;
    }
    /**
     * Grants permission to return detailed information about a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeRepository.html
     */
    toDescribeRepository() {
        this.to('codeartifact:DescribeRepository');
        return this;
    }
    /**
     * Grants permission to disassociate an external connection from a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisassociateExternalConnection.html
     */
    toDisassociateExternalConnection() {
        this.to('codeartifact:DisassociateExternalConnection');
        return this;
    }
    /**
     * Grants permission to set the status of package versions to Disposed and delete their assets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisposePackageVersions.html
     */
    toDisposePackageVersions() {
        this.to('codeartifact:DisposePackageVersions');
        return this;
    }
    /**
     * Grants permission to generate a temporary authentication token for accessing repositories in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetAuthorizationToken.html
     */
    toGetAuthorizationToken() {
        this.to('codeartifact:GetAuthorizationToken');
        return this;
    }
    /**
     * Grants permission to return a domain's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetDomainPermissionsPolicy.html
     */
    toGetDomainPermissionsPolicy() {
        this.to('codeartifact:GetDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return an asset (or file) that is part of a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionAsset.html
     */
    toGetPackageVersionAsset() {
        this.to('codeartifact:GetPackageVersionAsset');
        return this;
    }
    /**
     * Grants permission to return a package version's readme file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionReadme.html
     */
    toGetPackageVersionReadme() {
        this.to('codeartifact:GetPackageVersionReadme');
        return this;
    }
    /**
     * Grants permission to return an endpoint for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryEndpoint.html
     */
    toGetRepositoryEndpoint() {
        this.to('codeartifact:GetRepositoryEndpoint');
        return this;
    }
    /**
     * Grants permission to return a repository's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryPermissionsPolicy.html
     */
    toGetRepositoryPermissionsPolicy() {
        this.to('codeartifact:GetRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to list the domains in the current user's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        this.to('codeartifact:ListDomains');
        return this;
    }
    /**
     * Grants permission to list a package version's assets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionAssets.html
     */
    toListPackageVersionAssets() {
        this.to('codeartifact:ListPackageVersionAssets');
        return this;
    }
    /**
     * Grants permission to list the direct dependencies of a package version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionDependencies.html
     */
    toListPackageVersionDependencies() {
        this.to('codeartifact:ListPackageVersionDependencies');
        return this;
    }
    /**
     * Grants permission to list a package's versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html
     */
    toListPackageVersions() {
        this.to('codeartifact:ListPackageVersions');
        return this;
    }
    /**
     * Grants permission to list the packages in a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackages.html
     */
    toListPackages() {
        this.to('codeartifact:ListPackages');
        return this;
    }
    /**
     * Grants permission to list the repositories administered by the calling account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositories.html
     */
    toListRepositories() {
        this.to('codeartifact:ListRepositories');
        return this;
    }
    /**
     * Grants permission to list the repositories in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositoriesInDomain.html
     */
    toListRepositoriesInDomain() {
        this.to('codeartifact:ListRepositoriesInDomain');
        return this;
    }
    /**
     * Grants permission to publish assets and metadata to a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    toPublishPackageVersion() {
        this.to('codeartifact:PublishPackageVersion');
        return this;
    }
    /**
     * Grants permission to attach a resource policy to a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutDomainPermissionsPolicy.html
     */
    toPutDomainPermissionsPolicy() {
        this.to('codeartifact:PutDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to add, modify or remove package metadata using a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    toPutPackageMetadata() {
        this.to('codeartifact:PutPackageMetadata');
        return this;
    }
    /**
     * Grants permission to attach a resource policy to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutRepositoryPermissionsPolicy.html
     */
    toPutRepositoryPermissionsPolicy() {
        this.to('codeartifact:PutRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return package assets and metadata from a repository endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    toReadFromRepository() {
        this.to('codeartifact:ReadFromRepository');
        return this;
    }
    /**
     * Grants permission to modify the status of one or more versions of a package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageVersionsStatus.html
     */
    toUpdatePackageVersionsStatus() {
        this.to('codeartifact:UpdatePackageVersionsStatus');
        return this;
    }
    /**
     * Grants permission to modify the properties of a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdateRepository.html
     */
    toUpdateRepository() {
        this.to('codeartifact:UpdateRepository');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRepository(domainName, repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:repository/${DomainName}/${RepositoryName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type package to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param packageFormat - Identifier for the packageFormat.
     * @param packageNamespace - Identifier for the packageNamespace.
     * @param packageName - Identifier for the packageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPackage(domainName, repositoryName, packageFormat, packageNamespace, packageName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:package/${DomainName}/${RepositoryName}/${PackageFormat}/${PackageNamespace}/${PackageName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${PackageFormat}', packageFormat);
        arn = arn.replace('${PackageNamespace}', packageNamespace);
        arn = arn.replace('${PackageName}', packageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codeartifact = Codeartifact;
//# sourceMappingURL=data:application/json;base64,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