"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codecommit = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codecommit](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodecommit.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codecommit extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codecommit](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodecommit.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codecommit';
        this.accessLevelList = {
            "Write": [
                "AssociateApprovalRuleTemplateWithRepository",
                "BatchAssociateApprovalRuleTemplateWithRepositories",
                "BatchDisassociateApprovalRuleTemplateFromRepositories",
                "CreateApprovalRuleTemplate",
                "CreateBranch",
                "CreateCommit",
                "CreatePullRequest",
                "CreatePullRequestApprovalRule",
                "CreateRepository",
                "CreateUnreferencedMergeCommit",
                "DeleteApprovalRuleTemplate",
                "DeleteBranch",
                "DeleteCommentContent",
                "DeleteFile",
                "DeletePullRequestApprovalRule",
                "DeleteRepository",
                "DisassociateApprovalRuleTemplateFromRepository",
                "GitPush",
                "MergeBranchesByFastForward",
                "MergeBranchesBySquash",
                "MergeBranchesByThreeWay",
                "MergePullRequestByFastForward",
                "MergePullRequestBySquash",
                "MergePullRequestByThreeWay",
                "OverridePullRequestApprovalRules",
                "PostCommentForComparedCommit",
                "PostCommentForPullRequest",
                "PostCommentReply",
                "PutCommentReaction",
                "PutFile",
                "PutRepositoryTriggers",
                "TagResource",
                "TestRepositoryTriggers",
                "UntagResource",
                "UpdateApprovalRuleTemplateContent",
                "UpdateApprovalRuleTemplateDescription",
                "UpdateApprovalRuleTemplateName",
                "UpdateComment",
                "UpdateDefaultBranch",
                "UpdatePullRequestApprovalRuleContent",
                "UpdatePullRequestApprovalState",
                "UpdatePullRequestDescription",
                "UpdatePullRequestStatus",
                "UpdatePullRequestTitle",
                "UpdateRepositoryDescription",
                "UpdateRepositoryName",
                "UploadArchive"
            ],
            "Read": [
                "BatchDescribeMergeConflicts",
                "BatchGetCommits",
                "BatchGetPullRequests",
                "BatchGetRepositories",
                "CancelUploadArchive",
                "DescribeMergeConflicts",
                "DescribePullRequestEvents",
                "EvaluatePullRequestApprovalRules",
                "GetApprovalRuleTemplate",
                "GetBlob",
                "GetBranch",
                "GetComment",
                "GetCommentReactions",
                "GetCommentsForComparedCommit",
                "GetCommentsForPullRequest",
                "GetCommit",
                "GetCommitHistory",
                "GetCommitsFromMergeBase",
                "GetDifferences",
                "GetFile",
                "GetFolder",
                "GetMergeCommit",
                "GetMergeConflicts",
                "GetMergeOptions",
                "GetObjectIdentifier",
                "GetPullRequest",
                "GetPullRequestApprovalStates",
                "GetPullRequestOverrideState",
                "GetReferences",
                "GetRepository",
                "GetRepositoryTriggers",
                "GetTree",
                "GetUploadArchiveStatus",
                "GitPull"
            ],
            "List": [
                "ListApprovalRuleTemplates",
                "ListAssociatedApprovalRuleTemplatesForRepository",
                "ListBranches",
                "ListPullRequests",
                "ListRepositories",
                "ListRepositoriesForApprovalRuleTemplate",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Grants permission to associate an approval rule template with a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_AssociateApprovalRuleTemplateWithRepository.html
     */
    toAssociateApprovalRuleTemplateWithRepository() {
        this.to('codecommit:AssociateApprovalRuleTemplateWithRepository');
        return this;
    }
    /**
     * Grants permission to associate an approval rule template with multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchAssociateApprovalRuleTemplateWithRepositories.html
     */
    toBatchAssociateApprovalRuleTemplateWithRepositories() {
        this.to('codecommit:BatchAssociateApprovalRuleTemplateWithRepositories');
        return this;
    }
    /**
     * Grants permission to get information about multiple merge conflicts when attempting to merge two commits using either the three-way merge or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDescribeMergeConflicts.html
     */
    toBatchDescribeMergeConflicts() {
        this.to('codecommit:BatchDescribeMergeConflicts');
        return this;
    }
    /**
     * Grants permission to remove the association between an approval rule template and multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDisassociateApprovalRuleTemplateFromRepositories.html
     */
    toBatchDisassociateApprovalRuleTemplateFromRepositories() {
        this.to('codecommit:BatchDisassociateApprovalRuleTemplateFromRepositories');
        return this;
    }
    /**
     * Grants permission to get return information about one or more commits in an AWS CodeCommit repository.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetCommits.html
     */
    toBatchGetCommits() {
        this.to('codecommit:BatchGetCommits');
        return this;
    }
    /**
     * Grants permission to return information about one or more pull requests in an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    toBatchGetPullRequests() {
        this.to('codecommit:BatchGetPullRequests');
        return this;
    }
    /**
     * Grants permission to get information about multiple repositories
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetRepositories.html
     */
    toBatchGetRepositories() {
        this.to('codecommit:BatchGetRepositories');
        return this;
    }
    /**
     * Grants permission to cancel the uploading of an archive to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toCancelUploadArchive() {
        this.to('codecommit:CancelUploadArchive');
        return this;
    }
    /**
     * Grants permission to create an approval rule template that will automatically create approval rules in pull requests that match the conditions defined in the template; does not grant permission to create approval rules for individual pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateApprovalRuleTemplate.html
     */
    toCreateApprovalRuleTemplate() {
        this.to('codecommit:CreateApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to create a branch in an AWS CodeCommit repository with this API; does not control Git create branch actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateBranch.html
     */
    toCreateBranch() {
        this.to('codecommit:CreateBranch');
        return this;
    }
    /**
     * Grants permission to add, copy, move or update single or multiple files in a branch in an AWS CodeCommit repository, and generate a commit for the changes in the specified branch.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateCommit.html
     */
    toCreateCommit() {
        this.to('codecommit:CreateCommit');
        return this;
    }
    /**
     * Grants permission to create a pull request in the specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequest.html
     */
    toCreatePullRequest() {
        this.to('codecommit:CreatePullRequest');
        return this;
    }
    /**
     * Grants permission to create an approval rule specific to an individual pull request; does not grant permission to create approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequestApprovalRule.html
     */
    toCreatePullRequestApprovalRule() {
        this.to('codecommit:CreatePullRequestApprovalRule');
        return this;
    }
    /**
     * Grants permission to create an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        this.to('codecommit:CreateRepository');
        return this;
    }
    /**
     * Grants permission to create an unreferenced commit that contains the result of merging two commits using either the three-way or the squash merge option; does not control Git merge actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateUnreferencedMergeCommit.html
     */
    toCreateUnreferencedMergeCommit() {
        this.to('codecommit:CreateUnreferencedMergeCommit');
        return this;
    }
    /**
     * Grants permission to delete an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteApprovalRuleTemplate.html
     */
    toDeleteApprovalRuleTemplate() {
        this.to('codecommit:DeleteApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to delete a branch in an AWS CodeCommit repository with this API; does not control Git delete branch actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteBranch.html
     */
    toDeleteBranch() {
        this.to('codecommit:DeleteBranch');
        return this;
    }
    /**
     * Grants permission to delete the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteCommentContent.html
     */
    toDeleteCommentContent() {
        this.to('codecommit:DeleteCommentContent');
        return this;
    }
    /**
     * Grants permission to delete a specified file from a specified branch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteFile.html
     */
    toDeleteFile() {
        this.to('codecommit:DeleteFile');
        return this;
    }
    /**
     * Grants permission to delete approval rule created for a pull request if the rule was not created by an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeletePullRequestApprovalRule.html
     */
    toDeletePullRequestApprovalRule() {
        this.to('codecommit:DeletePullRequestApprovalRule');
        return this;
    }
    /**
     * Grants permission to delete an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        this.to('codecommit:DeleteRepository');
        return this;
    }
    /**
     * Grants permission to get information about specific merge conflicts when attempting to merge two commits using either the three-way or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribeMergeConflicts.html
     */
    toDescribeMergeConflicts() {
        this.to('codecommit:DescribeMergeConflicts');
        return this;
    }
    /**
     * Grants permission to return information about one or more pull request events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribePullRequestEvents.html
     */
    toDescribePullRequestEvents() {
        this.to('codecommit:DescribePullRequestEvents');
        return this;
    }
    /**
     * Grants permission to remove the association between an approval rule template and a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DisassociateApprovalRuleTemplateFromRepository.html
     */
    toDisassociateApprovalRuleTemplateFromRepository() {
        this.to('codecommit:DisassociateApprovalRuleTemplateFromRepository');
        return this;
    }
    /**
     * Grants permission to evaluate whether a pull request is mergable based on its current approval state and approval rule requirements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_EvaluatePullRequestApprovalRules.html
     */
    toEvaluatePullRequestApprovalRules() {
        this.to('codecommit:EvaluatePullRequestApprovalRules');
        return this;
    }
    /**
     * Grants permission to return information about an approval rule template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetApprovalRuleTemplate.html
     */
    toGetApprovalRuleTemplate() {
        this.to('codecommit:GetApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to view the encoded content of an individual file in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBlob.html
     */
    toGetBlob() {
        this.to('codecommit:GetBlob');
        return this;
    }
    /**
     * Grants permission to get details about a branch in an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBranch.html
     */
    toGetBranch() {
        this.to('codecommit:GetBranch');
        return this;
    }
    /**
     * Grants permission to get the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetComment.html
     */
    toGetComment() {
        this.to('codecommit:GetComment');
        return this;
    }
    /**
     * Grants permission to get the reactions on a comment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentReactions.html
     */
    toGetCommentReactions() {
        this.to('codecommit:GetCommentReactions');
        return this;
    }
    /**
     * Grants permission to get information about comments made on the comparison between two commits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForComparedCommit.html
     */
    toGetCommentsForComparedCommit() {
        this.to('codecommit:GetCommentsForComparedCommit');
        return this;
    }
    /**
     * Grants permission to get comments made on a pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForPullRequest.html
     */
    toGetCommentsForPullRequest() {
        this.to('codecommit:GetCommentsForPullRequest');
        return this;
    }
    /**
     * Grants permission to return information about a commit, including commit message and committer information, with this API; does not control Git log actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommit.html
     */
    toGetCommit() {
        this.to('codecommit:GetCommit');
        return this;
    }
    /**
     * Grants permission to get information about the history of commits in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetCommitHistory() {
        this.to('codecommit:GetCommitHistory');
        return this;
    }
    /**
     * Grants permission to get information about the difference between commits in the context of a potential merge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    toGetCommitsFromMergeBase() {
        this.to('codecommit:GetCommitsFromMergeBase');
        return this;
    }
    /**
     * Grants permission to view information about the differences between valid commit specifiers such as a branch, tag, HEAD, commit ID, or other fully qualified reference
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetDifferences.html
     */
    toGetDifferences() {
        this.to('codecommit:GetDifferences');
        return this;
    }
    /**
     * Grants permission to return the base-64 encoded contents of a specified file and its metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFile.html
     */
    toGetFile() {
        this.to('codecommit:GetFile');
        return this;
    }
    /**
     * Grants permission to return the contents of a specified folder in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFolder.html
     */
    toGetFolder() {
        this.to('codecommit:GetFolder');
        return this;
    }
    /**
     * Grants permission to get information about a merge commit created by one of the merge options for pull requests that creates merge commits. Not all merge options create merge commits. This permission does not control Git merge actions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeCommit.html
     */
    toGetMergeCommit() {
        this.to('codecommit:GetMergeCommit');
        return this;
    }
    /**
     * Grants permission to get information about merge conflicts between the before and after commit IDs for a pull request in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeConflicts.html
     */
    toGetMergeConflicts() {
        this.to('codecommit:GetMergeConflicts');
        return this;
    }
    /**
     * Grants permission to get information about merge options for pull requests that can be used to merge two commits; does not control Git merge actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeOptions.html
     */
    toGetMergeOptions() {
        this.to('codecommit:GetMergeOptions');
        return this;
    }
    /**
     * Grants permission to resolve blobs, trees, and commits to their identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetObjectIdentifier() {
        this.to('codecommit:GetObjectIdentifier');
        return this;
    }
    /**
     * Grants permission to get information about a pull request in a specified repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequest.html
     */
    toGetPullRequest() {
        this.to('codecommit:GetPullRequest');
        return this;
    }
    /**
     * Grants permission to retrieve the current approvals on an inputted pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestApprovalStates.html
     */
    toGetPullRequestApprovalStates() {
        this.to('codecommit:GetPullRequestApprovalStates');
        return this;
    }
    /**
     * Grants permission to retrieve the current override state of a given pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestOverrideState.html
     */
    toGetPullRequestOverrideState() {
        this.to('codecommit:GetPullRequestOverrideState');
        return this;
    }
    /**
     * Grants permission to get details about references in an AWS CodeCommit repository; does not control Git reference actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetReferences() {
        this.to('codecommit:GetReferences');
        return this;
    }
    /**
     * Grants permission to get information about an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepository.html
     */
    toGetRepository() {
        this.to('codecommit:GetRepository');
        return this;
    }
    /**
     * Grants permission to get information about triggers configured for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepositoryTriggers.html
     */
    toGetRepositoryTriggers() {
        this.to('codecommit:GetRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to view the contents of a specified tree in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetTree() {
        this.to('codecommit:GetTree');
        return this;
    }
    /**
     * Grants permission to get status information about an archive upload to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toGetUploadArchiveStatus() {
        this.to('codecommit:GetUploadArchiveStatus');
        return this;
    }
    /**
     * Grants permission to pull information from an AWS CodeCommit repository to a local repo
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    toGitPull() {
        this.to('codecommit:GitPull');
        return this;
    }
    /**
     * Grants permission to push information from a local repo to an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    toGitPush() {
        this.to('codecommit:GitPush');
        return this;
    }
    /**
     * Grants permission to list all approval rule templates in an AWS Region for the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListApprovalRuleTemplates.html
     */
    toListApprovalRuleTemplates() {
        this.to('codecommit:ListApprovalRuleTemplates');
        return this;
    }
    /**
     * Grants permission to list approval rule templates that are associated with a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListAssociatedApprovalRuleTemplatesForRepository.html
     */
    toListAssociatedApprovalRuleTemplatesForRepository() {
        this.to('codecommit:ListAssociatedApprovalRuleTemplatesForRepository');
        return this;
    }
    /**
     * Grants permission to list branches for an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListBranches.html
     */
    toListBranches() {
        this.to('codecommit:ListBranches');
        return this;
    }
    /**
     * Grants permission to list pull requests for a specified repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListPullRequests.html
     */
    toListPullRequests() {
        this.to('codecommit:ListPullRequests');
        return this;
    }
    /**
     * Grants permission to list information about AWS CodeCommit repositories in the current Region for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositories.html
     */
    toListRepositories() {
        this.to('codecommit:ListRepositories');
        return this;
    }
    /**
     * Grants permission to list repositories that are associated with an approval rule template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositoriesForApprovalRuleTemplate.html
     */
    toListRepositoriesForApprovalRuleTemplate() {
        this.to('codecommit:ListRepositoriesForApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to list the resource attached to a CodeCommit resource ARN
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('codecommit:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the fast-forward merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByFastForward.html
     */
    toMergeBranchesByFastForward() {
        this.to('codecommit:MergeBranchesByFastForward');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the squash merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesBySquash.html
     */
    toMergeBranchesBySquash() {
        this.to('codecommit:MergeBranchesBySquash');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the three-way merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByThreeWay.html
     */
    toMergeBranchesByThreeWay() {
        this.to('codecommit:MergeBranchesByThreeWay');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the fast-forward merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByFastForward.html
     */
    toMergePullRequestByFastForward() {
        this.to('codecommit:MergePullRequestByFastForward');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the squash merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestBySquash.html
     */
    toMergePullRequestBySquash() {
        this.to('codecommit:MergePullRequestBySquash');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the three-way merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByThreeWay.html
     */
    toMergePullRequestByThreeWay() {
        this.to('codecommit:MergePullRequestByThreeWay');
        return this;
    }
    /**
     * Grants permission to override all approval rules for a pull request, including approval rules created by a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_OverridePullRequestApprovalRules.html
     */
    toOverridePullRequestApprovalRules() {
        this.to('codecommit:OverridePullRequestApprovalRules');
        return this;
    }
    /**
     * Grants permission to post a comment on the comparison between two commits
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForComparedCommit.html
     */
    toPostCommentForComparedCommit() {
        this.to('codecommit:PostCommentForComparedCommit');
        return this;
    }
    /**
     * Grants permission to post a comment on a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForPullRequest.html
     */
    toPostCommentForPullRequest() {
        this.to('codecommit:PostCommentForPullRequest');
        return this;
    }
    /**
     * Grants permission to post a comment in reply to a comment on a comparison between commits or a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentReply.html
     */
    toPostCommentReply() {
        this.to('codecommit:PostCommentReply');
        return this;
    }
    /**
     * Grants permission to post a reaction on a comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutCommentReaction.html
     */
    toPutCommentReaction() {
        this.to('codecommit:PutCommentReaction');
        return this;
    }
    /**
     * Grants permission to add or update a file in a branch in an AWS CodeCommit repository, and generate a commit for the addition in the specified branch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutFile.html
     */
    toPutFile() {
        this.to('codecommit:PutFile');
        return this;
    }
    /**
     * Grants permission to create, update, or delete triggers for a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutRepositoryTriggers.html
     */
    toPutRepositoryTriggers() {
        this.to('codecommit:PutRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to attach resource tags to a CodeCommit resource ARN
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('codecommit:TagResource');
        return this;
    }
    /**
     * Grants permission to test the functionality of repository triggers by sending information to the trigger target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TestRepositoryTriggers.html
     */
    toTestRepositoryTriggers() {
        this.to('codecommit:TestRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to disassociate resource tags from a CodeCommit resource ARN
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('codecommit:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the content of approval rule templates; does not grant permission to update content of approval rules created specifically for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateContent.html
     */
    toUpdateApprovalRuleTemplateContent() {
        this.to('codecommit:UpdateApprovalRuleTemplateContent');
        return this;
    }
    /**
     * Grants permission to update the description of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateDescription.html
     */
    toUpdateApprovalRuleTemplateDescription() {
        this.to('codecommit:UpdateApprovalRuleTemplateDescription');
        return this;
    }
    /**
     * Grants permission to update the name of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateName.html
     */
    toUpdateApprovalRuleTemplateName() {
        this.to('codecommit:UpdateApprovalRuleTemplateName');
        return this;
    }
    /**
     * Grants permission to update the contents of a comment if the identity matches the identity used to create the comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateComment.html
     */
    toUpdateComment() {
        this.to('codecommit:UpdateComment');
        return this;
    }
    /**
     * Grants permission to change the default branch in an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateDefaultBranch.html
     */
    toUpdateDefaultBranch() {
        this.to('codecommit:UpdateDefaultBranch');
        return this;
    }
    /**
     * Grants permission to update the content for approval rules created for a specific pull requests; does not grant permission to update approval rule content for rules created with an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalRuleContent.html
     */
    toUpdatePullRequestApprovalRuleContent() {
        this.to('codecommit:UpdatePullRequestApprovalRuleContent');
        return this;
    }
    /**
     * Grants permission to update the approval state for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalState.html
     */
    toUpdatePullRequestApprovalState() {
        this.to('codecommit:UpdatePullRequestApprovalState');
        return this;
    }
    /**
     * Grants permission to update the description of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestDescription.html
     */
    toUpdatePullRequestDescription() {
        this.to('codecommit:UpdatePullRequestDescription');
        return this;
    }
    /**
     * Grants permission to update the status of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestStatus.html
     */
    toUpdatePullRequestStatus() {
        this.to('codecommit:UpdatePullRequestStatus');
        return this;
    }
    /**
     * Grants permission to update the title of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestTitle.html
     */
    toUpdatePullRequestTitle() {
        this.to('codecommit:UpdatePullRequestTitle');
        return this;
    }
    /**
     * Grants permission to change the description of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryDescription.html
     */
    toUpdateRepositoryDescription() {
        this.to('codecommit:UpdateRepositoryDescription');
        return this;
    }
    /**
     * Grants permission to change the name of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryName.html
     */
    toUpdateRepositoryName() {
        this.to('codecommit:UpdateRepositoryName');
        return this;
    }
    /**
     * Grants permission to the service role for AWS CodePipeline to upload repository changes into a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toUploadArchive() {
        this.to('codecommit:UploadArchive');
        return this;
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRepository(repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}';
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by Git reference to specified AWS CodeCommit actions
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/how-to-conditional-branch.html
     *
     * Applies to actions:
     * - .toCreateBranch()
     * - .toCreateCommit()
     * - .toCreateUnreferencedMergeCommit()
     * - .toDeleteBranch()
     * - .toDeleteFile()
     * - .toGetMergeCommit()
     * - .toGitPush()
     * - .toMergeBranchesByFastForward()
     * - .toMergeBranchesBySquash()
     * - .toMergeBranchesByThreeWay()
     * - .toMergePullRequestByFastForward()
     * - .toMergePullRequestBySquash()
     * - .toMergePullRequestByThreeWay()
     * - .toPutFile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReferences(value, operator) {
        return this.if(`codecommit:References`, value, operator || 'StringLike');
    }
}
exports.Codecommit = Codecommit;
//# sourceMappingURL=data:application/json;base64,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