"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Signer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [signer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodesigningforamazonfreertos.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Signer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [signer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodesigningforamazonfreertos.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'signer';
        this.accessLevelList = {
            "Write": [
                "CancelSigningProfile",
                "PutSigningProfile",
                "StartSigningJob"
            ],
            "Read": [
                "DescribeSigningJob",
                "GetSigningPlatform",
                "GetSigningProfile"
            ],
            "List": [
                "ListSigningJobs",
                "ListSigningPlatforms",
                "ListSigningProfiles",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Cancels a signing profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_CancelSigningProfile.html
     */
    toCancelSigningProfile() {
        this.to('signer:CancelSigningProfile');
        return this;
    }
    /**
     * Describe a signing job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_DescribeSigningJob.html
     */
    toDescribeSigningJob() {
        this.to('signer:DescribeSigningJob');
        return this;
    }
    /**
     * Retrieves a signing platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_GetSigningPlatform.html
     */
    toGetSigningPlatform() {
        this.to('signer:GetSigningPlatform');
        return this;
    }
    /**
     * Retrieves a signing profile.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_GetSigningProfile.html
     */
    toGetSigningProfile() {
        this.to('signer:GetSigningProfile');
        return this;
    }
    /**
     * List signing jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningJobs.html
     */
    toListSigningJobs() {
        this.to('signer:ListSigningJobs');
        return this;
    }
    /**
     * List all signing platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningPlatforms.html
     */
    toListSigningPlatforms() {
        this.to('signer:ListSigningPlatforms');
        return this;
    }
    /**
     * List all signing profile associated with the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningProfiles.html
     */
    toListSigningProfiles() {
        this.to('signer:ListSigningProfiles');
        return this;
    }
    /**
     * Lists the tags associated with the Signing Profile resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('signer:ListTagsForResource');
        return this;
    }
    /**
     * Creates a new signing profile if not exists.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_PutSigningProfile.html
     */
    toPutSigningProfile() {
        this.to('signer:PutSigningProfile');
        return this;
    }
    /**
     * Starts a code signing request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_StartSigningJob.html
     */
    toStartSigningJob() {
        this.to('signer:StartSigningJob');
        return this;
    }
    /**
     * Adds one or more tags to an Signing Profile resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('signer:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from an Signing Profile resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('signer:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type signing-profile to the statement
     *
     * https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html
     *
     * @param profileName - Identifier for the profileName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSigningProfile(profileName, region, partition) {
        var arn = 'arn:${Partition}:signer:${Region}::/signing-profiles/${ProfileName}';
        arn = arn.replace('${ProfileName}', profileName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type signing-job to the statement
     *
     * https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html
     *
     * @param jobId - Identifier for the jobId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSigningJob(jobId, region, partition) {
        var arn = 'arn:${Partition}:signer:${Region}::/signing-jobs/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Signer = Signer;
//# sourceMappingURL=data:application/json;base64,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