"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ce = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscostexplorerservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ce extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscostexplorerservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ce';
        this.accessLevelList = {
            "Write": [
                "CreateAnomalyMonitor",
                "CreateAnomalySubscription",
                "CreateCostCategoryDefinition",
                "CreateNotificationSubscription",
                "CreateReport",
                "DeleteAnomalyMonitor",
                "DeleteAnomalySubscription",
                "DeleteCostCategoryDefinition",
                "DeleteNotificationSubscription",
                "DeleteReport",
                "ProvideAnomalyFeedback",
                "UpdateAnomalyMonitor",
                "UpdateAnomalySubscription",
                "UpdateCostCategoryDefinition",
                "UpdateNotificationSubscription",
                "UpdatePreferences",
                "UpdateReport"
            ],
            "Read": [
                "DescribeCostCategoryDefinition",
                "DescribeNotificationSubscription",
                "DescribeReport",
                "GetAnomalies",
                "GetAnomalyMonitors",
                "GetAnomalySubscriptions",
                "GetCostAndUsage",
                "GetCostAndUsageWithResources",
                "GetCostForecast",
                "GetDimensionValues",
                "GetPreferences",
                "GetReservationCoverage",
                "GetReservationPurchaseRecommendation",
                "GetReservationUtilization",
                "GetRightsizingRecommendation",
                "GetSavingsPlansCoverage",
                "GetSavingsPlansPurchaseRecommendation",
                "GetSavingsPlansUtilization",
                "GetSavingsPlansUtilizationDetails",
                "GetTags",
                "GetUsageForecast"
            ],
            "List": [
                "ListCostCategoryDefinitions"
            ]
        };
    }
    /**
     * Grants permission to create a new Anomaly Monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateAnomalyMonitor.html
     */
    toCreateAnomalyMonitor() {
        this.to('ce:CreateAnomalyMonitor');
        return this;
    }
    /**
     * Grants permission to create a new Anomaly Subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateAnomalySubscription.html
     */
    toCreateAnomalySubscription() {
        this.to('ce:CreateAnomalySubscription');
        return this;
    }
    /**
     * Grants permission to create a new Cost Category with the requested name and rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateCostCategoryDefinition.html
     */
    toCreateCostCategoryDefinition() {
        this.to('ce:CreateCostCategoryDefinition');
        return this;
    }
    /**
     * Grants permission to create Reservation expiration alerts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toCreateNotificationSubscription() {
        this.to('ce:CreateNotificationSubscription');
        return this;
    }
    /**
     * Grants permission to create Cost Explorer Reports
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toCreateReport() {
        this.to('ce:CreateReport');
        return this;
    }
    /**
     * Grants permission to delete an Anomaly Monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteAnomalyMonitor.html
     */
    toDeleteAnomalyMonitor() {
        this.to('ce:DeleteAnomalyMonitor');
        return this;
    }
    /**
     * Grants permission to delete an Anomaly Subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteAnomalySubscription.html
     */
    toDeleteAnomalySubscription() {
        this.to('ce:DeleteAnomalySubscription');
        return this;
    }
    /**
     * Grants permission to delete a Cost Category
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteCostCategoryDefinition.html
     */
    toDeleteCostCategoryDefinition() {
        this.to('ce:DeleteCostCategoryDefinition');
        return this;
    }
    /**
     * Grants permission to delete Reservation expiration alerts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDeleteNotificationSubscription() {
        this.to('ce:DeleteNotificationSubscription');
        return this;
    }
    /**
     * Grants permission to delete Cost Explorer Reports
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDeleteReport() {
        this.to('ce:DeleteReport');
        return this;
    }
    /**
     * Grants permission to retrieve descriptions such as the name, ARN, rules, definition, and effective dates of a Cost Category
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DescribeCostCategoryDefinition.html
     */
    toDescribeCostCategoryDefinition() {
        this.to('ce:DescribeCostCategoryDefinition');
        return this;
    }
    /**
     * Grants permission to view Reservation expiration alerts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDescribeNotificationSubscription() {
        this.to('ce:DescribeNotificationSubscription');
        return this;
    }
    /**
     * Grants permission to view Cost Explorer Reports page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDescribeReport() {
        this.to('ce:DescribeReport');
        return this;
    }
    /**
     * Grants permission to retrieve anomalies
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalies.html
     */
    toGetAnomalies() {
        this.to('ce:GetAnomalies');
        return this;
    }
    /**
     * Grants permission to query Anomaly Monitors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalyMonitors.html
     */
    toGetAnomalyMonitors() {
        this.to('ce:GetAnomalyMonitors');
        return this;
    }
    /**
     * Grants permission to query Anomaly Subscriptions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalySubscriptions.html
     */
    toGetAnomalySubscriptions() {
        this.to('ce:GetAnomalySubscriptions');
        return this;
    }
    /**
     * Grants permission to retrieve the cost and usage metrics for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostAndUsage.html
     */
    toGetCostAndUsage() {
        this.to('ce:GetCostAndUsage');
        return this;
    }
    /**
     * Grants permission to retrieve the cost and usage metrics with resources for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostAndUsageWithResources.html
     */
    toGetCostAndUsageWithResources() {
        this.to('ce:GetCostAndUsageWithResources');
        return this;
    }
    /**
     * Grants permission to retrieve a cost forecast for a forecast time period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostForecast.html
     */
    toGetCostForecast() {
        this.to('ce:GetCostForecast');
        return this;
    }
    /**
     * Grants permission to retrieve all available filter values for a filter for a period of time
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetDimensionValues.html
     */
    toGetDimensionValues() {
        this.to('ce:GetDimensionValues');
        return this;
    }
    /**
     * Grants permission to view Cost Explorer Preferences page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toGetPreferences() {
        this.to('ce:GetPreferences');
        return this;
    }
    /**
     * Grants permission to retrieve the reservation coverage for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationCoverage.html
     */
    toGetReservationCoverage() {
        this.to('ce:GetReservationCoverage');
        return this;
    }
    /**
     * Grants permission to retrieve the reservation recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationPurchaseRecommendation.html
     */
    toGetReservationPurchaseRecommendation() {
        this.to('ce:GetReservationPurchaseRecommendation');
        return this;
    }
    /**
     * Grants permission to retrieve the reservation utilization for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationUtilization.html
     */
    toGetReservationUtilization() {
        this.to('ce:GetReservationUtilization');
        return this;
    }
    /**
     * Grants permission to retrieve the rightsizing recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetRightsizingRecommendation.html
     */
    toGetRightsizingRecommendation() {
        this.to('ce:GetRightsizingRecommendation');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans coverage for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansCoverage.html
     */
    toGetSavingsPlansCoverage() {
        this.to('ce:GetSavingsPlansCoverage');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansPurchaseRecommendation.html
     */
    toGetSavingsPlansPurchaseRecommendation() {
        this.to('ce:GetSavingsPlansPurchaseRecommendation');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans utilization for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansUtilization.html
     */
    toGetSavingsPlansUtilization() {
        this.to('ce:GetSavingsPlansUtilization');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans utilization details for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansUtilizationDetails.html
     */
    toGetSavingsPlansUtilizationDetails() {
        this.to('ce:GetSavingsPlansUtilizationDetails');
        return this;
    }
    /**
     * Grants permission to query tags for a specified time period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetTags.html
     */
    toGetTags() {
        this.to('ce:GetTags');
        return this;
    }
    /**
     * Grants permission to retrieve a usage forecast for a forecast time period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetUsageForecast.html
     */
    toGetUsageForecast() {
        this.to('ce:GetUsageForecast');
        return this;
    }
    /**
     * Grants permission to retrieve names, ARN, and effective dates for all Cost Categories
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListCostCategoryDefinitions.html
     */
    toListCostCategoryDefinitions() {
        this.to('ce:ListCostCategoryDefinitions');
        return this;
    }
    /**
     * Grants permission to provide feedback on detected anomalies
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ProvideAnomalyFeedback.html
     */
    toProvideAnomalyFeedback() {
        this.to('ce:ProvideAnomalyFeedback');
        return this;
    }
    /**
     * Grants permission to update an existing Anomaly Monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateAnomalyMonitor.html
     */
    toUpdateAnomalyMonitor() {
        this.to('ce:UpdateAnomalyMonitor');
        return this;
    }
    /**
     * Grants permission to update an existing Anomaly Subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateAnomalySubscription.html
     */
    toUpdateAnomalySubscription() {
        this.to('ce:UpdateAnomalySubscription');
        return this;
    }
    /**
     * Grants permission to update an existing Cost Category
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateCostCategoryDefinition.html
     */
    toUpdateCostCategoryDefinition() {
        this.to('ce:UpdateCostCategoryDefinition');
        return this;
    }
    /**
     * Grants permission to update Reservation expiration alerts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toUpdateNotificationSubscription() {
        this.to('ce:UpdateNotificationSubscription');
        return this;
    }
    /**
     * Grants permission to edit Cost Explorer Preferences page
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toUpdatePreferences() {
        this.to('ce:UpdatePreferences');
        return this;
    }
    /**
     * Grants permission to update Cost Explorer Reports
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toUpdateReport() {
        this.to('ce:UpdateReport');
        return this;
    }
}
exports.Ce = Ce;
//# sourceMappingURL=data:application/json;base64,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