"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dms';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Write": [
                "ApplyPendingMaintenanceAction",
                "CreateEndpoint",
                "CreateEventSubscription",
                "CreateReplicationInstance",
                "CreateReplicationSubnetGroup",
                "CreateReplicationTask",
                "DeleteCertificate",
                "DeleteEndpoint",
                "DeleteEventSubscription",
                "DeleteReplicationInstance",
                "DeleteReplicationSubnetGroup",
                "DeleteReplicationTask",
                "ImportCertificate",
                "ModifyEndpoint",
                "ModifyEventSubscription",
                "ModifyReplicationInstance",
                "ModifyReplicationSubnetGroup",
                "ModifyReplicationTask",
                "RebootReplicationInstance",
                "RefreshSchemas",
                "ReloadTables",
                "StartReplicationTask",
                "StartReplicationTaskAssessment",
                "StopReplicationTask"
            ],
            "Read": [
                "DescribeAccountAttributes",
                "DescribeCertificates",
                "DescribeConnections",
                "DescribeEndpointTypes",
                "DescribeEndpoints",
                "DescribeEventCategories",
                "DescribeEventSubscriptions",
                "DescribeEvents",
                "DescribeOrderableReplicationInstances",
                "DescribeRefreshSchemasStatus",
                "DescribeReplicationInstanceTaskLogs",
                "DescribeReplicationInstances",
                "DescribeReplicationSubnetGroups",
                "DescribeReplicationTaskAssessmentResults",
                "DescribeReplicationTasks",
                "DescribeSchemas",
                "DescribeTableStatistics",
                "TestConnection"
            ],
            "List": [
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Adds metadata tags to a DMS resource, including replication instance, endpoint, security group, and migration task
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('dms:AddTagsToResource');
        return this;
    }
    /**
     * Applies a pending maintenance action to a resource (for example, to a replication instance).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    toApplyPendingMaintenanceAction() {
        this.to('dms:ApplyPendingMaintenanceAction');
        return this;
    }
    /**
     * Creates an endpoint using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        this.to('dms:CreateEndpoint');
        return this;
    }
    /**
     * Creates an AWS DMS event notification subscription.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription() {
        this.to('dms:CreateEventSubscription');
        return this;
    }
    /**
     * Creates the replication instance using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationInstance.html
     */
    toCreateReplicationInstance() {
        this.to('dms:CreateReplicationInstance');
        return this;
    }
    /**
     * Creates a replication subnet group given a list of the subnet IDs in a VPC
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationSubnetGroup.html
     */
    toCreateReplicationSubnetGroup() {
        this.to('dms:CreateReplicationSubnetGroup');
        return this;
    }
    /**
     * Creates a replication task using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html
     */
    toCreateReplicationTask() {
        this.to('dms:CreateReplicationTask');
        return this;
    }
    /**
     * Deletes the specified certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteCertificate.html
     */
    toDeleteCertificate() {
        this.to('dms:DeleteCertificate');
        return this;
    }
    /**
     * Deletes the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.to('dms:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes an AWS DMS event subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription() {
        this.to('dms:DeleteEventSubscription');
        return this;
    }
    /**
     * Deletes the specified replication instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationInstance.html
     */
    toDeleteReplicationInstance() {
        this.to('dms:DeleteReplicationInstance');
        return this;
    }
    /**
     * Deletes a subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationSubnetGroup.html
     */
    toDeleteReplicationSubnetGroup() {
        this.to('dms:DeleteReplicationSubnetGroup');
        return this;
    }
    /**
     * Deletes the specified replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTask.html
     */
    toDeleteReplicationTask() {
        this.to('dms:DeleteReplicationTask');
        return this;
    }
    /**
     * Lists all of the AWS DMS attributes for a customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        this.to('dms:DescribeAccountAttributes');
        return this;
    }
    /**
     * Provides a description of the certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeCertificates.html
     */
    toDescribeCertificates() {
        this.to('dms:DescribeCertificates');
        return this;
    }
    /**
     * Describes the status of the connections that have been made between the replication instance and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections() {
        this.to('dms:DescribeConnections');
        return this;
    }
    /**
     * Returns information about the type of endpoints available
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointTypes.html
     */
    toDescribeEndpointTypes() {
        this.to('dms:DescribeEndpointTypes');
        return this;
    }
    /**
     * Returns information about the endpoints for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpoints.html
     */
    toDescribeEndpoints() {
        this.to('dms:DescribeEndpoints');
        return this;
    }
    /**
     * Lists categories for all event source types, or, if specified, for a specified source type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories() {
        this.to('dms:DescribeEventCategories');
        return this;
    }
    /**
     * Lists all the event subscriptions for a customer account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions() {
        this.to('dms:DescribeEventSubscriptions');
        return this;
    }
    /**
     * Lists events for a given source identifier and source type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('dms:DescribeEvents');
        return this;
    }
    /**
     * Returns information about the replication instance types that can be created in the specified region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeOrderableReplicationInstances.html
     */
    toDescribeOrderableReplicationInstances() {
        this.to('dms:DescribeOrderableReplicationInstances');
        return this;
    }
    /**
     * Returns the status of the RefreshSchemas operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRefreshSchemasStatus.html
     */
    toDescribeRefreshSchemasStatus() {
        this.to('dms:DescribeRefreshSchemasStatus');
        return this;
    }
    /**
     * Returns information about the task logs for the specified task.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstanceTaskLogs.html
     */
    toDescribeReplicationInstanceTaskLogs() {
        this.to('dms:DescribeReplicationInstanceTaskLogs');
        return this;
    }
    /**
     * Returns information about replication instances for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstances.html
     */
    toDescribeReplicationInstances() {
        this.to('dms:DescribeReplicationInstances');
        return this;
    }
    /**
     * Returns information about the replication subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationSubnetGroups.html
     */
    toDescribeReplicationSubnetGroups() {
        this.to('dms:DescribeReplicationSubnetGroups');
        return this;
    }
    /**
     * Returns the task assessment results from Amazon S3. This action always returns the latest results.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentResults.html
     */
    toDescribeReplicationTaskAssessmentResults() {
        this.to('dms:DescribeReplicationTaskAssessmentResults');
        return this;
    }
    /**
     * Returns information about replication tasks for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTasks.html
     */
    toDescribeReplicationTasks() {
        this.to('dms:DescribeReplicationTasks');
        return this;
    }
    /**
     * Returns information about the schema for the specified endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeSchemas.html
     */
    toDescribeSchemas() {
        this.to('dms:DescribeSchemas');
        return this;
    }
    /**
     * Returns table statistics on the database migration task, including table name, rows inserted, rows updated, and rows deleted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeTableStatistics.html
     */
    toDescribeTableStatistics() {
        this.to('dms:DescribeTableStatistics');
        return this;
    }
    /**
     * Uploads the specified certificate.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ImportCertificate.html
     */
    toImportCertificate() {
        this.to('dms:ImportCertificate');
        return this;
    }
    /**
     * Lists all tags for an AWS DMS resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('dms:ListTagsForResource');
        return this;
    }
    /**
     * Modifies the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEndpoint.html
     */
    toModifyEndpoint() {
        this.to('dms:ModifyEndpoint');
        return this;
    }
    /**
     * Modifies an existing AWS DMS event notification subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription() {
        this.to('dms:ModifyEventSubscription');
        return this;
    }
    /**
     * Modifies the replication instance to apply new settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationInstance.html
     */
    toModifyReplicationInstance() {
        this.to('dms:ModifyReplicationInstance');
        return this;
    }
    /**
     * Modifies the settings for the specified replication subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationSubnetGroup.html
     */
    toModifyReplicationSubnetGroup() {
        this.to('dms:ModifyReplicationSubnetGroup');
        return this;
    }
    /**
     * Modifies the specified replication task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html
     */
    toModifyReplicationTask() {
        this.to('dms:ModifyReplicationTask');
        return this;
    }
    /**
     * Reboots a replication instance. Rebooting results in a momentary outage, until the replication instance becomes available again.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RebootReplicationInstance.html
     */
    toRebootReplicationInstance() {
        this.to('dms:RebootReplicationInstance');
        return this;
    }
    /**
     * Populates the schema for the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RefreshSchemas.html
     */
    toRefreshSchemas() {
        this.to('dms:RefreshSchemas');
        return this;
    }
    /**
     * Reloads the target database table with the source data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReloadTables.html
     */
    toReloadTables() {
        this.to('dms:ReloadTables');
        return this;
    }
    /**
     * Removes metadata tags from a DMS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('dms:RemoveTagsFromResource');
        return this;
    }
    /**
     * Starts the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html
     */
    toStartReplicationTask() {
        this.to('dms:StartReplicationTask');
        return this;
    }
    /**
     * Starts the replication task assessment for unsupported data types in the source database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessment.html
     */
    toStartReplicationTaskAssessment() {
        this.to('dms:StartReplicationTaskAssessment');
        return this;
    }
    /**
     * Stops the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StopReplicationTask.html
     */
    toStopReplicationTask() {
        this.to('dms:StopReplicationTask');
        return this;
    }
    /**
     * Tests the connection between the replication instance and the endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_TestConnection.html
     */
    toTestConnection() {
        this.to('dms:TestConnection');
        return this;
    }
    /**
     * Adds a resource of type ReplicationInstance to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationInstance.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRepTag()
     */
    onReplicationInstance(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:rep:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ReplicationTask to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTask.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTaskTag()
     */
    onReplicationTask(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:task:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Endpoint to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Endpoint.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEndpointTag()
     */
    onEndpoint(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:endpoint:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Certificate to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Certificate.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCertTag()
     */
    onCertificate(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:cert:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type EventSubscription to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_EventSubscription.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEventSubscription(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:es:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ReplicationSubnetGroup to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationSubnetGroup.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onReplicationSubnetGroup(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:subgrp:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag keys in the request for Certificate
     *
     * Applies to resource types:
     * - Certificate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCertTag(tagKey, value, operator) {
        return this.if(`dms:cert-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for Endpoint
     *
     * Applies to resource types:
     * - Endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointTag(tagKey, value, operator) {
        return this.if(`dms:endpoint-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for EventSubscription
     *
     * Applies to resource types:
     * - EventSubscription
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey, value, operator) {
        return this.if(`dms:es-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationInstance
     *
     * Applies to resource types:
     * - ReplicationInstance
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepTag(tagKey, value, operator) {
        return this.if(`dms:rep-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey, value, operator) {
        return this.if(`dms:req-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationSubnetGroup
     *
     * Applies to resource types:
     * - ReplicationSubnetGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey, value, operator) {
        return this.if(`dms:subgrp-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationTask
     *
     * Applies to resource types:
     * - ReplicationTask
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskTag(tagKey, value, operator) {
        return this.if(`dms:task-tag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Dms = Dms;
//# sourceMappingURL=data:application/json;base64,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