"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dataexchange = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dataexchange](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdataexchange.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dataexchange extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dataexchange](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdataexchange.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dataexchange';
        this.accessLevelList = {
            "Write": [
                "CancelJob",
                "CreateAsset",
                "CreateDataSet",
                "CreateJob",
                "CreateRevision",
                "DeleteAsset",
                "DeleteDataSet",
                "DeleteRevision",
                "GetJob",
                "StartJob",
                "UpdateAsset",
                "UpdateDataSet",
                "UpdateRevision"
            ],
            "Read": [
                "GetAsset",
                "GetDataSet",
                "GetRevision",
                "ListTagsForResource"
            ],
            "List": [
                "ListDataSetRevisions",
                "ListDataSets",
                "ListJobs",
                "ListRevisionAssets"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permissions to cancel a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-jobs.html#CancelJob
     */
    toCancelJob() {
        this.to('dataexchange:CancelJob');
        return this;
    }
    /**
     * Grants permission to create an asset (for example, in a Job).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions.html#CreateAsset
     */
    toCreateAsset() {
        this.to('dataexchange:CreateAsset');
        return this;
    }
    /**
     * Grants permission to create a data set.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets.html#CreateDataSet
     */
    toCreateDataSet() {
        this.to('dataexchange:CreateDataSet');
        return this;
    }
    /**
     * Grants permissions to create a job to import or export assets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-jobs.html#CreateJob
     */
    toCreateJob() {
        this.to('dataexchange:CreateJob');
        return this;
    }
    /**
     * Grants permission to create a revision.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions.html#CreateRevision
     */
    toCreateRevision() {
        this.to('dataexchange:CreateRevision');
        return this;
    }
    /**
     * Grants permissions to delete an asset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions-revisionid-assets-assetid.html#DeleteAsset
     */
    toDeleteAsset() {
        this.to('dataexchange:DeleteAsset');
        return this;
    }
    /**
     * Grants permissions to delete a data set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid.html#DeleteDataSet
     */
    toDeleteDataSet() {
        this.to('dataexchange:DeleteDataSet');
        return this;
    }
    /**
     * Grants permissions to delete a revision.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions-revisionid.html#DeleteRevision
     */
    toDeleteRevision() {
        this.to('dataexchange:DeleteRevision');
        return this;
    }
    /**
     * Grants permissions to get information about an asset and to export it (for example, in a Job).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions-revisionid-assets-assetid.html#GetAsset
     */
    toGetAsset() {
        this.to('dataexchange:GetAsset');
        return this;
    }
    /**
     * Grants permission to get information about a data set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid.html#GetDataSet
     */
    toGetDataSet() {
        this.to('dataexchange:GetDataSet');
        return this;
    }
    /**
     * Grants permissions to get information about a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-jobs.html#GetJob
     */
    toGetJob() {
        this.to('dataexchange:GetJob');
        return this;
    }
    /**
     * Grants permission to get information about a revision.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions-revisionid.html#GetRevision
     */
    toGetRevision() {
        this.to('dataexchange:GetRevision');
        return this;
    }
    /**
     * Grants permissions to list the revisions of a data set.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions.html#ListDataSetRevisions
     */
    toListDataSetRevisions() {
        this.to('dataexchange:ListDataSetRevisions');
        return this;
    }
    /**
     * Grants permission to list data sets for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets.html#ListDataSets
     */
    toListDataSets() {
        this.to('dataexchange:ListDataSets');
        return this;
    }
    /**
     * Grants permissions to list jobs for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-jobs.html#ListJobs
     */
    toListJobs() {
        this.to('dataexchange:ListJobs');
        return this;
    }
    /**
     * Grants permissions to get list the assets of a revision.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions-revisionid-assets.html#ListRevisionAssets
     */
    toListRevisionAssets() {
        this.to('dataexchange:ListRevisionAssets');
        return this;
    }
    /**
     * Grants permission to list the tags that you associated with the specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/tags-resource-arn.html#ListTagsForResource
     */
    toListTagsForResource() {
        this.to('dataexchange:ListTagsForResource');
        return this;
    }
    /**
     * Grants permissions to start a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-jobs.html#StartJob
     */
    toStartJob() {
        this.to('dataexchange:StartJob');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/tags-resource-arn.html#TagResource
     */
    toTagResource() {
        this.to('dataexchange:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/tags-resource-arn.html#UntagResource
     */
    toUntagResource() {
        this.to('dataexchange:UntagResource');
        return this;
    }
    /**
     * Grants permissions to get update information about an asset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions-revisionid-assets-assetid.html#UpdateAsset
     */
    toUpdateAsset() {
        this.to('dataexchange:UpdateAsset');
        return this;
    }
    /**
     * Grants permissions to update information about a data set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid.html#UpdateDataSet
     */
    toUpdateDataSet() {
        this.to('dataexchange:UpdateDataSet');
        return this;
    }
    /**
     * Grants permissions to update information about a revision.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/v1-data-sets-datasetid-revisions-revisionid.html#UpdateRevision
     */
    toUpdateRevision() {
        this.to('dataexchange:UpdateRevision');
        return this;
    }
    /**
     * Adds a resource of type jobs to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifJobType()
     */
    onJobs(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:dataexchange:${Region}:${Account}:jobs/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type data-sets to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDataSets(dataSetId, account, region, partition) {
        var arn = 'arn:${Partition}:dataexchange:${Region}:${Account}:data-sets/${DataSetId}';
        arn = arn.replace('${DataSetId}', dataSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type revisions to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html#revisions
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param revisionId - Identifier for the revisionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRevisions(dataSetId, revisionId, account, region, partition) {
        var arn = 'arn:${Partition}:dataexchange:${Region}:${Account}:data-sets/${DataSetId}/revisions/${RevisionId}';
        arn = arn.replace('${DataSetId}', dataSetId);
        arn = arn.replace('${RevisionId}', revisionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type assets to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html#assets
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param revisionId - Identifier for the revisionId.
     * @param assetId - Identifier for the assetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssets(dataSetId, revisionId, assetId, account, region, partition) {
        var arn = 'arn:${Partition}:dataexchange:${Region}:${Account}:data-sets/${DataSetId}/revisions/${RevisionId}/assets/${AssetId}';
        arn = arn.replace('${DataSetId}', dataSetId);
        arn = arn.replace('${RevisionId}', revisionId);
        arn = arn.replace('${AssetId}', assetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Indicates that the action can only be performed on the specified job type.
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/access-control.html
     *
     * Applies to resource types:
     * - jobs
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobType(value, operator) {
        return this.if(`dataexchange:JobType`, value, operator || 'StringLike');
    }
}
exports.Dataexchange = Dataexchange;
//# sourceMappingURL=data:application/json;base64,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