"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deeplens = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [deeplens](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeeplens.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deeplens extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [deeplens](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeeplens.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'deeplens';
        this.accessLevelList = {
            "Permissions management": [
                "AssociateServiceRoleToAccount"
            ],
            "Read": [
                "BatchGetDevice",
                "BatchGetModel",
                "BatchGetProject",
                "GetAssociatedResources",
                "GetDeploymentStatus",
                "GetDevice",
                "GetModel",
                "GetProject"
            ],
            "Write": [
                "CreateDeviceCertificates",
                "CreateModel",
                "CreateProject",
                "DeleteModel",
                "DeleteProject",
                "DeployProject",
                "DeregisterDevice",
                "ImportProjectFromTemplate",
                "RegisterDevice",
                "RemoveProject",
                "UpdateProject"
            ],
            "List": [
                "ListDeployments",
                "ListDevices",
                "ListModels",
                "ListProjects"
            ]
        };
    }
    /**
     * Associates the user's account with IAM roles controlling various permissions needed by AWS DeepLens for proper functionality.
     *
     * Access Level: Permissions management
     */
    toAssociateServiceRoleToAccount() {
        this.to('deeplens:AssociateServiceRoleToAccount');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens devices.
     *
     * Access Level: Read
     */
    toBatchGetDevice() {
        this.to('deeplens:BatchGetDevice');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Models.
     *
     * Access Level: Read
     */
    toBatchGetModel() {
        this.to('deeplens:BatchGetModel');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Projects.
     *
     * Access Level: Read
     */
    toBatchGetProject() {
        this.to('deeplens:BatchGetProject');
        return this;
    }
    /**
     * Creates a certificate package that is used to successfully authenticate and Register an AWS DeepLens device.
     *
     * Access Level: Write
     */
    toCreateDeviceCertificates() {
        this.to('deeplens:CreateDeviceCertificates');
        return this;
    }
    /**
     * Creates a new AWS DeepLens Model.
     *
     * Access Level: Write
     */
    toCreateModel() {
        this.to('deeplens:CreateModel');
        return this;
    }
    /**
     * Creates a new AWS DeepLens Project.
     *
     * Access Level: Write
     */
    toCreateProject() {
        this.to('deeplens:CreateProject');
        return this;
    }
    /**
     * Deletes an AWS DeepLens Model.
     *
     * Access Level: Write
     */
    toDeleteModel() {
        this.to('deeplens:DeleteModel');
        return this;
    }
    /**
     * Deletes an AWS DeepLens Project.
     *
     * Access Level: Write
     */
    toDeleteProject() {
        this.to('deeplens:DeleteProject');
        return this;
    }
    /**
     * Deploys an AWS DeepLens project to a registered AWS DeepLens device.
     *
     * Access Level: Write
     */
    toDeployProject() {
        this.to('deeplens:DeployProject');
        return this;
    }
    /**
     * Begins a device de-registration workflow for a registered AWS DeepLens device.
     *
     * Access Level: Write
     */
    toDeregisterDevice() {
        this.to('deeplens:DeregisterDevice');
        return this;
    }
    /**
     * Retrieves the account level resources associated with the user's account.
     *
     * Access Level: Read
     */
    toGetAssociatedResources() {
        this.to('deeplens:GetAssociatedResources');
        return this;
    }
    /**
     * Retrieves the the deployment status of a particular AWS DeepLens device, along with any associated metadata.
     *
     * Access Level: Read
     */
    toGetDeploymentStatus() {
        this.to('deeplens:GetDeploymentStatus');
        return this;
    }
    /**
     * Retrieves information about an AWS DeepLens device.
     *
     * Access Level: Read
     */
    toGetDevice() {
        this.to('deeplens:GetDevice');
        return this;
    }
    /**
     * Retrieves an AWS DeepLens Model.
     *
     * Access Level: Read
     */
    toGetModel() {
        this.to('deeplens:GetModel');
        return this;
    }
    /**
     * Retrieves an AWS DeepLens Project.
     *
     * Access Level: Read
     */
    toGetProject() {
        this.to('deeplens:GetProject');
        return this;
    }
    /**
     * Creates a new AWS DeepLens project from a sample project template.
     *
     * Access Level: Write
     */
    toImportProjectFromTemplate() {
        this.to('deeplens:ImportProjectFromTemplate');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Deployment identifiers.
     *
     * Access Level: List
     */
    toListDeployments() {
        this.to('deeplens:ListDeployments');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens device identifiers.
     *
     * Access Level: List
     */
    toListDevices() {
        this.to('deeplens:ListDevices');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Model identifiers.
     *
     * Access Level: List
     */
    toListModels() {
        this.to('deeplens:ListModels');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Project identifiers.
     *
     * Access Level: List
     */
    toListProjects() {
        this.to('deeplens:ListProjects');
        return this;
    }
    /**
     * Begins a device registration workflow for an AWS DeepLens device.
     *
     * Access Level: Write
     */
    toRegisterDevice() {
        this.to('deeplens:RegisterDevice');
        return this;
    }
    /**
     * Removes a deployed AWS DeepLens project from an AWS DeepLens device.
     *
     * Access Level: Write
     */
    toRemoveProject() {
        this.to('deeplens:RemoveProject');
        return this;
    }
    /**
     * Updates an existing AWS DeepLens Project.
     *
     * Access Level: Write
     */
    toUpdateProject() {
        this.to('deeplens:UpdateProject');
        return this;
    }
    /**
     * Adds a resource of type device to the statement
     *
     * @param deviceName - Identifier for the deviceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevice(deviceName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:device/${DeviceName}';
        arn = arn.replace('${DeviceName}', deviceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:project/${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onModel(modelName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:model/${ModelName}';
        arn = arn.replace('${ModelName}', modelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deeplens = Deeplens;
//# sourceMappingURL=data:application/json;base64,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