"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticbeanstalk = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticbeanstalk](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticbeanstalk.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticbeanstalk extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticbeanstalk](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticbeanstalk.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticbeanstalk';
        this.accessLevelList = {
            "Write": [
                "AbortEnvironmentUpdate",
                "ApplyEnvironmentManagedAction",
                "AssociateEnvironmentOperationsRole",
                "ComposeEnvironments",
                "CreateApplication",
                "CreateApplicationVersion",
                "CreateConfigurationTemplate",
                "CreateEnvironment",
                "CreatePlatformVersion",
                "CreateStorageLocation",
                "DeleteApplication",
                "DeleteApplicationVersion",
                "DeleteConfigurationTemplate",
                "DeleteEnvironmentConfiguration",
                "DeletePlatformVersion",
                "DisassociateEnvironmentOperationsRole",
                "PutInstanceStatistics",
                "RebuildEnvironment",
                "RestartAppServer",
                "SwapEnvironmentCNAMEs",
                "TerminateEnvironment",
                "UpdateApplication",
                "UpdateApplicationResourceLifecycle",
                "UpdateApplicationVersion",
                "UpdateConfigurationTemplate",
                "UpdateEnvironment"
            ],
            "Tagging": [
                "AddTags",
                "RemoveTags"
            ],
            "Read": [
                "CheckDNSAvailability",
                "DescribeAccountAttributes",
                "DescribeConfigurationOptions",
                "DescribeConfigurationSettings",
                "DescribeEnvironmentHealth",
                "DescribeEnvironmentManagedActionHistory",
                "DescribeEnvironmentManagedActions",
                "DescribeEnvironmentResources",
                "DescribeEvents",
                "DescribeInstancesHealth",
                "DescribePlatformVersion",
                "ListTagsForResource",
                "RequestEnvironmentInfo",
                "RetrieveEnvironmentInfo",
                "ValidateConfigurationSettings"
            ],
            "List": [
                "DescribeApplicationVersions",
                "DescribeApplications",
                "DescribeEnvironments",
                "ListAvailableSolutionStacks",
                "ListPlatformBranches",
                "ListPlatformVersions"
            ]
        };
    }
    /**
     * Grants permission to cancel in-progress environment configuration update or application version deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AbortEnvironmentUpdate.html
     */
    toAbortEnvironmentUpdate() {
        this.to('elasticbeanstalk:AbortEnvironmentUpdate');
        return this;
    }
    /**
     * Grants permission to add tags to an Elastic Beanstalk resource and to update tag values.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html
     */
    toAddTags() {
        this.to('elasticbeanstalk:AddTags');
        return this;
    }
    /**
     * Grants permission to apply a scheduled managed action immediately.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ApplyEnvironmentManagedAction.html
     */
    toApplyEnvironmentManagedAction() {
        this.to('elasticbeanstalk:ApplyEnvironmentManagedAction');
        return this;
    }
    /**
     * Grants permission to associate an operations role with an environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AssociateEnvironmentOperationsRole.html
     */
    toAssociateEnvironmentOperationsRole() {
        this.to('elasticbeanstalk:AssociateEnvironmentOperationsRole');
        return this;
    }
    /**
     * Grants permission to check CNAME availability.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CheckDNSAvailability.html
     */
    toCheckDNSAvailability() {
        this.to('elasticbeanstalk:CheckDNSAvailability');
        return this;
    }
    /**
     * Grants permission to create or update a group of environments, each running a separate component of a single application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ComposeEnvironments.html
     */
    toComposeEnvironments() {
        this.to('elasticbeanstalk:ComposeEnvironments');
        return this;
    }
    /**
     * Grants permission to create a new application.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplication.html
     */
    toCreateApplication() {
        this.to('elasticbeanstalk:CreateApplication');
        return this;
    }
    /**
     * Grants permission to create an application version for an application.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplicationVersion.html
     */
    toCreateApplicationVersion() {
        this.to('elasticbeanstalk:CreateApplicationVersion');
        return this;
    }
    /**
     * Grants permission to create a configuration template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplication()
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromEnvironment()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateConfigurationTemplate.html
     */
    toCreateConfigurationTemplate() {
        this.to('elasticbeanstalk:CreateConfigurationTemplate');
        return this;
    }
    /**
     * Grants permission to launch an environment for an application.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateEnvironment.html
     */
    toCreateEnvironment() {
        this.to('elasticbeanstalk:CreateEnvironment');
        return this;
    }
    /**
     * Grants permission to create a new version of a custom platform.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreatePlatformVersion.html
     */
    toCreatePlatformVersion() {
        this.to('elasticbeanstalk:CreatePlatformVersion');
        return this;
    }
    /**
     * Grants permission to create the Amazon S3 storage location for the account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateStorageLocation.html
     */
    toCreateStorageLocation() {
        this.to('elasticbeanstalk:CreateStorageLocation');
        return this;
    }
    /**
     * Grants permission to delete an application along with all associated versions and configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplication.html
     */
    toDeleteApplication() {
        this.to('elasticbeanstalk:DeleteApplication');
        return this;
    }
    /**
     * Grants permission to delete an application version from an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplicationVersion.html
     */
    toDeleteApplicationVersion() {
        this.to('elasticbeanstalk:DeleteApplicationVersion');
        return this;
    }
    /**
     * Grants permission to delete a configuration template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteConfigurationTemplate.html
     */
    toDeleteConfigurationTemplate() {
        this.to('elasticbeanstalk:DeleteConfigurationTemplate');
        return this;
    }
    /**
     * Grants permission to delete the draft configuration associated with the running environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteEnvironmentConfiguration.html
     */
    toDeleteEnvironmentConfiguration() {
        this.to('elasticbeanstalk:DeleteEnvironmentConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a version of a custom platform.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeletePlatformVersion.html
     */
    toDeletePlatformVersion() {
        this.to('elasticbeanstalk:DeletePlatformVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a list of account attributes, including resource quotas.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        this.to('elasticbeanstalk:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to retrieve a list of application versions stored in an AWS Elastic Beanstalk storage bucket.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplicationVersions.html
     */
    toDescribeApplicationVersions() {
        this.to('elasticbeanstalk:DescribeApplicationVersions');
        return this;
    }
    /**
     * Grants permission to retrieve the descriptions of existing applications.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplications.html
     */
    toDescribeApplications() {
        this.to('elasticbeanstalk:DescribeApplications');
        return this;
    }
    /**
     * Grants permission to retrieve descriptions of environment configuration options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationOptions.html
     */
    toDescribeConfigurationOptions() {
        this.to('elasticbeanstalk:DescribeConfigurationOptions');
        return this;
    }
    /**
     * Grants permission to retrieve a description of the settings for a configuration set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationSettings.html
     */
    toDescribeConfigurationSettings() {
        this.to('elasticbeanstalk:DescribeConfigurationSettings');
        return this;
    }
    /**
     * Grants permission to retrieve information about the overall health of an environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentHealth.html
     */
    toDescribeEnvironmentHealth() {
        this.to('elasticbeanstalk:DescribeEnvironmentHealth');
        return this;
    }
    /**
     * Grants permission to retrieve a list of an environment's completed and failed managed actions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActionHistory.html
     */
    toDescribeEnvironmentManagedActionHistory() {
        this.to('elasticbeanstalk:DescribeEnvironmentManagedActionHistory');
        return this;
    }
    /**
     * Grants permission to retrieve a list of an environment's upcoming and in-progress managed actions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActions.html
     */
    toDescribeEnvironmentManagedActions() {
        this.to('elasticbeanstalk:DescribeEnvironmentManagedActions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS resources for an environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentResources.html
     */
    toDescribeEnvironmentResources() {
        this.to('elasticbeanstalk:DescribeEnvironmentResources');
        return this;
    }
    /**
     * Grants permission to retrieve descriptions for existing environments.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironments.html
     */
    toDescribeEnvironments() {
        this.to('elasticbeanstalk:DescribeEnvironments');
        return this;
    }
    /**
     * Grants permission to retrieve a list of event descriptions matching a set of criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('elasticbeanstalk:DescribeEvents');
        return this;
    }
    /**
     * Grants permission to retrieve more detailed information about the health of environment instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeInstancesHealth.html
     */
    toDescribeInstancesHealth() {
        this.to('elasticbeanstalk:DescribeInstancesHealth');
        return this;
    }
    /**
     * Grants permission to retrieve a description of a platform version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribePlatformVersion.html
     */
    toDescribePlatformVersion() {
        this.to('elasticbeanstalk:DescribePlatformVersion');
        return this;
    }
    /**
     * Grants permission to disassociate an operations role with an environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DisassociateEnvironmentOperationsRole.html
     */
    toDisassociateEnvironmentOperationsRole() {
        this.to('elasticbeanstalk:DisassociateEnvironmentOperationsRole');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the available solution stack names.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html
     */
    toListAvailableSolutionStacks() {
        this.to('elasticbeanstalk:ListAvailableSolutionStacks');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the available platform branches.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformBranches.html
     */
    toListPlatformBranches() {
        this.to('elasticbeanstalk:ListPlatformBranches');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the available platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformVersions.html
     */
    toListPlatformVersions() {
        this.to('elasticbeanstalk:ListPlatformVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags of an Elastic Beanstalk resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('elasticbeanstalk:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to submit instance statistics for enhanced health.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced.html#health-enhanced-authz
     */
    toPutInstanceStatistics() {
        this.to('elasticbeanstalk:PutInstanceStatistics');
        return this;
    }
    /**
     * Grants permission to delete and recreate all of the AWS resources for an environment and to force a restart.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RebuildEnvironment.html
     */
    toRebuildEnvironment() {
        this.to('elasticbeanstalk:RebuildEnvironment');
        return this;
    }
    /**
     * Grants permission to remove tags from an Elastic Beanstalk resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html
     */
    toRemoveTags() {
        this.to('elasticbeanstalk:RemoveTags');
        return this;
    }
    /**
     * Grants permission to initiate a request to compile information of the deployed environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RequestEnvironmentInfo.html
     */
    toRequestEnvironmentInfo() {
        this.to('elasticbeanstalk:RequestEnvironmentInfo');
        return this;
    }
    /**
     * Grants permission to request an environment to restart the application container server running on each Amazon EC2 instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RestartAppServer.html
     */
    toRestartAppServer() {
        this.to('elasticbeanstalk:RestartAppServer');
        return this;
    }
    /**
     * Grants permission to retrieve the compiled information from a RequestEnvironmentInfo request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RetrieveEnvironmentInfo.html
     */
    toRetrieveEnvironmentInfo() {
        this.to('elasticbeanstalk:RetrieveEnvironmentInfo');
        return this;
    }
    /**
     * Grants permission to swap the CNAMEs of two environments.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromEnvironment()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_SwapEnvironmentCNAMEs.html
     */
    toSwapEnvironmentCNAMEs() {
        this.to('elasticbeanstalk:SwapEnvironmentCNAMEs');
        return this;
    }
    /**
     * Grants permission to terminate an environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_TerminateEnvironment.html
     */
    toTerminateEnvironment() {
        this.to('elasticbeanstalk:TerminateEnvironment');
        return this;
    }
    /**
     * Grants permission to update an application with specified properties.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplication.html
     */
    toUpdateApplication() {
        this.to('elasticbeanstalk:UpdateApplication');
        return this;
    }
    /**
     * Grants permission to update the application version lifecycle policy associated with the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationResourceLifecycle.html
     */
    toUpdateApplicationResourceLifecycle() {
        this.to('elasticbeanstalk:UpdateApplicationResourceLifecycle');
        return this;
    }
    /**
     * Grants permission to update an application version with specified properties.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationVersion.html
     */
    toUpdateApplicationVersion() {
        this.to('elasticbeanstalk:UpdateApplicationVersion');
        return this;
    }
    /**
     * Grants permission to update a configuration template with specified properties or configuration option values.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplication()
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromEnvironment()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateConfigurationTemplate.html
     */
    toUpdateConfigurationTemplate() {
        this.to('elasticbeanstalk:UpdateConfigurationTemplate');
        return this;
    }
    /**
     * Grants permission to update an environment.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateEnvironment.html
     */
    toUpdateEnvironment() {
        this.to('elasticbeanstalk:UpdateEnvironment');
        return this;
    }
    /**
     * Grants permission to check the validity of a set of configuration settings for a configuration template or an environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ValidateConfigurationSettings.html
     */
    toValidateConfigurationSettings() {
        this.to('elasticbeanstalk:ValidateConfigurationSettings');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:application/${ApplicationName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type applicationversion to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param versionLabel - Identifier for the versionLabel.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInApplication()
     */
    onApplicationversion(applicationName, versionLabel, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:applicationversion/${ApplicationName}/${VersionLabel}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${VersionLabel}', versionLabel);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type configurationtemplate to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInApplication()
     */
    onConfigurationtemplate(applicationName, templateName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:configurationtemplate/${ApplicationName}/${TemplateName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param environmentName - Identifier for the environmentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInApplication()
     */
    onEnvironment(applicationName, environmentName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:environment/${ApplicationName}/${EnvironmentName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${EnvironmentName}', environmentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type solutionstack to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param solutionStackName - Identifier for the solutionStackName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSolutionstack(solutionStackName, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}::solutionstack/${SolutionStackName}';
        arn = arn.replace('${SolutionStackName}', solutionStackName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type platform to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param platformNameWithVersion - Identifier for the platformNameWithVersion.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPlatform(platformNameWithVersion, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}::platform/${PlatformNameWithVersion}';
        arn = arn.replace('${PlatformNameWithVersion}', platformNameWithVersion);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by an application as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toUpdateConfigurationTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromApplication(value, operator) {
        return this.if(`elasticbeanstalk:FromApplication`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by an application version as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromApplicationVersion(value, operator) {
        return this.if(`elasticbeanstalk:FromApplicationVersion`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by a configuration template as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromConfigurationTemplate(value, operator) {
        return this.if(`elasticbeanstalk:FromConfigurationTemplate`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by an environment as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toSwapEnvironmentCNAMEs()
     * - .toUpdateConfigurationTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromEnvironment(value, operator) {
        return this.if(`elasticbeanstalk:FromEnvironment`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by a platform as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromPlatform(value, operator) {
        return this.if(`elasticbeanstalk:FromPlatform`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by a solution stack as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromSolutionStack(value, operator) {
        return this.if(`elasticbeanstalk:FromSolutionStack`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by the application that contains the resource that the action operates on.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toAbortEnvironmentUpdate()
     * - .toApplyEnvironmentManagedAction()
     * - .toComposeEnvironments()
     * - .toCreateApplicationVersion()
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toDeleteApplicationVersion()
     * - .toDeleteConfigurationTemplate()
     * - .toDeleteEnvironmentConfiguration()
     * - .toDescribeApplicationVersions()
     * - .toDescribeConfigurationOptions()
     * - .toDescribeConfigurationSettings()
     * - .toDescribeEnvironmentManagedActionHistory()
     * - .toDescribeEnvironmentManagedActions()
     * - .toDescribeEnvironmentResources()
     * - .toDescribeEnvironments()
     * - .toDescribeEvents()
     * - .toRebuildEnvironment()
     * - .toRequestEnvironmentInfo()
     * - .toRestartAppServer()
     * - .toRetrieveEnvironmentInfo()
     * - .toSwapEnvironmentCNAMEs()
     * - .toTerminateEnvironment()
     * - .toUpdateApplicationVersion()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     * - .toValidateConfigurationSettings()
     *
     * Applies to resource types:
     * - applicationversion
     * - configurationtemplate
     * - environment
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifInApplication(value, operator) {
        return this.if(`elasticbeanstalk:InApplication`, value, operator || 'ArnLike');
    }
}
exports.Elasticbeanstalk = Elasticbeanstalk;
//# sourceMappingURL=data:application/json;base64,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