"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Databrew = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [databrew](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsgluedatabrew.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Databrew extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [databrew](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsgluedatabrew.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'databrew';
        this.accessLevelList = {
            "Write": [
                "BatchDeleteRecipeVersion",
                "CreateDataset",
                "CreateProfileJob",
                "CreateProject",
                "CreateRecipe",
                "CreateRecipeJob",
                "CreateSchedule",
                "DeleteDataset",
                "DeleteJob",
                "DeleteProject",
                "DeleteRecipeVersion",
                "DeleteSchedule",
                "PublishRecipe",
                "SendProjectSessionAction",
                "StartJobRun",
                "StartProjectSession",
                "StopJobRun",
                "UpdateDataset",
                "UpdateProfileJob",
                "UpdateProject",
                "UpdateRecipe",
                "UpdateRecipeJob",
                "UpdateSchedule"
            ],
            "Read": [
                "DescribeDataset",
                "DescribeJob",
                "DescribeProject",
                "DescribeRecipe",
                "DescribeSchedule"
            ],
            "List": [
                "ListDatasets",
                "ListJobRuns",
                "ListJobs",
                "ListProjects",
                "ListRecipeVersions",
                "ListRecipes",
                "ListSchedules",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to delete one or more recipe versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_BatchDeleteRecipeVersion.html
     */
    toBatchDeleteRecipeVersion() {
        this.to('databrew:BatchDeleteRecipeVersion');
        return this;
    }
    /**
     * Grants permission to create a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_CreateDataset.html
     */
    toCreateDataset() {
        this.to('databrew:CreateDataset');
        return this;
    }
    /**
     * Grants permission to create a profile job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_CreateProfileJob.html
     */
    toCreateProfileJob() {
        this.to('databrew:CreateProfileJob');
        return this;
    }
    /**
     * Grants permission to create a project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_CreateProject.html
     */
    toCreateProject() {
        this.to('databrew:CreateProject');
        return this;
    }
    /**
     * Grants permission to create a recipe
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_CreateRecipe.html
     */
    toCreateRecipe() {
        this.to('databrew:CreateRecipe');
        return this;
    }
    /**
     * Grants permission to create a recipe job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_CreateRecipeJob.html
     */
    toCreateRecipeJob() {
        this.to('databrew:CreateRecipeJob');
        return this;
    }
    /**
     * Grants permission to create a schedule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_CreateSchedule.html
     */
    toCreateSchedule() {
        this.to('databrew:CreateSchedule');
        return this;
    }
    /**
     * Grants permission to delete a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DeleteDataset.html
     */
    toDeleteDataset() {
        this.to('databrew:DeleteDataset');
        return this;
    }
    /**
     * Grants permission to delete a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DeleteJob.html
     */
    toDeleteJob() {
        this.to('databrew:DeleteJob');
        return this;
    }
    /**
     * Grants permission to delete a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DeleteProject.html
     */
    toDeleteProject() {
        this.to('databrew:DeleteProject');
        return this;
    }
    /**
     * Grants permission to delete a recipe version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DeleteRecipeVersion.html
     */
    toDeleteRecipeVersion() {
        this.to('databrew:DeleteRecipeVersion');
        return this;
    }
    /**
     * Grants permission to delete a schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DeleteSchedule.html
     */
    toDeleteSchedule() {
        this.to('databrew:DeleteSchedule');
        return this;
    }
    /**
     * Grants permission to view details about a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DescribeDataset.html
     */
    toDescribeDataset() {
        this.to('databrew:DescribeDataset');
        return this;
    }
    /**
     * Grants permission to view details about a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DescribeJob.html
     */
    toDescribeJob() {
        this.to('databrew:DescribeJob');
        return this;
    }
    /**
     * Grants permission to view details about a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DescribeProject.html
     */
    toDescribeProject() {
        this.to('databrew:DescribeProject');
        return this;
    }
    /**
     * Grants permission to view details about a recipe
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DescribeRecipe.html
     */
    toDescribeRecipe() {
        this.to('databrew:DescribeRecipe');
        return this;
    }
    /**
     * Grants permission to view details about a schedule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_DescribeSchedule.html
     */
    toDescribeSchedule() {
        this.to('databrew:DescribeSchedule');
        return this;
    }
    /**
     * Grants permission to list datasets in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListDatasets.html
     */
    toListDatasets() {
        this.to('databrew:ListDatasets');
        return this;
    }
    /**
     * Grants permission to list job runs for a given job
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListJobRuns.html
     */
    toListJobRuns() {
        this.to('databrew:ListJobRuns');
        return this;
    }
    /**
     * Grants permission to list jobs in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListJobs.html
     */
    toListJobs() {
        this.to('databrew:ListJobs');
        return this;
    }
    /**
     * Grants permission to list projects in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListProjects.html
     */
    toListProjects() {
        this.to('databrew:ListProjects');
        return this;
    }
    /**
     * Grants permission to list versions in your recipe
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListRecipeVersions.html
     */
    toListRecipeVersions() {
        this.to('databrew:ListRecipeVersions');
        return this;
    }
    /**
     * Grants permission to list recipes in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListRecipes.html
     */
    toListRecipes() {
        this.to('databrew:ListRecipes');
        return this;
    }
    /**
     * Grants permission to list schedules in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListSchedules.html
     */
    toListSchedules() {
        this.to('databrew:ListSchedules');
        return this;
    }
    /**
     * Grants permission to retrieve tags associated with a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('databrew:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to publish a major verison of a recipe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_PublishRecipe.html
     */
    toPublishRecipe() {
        this.to('databrew:PublishRecipe');
        return this;
    }
    /**
     * Grants permission to submit an action to the interactive session for a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_SendProjectSessionAction.html
     */
    toSendProjectSessionAction() {
        this.to('databrew:SendProjectSessionAction');
        return this;
    }
    /**
     * Grants permission to start running a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_StartJobRun.html
     */
    toStartJobRun() {
        this.to('databrew:StartJobRun');
        return this;
    }
    /**
     * Grants permission to start an interactive session for a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_StartProjectSession.html
     */
    toStartProjectSession() {
        this.to('databrew:StartProjectSession');
        return this;
    }
    /**
     * Grants permission to stop a job run for a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_StopJobRun.html
     */
    toStopJobRun() {
        this.to('databrew:StopJobRun');
        return this;
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.to('databrew:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags associated with a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('databrew:UntagResource');
        return this;
    }
    /**
     * Grants permission to modify a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_UpdateDataset.html
     */
    toUpdateDataset() {
        this.to('databrew:UpdateDataset');
        return this;
    }
    /**
     * Grants permission to modify a profile job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_UpdateProfileJob.html
     */
    toUpdateProfileJob() {
        this.to('databrew:UpdateProfileJob');
        return this;
    }
    /**
     * Grants permission to modify a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_UpdateProject.html
     */
    toUpdateProject() {
        this.to('databrew:UpdateProject');
        return this;
    }
    /**
     * Grants permission to modify a recipe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_UpdateRecipe.html
     */
    toUpdateRecipe() {
        this.to('databrew:UpdateRecipe');
        return this;
    }
    /**
     * Grants permission to modify a recipe job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_UpdateRecipeJob.html
     */
    toUpdateRecipeJob() {
        this.to('databrew:UpdateRecipeJob');
        return this;
    }
    /**
     * Grants permission to modify a schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/API_UpdateSchedule.html
     */
    toUpdateSchedule() {
        this.to('databrew:UpdateSchedule');
        return this;
    }
    /**
     * Adds a resource of type Project to the statement
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/projects.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(resourceId, account, partition) {
        var arn = 'arn:${Partition}:databrew::${Account}:project/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Dataset to the statement
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/datasets.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(resourceId, account, partition) {
        var arn = 'arn:${Partition}:databrew::${Account}:dataset/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Recipe to the statement
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/recipes.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRecipe(resourceId, account, partition) {
        var arn = 'arn:${Partition}:databrew::${Account}:recipe/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Job to the statement
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/jobs.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(resourceId, account, partition) {
        var arn = 'arn:${Partition}:databrew::${Account}:job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Schedule to the statement
     *
     * https://docs.aws.amazon.com/databrew/latest/dg/jobs.html#jobs.scheduling
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSchedule(resourceId, account, partition) {
        var arn = 'arn:${Partition}:databrew::${Account}:schedule/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Databrew = Databrew;
//# sourceMappingURL=data:application/json;base64,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