"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Health = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [health](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthapisandnotifications.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Health extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [health](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthapisandnotifications.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'health';
        this.accessLevelList = {
            "Read": [
                "DescribeAffectedAccountsForOrganization",
                "DescribeAffectedEntities",
                "DescribeAffectedEntitiesForOrganization",
                "DescribeEntityAggregates",
                "DescribeEventAggregates",
                "DescribeEventDetails",
                "DescribeEventDetailsForOrganization",
                "DescribeEventTypes",
                "DescribeEvents",
                "DescribeEventsForOrganization"
            ],
            "Permissions management": [
                "DescribeHealthServiceStatusForOrganization",
                "DisableHealthServiceAccessForOrganization",
                "EnableHealthServiceAccessForOrganization"
            ]
        };
    }
    /**
     * Gets a list of accounts that have been affected by the specified events in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html
     */
    toDescribeAffectedAccountsForOrganization() {
        this.to('health:DescribeAffectedAccountsForOrganization');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEventTypeCode()
     * - .ifService()
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html
     */
    toDescribeAffectedEntities() {
        this.to('health:DescribeAffectedEntities');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events and accounts in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html
     */
    toDescribeAffectedEntitiesForOrganization() {
        this.to('health:DescribeAffectedEntitiesForOrganization');
        return this;
    }
    /**
     * Returns the number of entities that are affected by each of the specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html
     */
    toDescribeEntityAggregates() {
        this.to('health:DescribeEntityAggregates');
        return this;
    }
    /**
     * Returns the number of events of each event type (issue, scheduled change, and account notification).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html
     */
    toDescribeEventAggregates() {
        this.to('health:DescribeEventAggregates');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEventTypeCode()
     * - .ifService()
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html
     */
    toDescribeEventDetails() {
        this.to('health:DescribeEventDetails');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events for provided accounts in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html
     */
    toDescribeEventDetailsForOrganization() {
        this.to('health:DescribeEventDetailsForOrganization');
        return this;
    }
    /**
     * Returns the event types that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html
     */
    toDescribeEventTypes() {
        this.to('health:DescribeEventTypes');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('health:DescribeEvents');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html
     */
    toDescribeEventsForOrganization() {
        this.to('health:DescribeEventsForOrganization');
        return this;
    }
    /**
     * Returns the status of enabling or disabling the Organizational View feature
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html
     */
    toDescribeHealthServiceStatusForOrganization() {
        this.to('health:DescribeHealthServiceStatusForOrganization');
        return this;
    }
    /**
     * Disables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - organizations:DisableAWSServiceAccess
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html
     */
    toDisableHealthServiceAccessForOrganization() {
        this.to('health:DisableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Enables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html
     */
    toEnableHealthServiceAccessForOrganization() {
        this.to('health:EnableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Adds a resource of type event to the statement
     *
     * https://docs.aws.amazon.com/health/latest/ug/supported-operations.html
     *
     * @param service - Identifier for the service.
     * @param eventTypeCode - Identifier for the eventTypeCode.
     * @param eventTypePlusId - Identifier for the eventTypePlusId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvent(service, eventTypeCode, eventTypePlusId, account, region, partition) {
        var arn = 'arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}';
        arn = arn.replace('${Service}', service);
        arn = arn.replace('${EventTypeCode}', eventTypeCode);
        arn = arn.replace('${EventTypePlusId}', eventTypePlusId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The type of event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * Applies to actions:
     * - .toDescribeAffectedEntities()
     * - .toDescribeEventDetails()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEventTypeCode(value, operator) {
        return this.if(`health:eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * The service of the event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * Applies to actions:
     * - .toDescribeAffectedEntities()
     * - .toDescribeEventDetails()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`health:service`, value, operator || 'StringLike');
    }
}
exports.Health = Health;
//# sourceMappingURL=data:application/json;base64,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