"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotanalytics = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotanalytics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotanalytics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotanalytics extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotanalytics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotanalytics.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotanalytics';
        this.accessLevelList = {
            "Write": [
                "BatchPutMessage",
                "CancelPipelineReprocessing",
                "CreateChannel",
                "CreateDataset",
                "CreateDatasetContent",
                "CreateDatastore",
                "CreatePipeline",
                "DeleteChannel",
                "DeleteDataset",
                "DeleteDatasetContent",
                "DeleteDatastore",
                "DeletePipeline",
                "PutLoggingOptions",
                "StartPipelineReprocessing",
                "UpdateChannel",
                "UpdateDataset",
                "UpdateDatastore",
                "UpdatePipeline"
            ],
            "Read": [
                "DescribeChannel",
                "DescribeDataset",
                "DescribeDatastore",
                "DescribeLoggingOptions",
                "DescribePipeline",
                "GetDatasetContent",
                "ListTagsForResource",
                "RunPipelineActivity",
                "SampleChannelData"
            ],
            "List": [
                "ListChannels",
                "ListDatasets",
                "ListDatastores",
                "ListPipelines"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Puts a batch of messages into the specified channel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_BatchPutMessage.html
     */
    toBatchPutMessage() {
        this.to('iotanalytics:BatchPutMessage');
        return this;
    }
    /**
     * Cancels reprocessing for the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CancelPipelineReprocessing.html
     */
    toCancelPipelineReprocessing() {
        this.to('iotanalytics:CancelPipelineReprocessing');
        return this;
    }
    /**
     * Creates a channel.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        this.to('iotanalytics:CreateChannel');
        return this;
    }
    /**
     * Creates a dataset.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDataset.html
     */
    toCreateDataset() {
        this.to('iotanalytics:CreateDataset');
        return this;
    }
    /**
     * Generates content of the specified dataset (by executing the dataset actions).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDatasetContent.html
     */
    toCreateDatasetContent() {
        this.to('iotanalytics:CreateDatasetContent');
        return this;
    }
    /**
     * Creates a datastore.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreateDatastore.html
     */
    toCreateDatastore() {
        this.to('iotanalytics:CreateDatastore');
        return this;
    }
    /**
     * Creates a pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_CreatePipeline.html
     */
    toCreatePipeline() {
        this.to('iotanalytics:CreatePipeline');
        return this;
    }
    /**
     * Deletes the specified channel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteChannel.html
     */
    toDeleteChannel() {
        this.to('iotanalytics:DeleteChannel');
        return this;
    }
    /**
     * Deletes the specified dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDataset.html
     */
    toDeleteDataset() {
        this.to('iotanalytics:DeleteDataset');
        return this;
    }
    /**
     * Deletes the content of the specified dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDatasetContent.html
     */
    toDeleteDatasetContent() {
        this.to('iotanalytics:DeleteDatasetContent');
        return this;
    }
    /**
     * Deletes the specified datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeleteDatastore.html
     */
    toDeleteDatastore() {
        this.to('iotanalytics:DeleteDatastore');
        return this;
    }
    /**
     * Deletes the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeletePipeline.html
     */
    toDeletePipeline() {
        this.to('iotanalytics:DeletePipeline');
        return this;
    }
    /**
     * Describes the specified channel.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeChannel.html
     */
    toDescribeChannel() {
        this.to('iotanalytics:DescribeChannel');
        return this;
    }
    /**
     * Describes the specified dataset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        this.to('iotanalytics:DescribeDataset');
        return this;
    }
    /**
     * Describes the specified datastore.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeDatastore.html
     */
    toDescribeDatastore() {
        this.to('iotanalytics:DescribeDatastore');
        return this;
    }
    /**
     * Describes logging options for the the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribeLoggingOptions.html
     */
    toDescribeLoggingOptions() {
        this.to('iotanalytics:DescribeLoggingOptions');
        return this;
    }
    /**
     * Describes the specified pipeline.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DescribePipeline.html
     */
    toDescribePipeline() {
        this.to('iotanalytics:DescribePipeline');
        return this;
    }
    /**
     * Gets the content of the specified dataset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_GetDatasetContent.html
     */
    toGetDatasetContent() {
        this.to('iotanalytics:GetDatasetContent');
        return this;
    }
    /**
     * Lists the channels for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        this.to('iotanalytics:ListChannels');
        return this;
    }
    /**
     * Lists the datasets for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListDatasets.html
     */
    toListDatasets() {
        this.to('iotanalytics:ListDatasets');
        return this;
    }
    /**
     * Lists the datastores for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListDatastores.html
     */
    toListDatastores() {
        this.to('iotanalytics:ListDatastores');
        return this;
    }
    /**
     * Lists the pipelines for the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListPipelines.html
     */
    toListPipelines() {
        this.to('iotanalytics:ListPipelines');
        return this;
    }
    /**
     * Lists the tags (metadata) which you have assigned to the resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('iotanalytics:ListTagsForResource');
        return this;
    }
    /**
     * Puts logging options for the the account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        this.to('iotanalytics:PutLoggingOptions');
        return this;
    }
    /**
     * Runs the specified pipeline activity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_RunPipelineActivity.html
     */
    toRunPipelineActivity() {
        this.to('iotanalytics:RunPipelineActivity');
        return this;
    }
    /**
     * Samples the specified channel's data.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_SampleChannelData.html
     */
    toSampleChannelData() {
        this.to('iotanalytics:SampleChannelData');
        return this;
    }
    /**
     * Starts reprocessing for the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_StartPipelineReprocessing.html
     */
    toStartPipelineReprocessing() {
        this.to('iotanalytics:StartPipelineReprocessing');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('iotanalytics:TagResource');
        return this;
    }
    /**
     * Removes the given tags (metadata) from the resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('iotanalytics:UntagResource');
        return this;
    }
    /**
     * Updates the specified channel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        this.to('iotanalytics:UpdateChannel');
        return this;
    }
    /**
     * Updates the specified dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateDataset.html
     */
    toUpdateDataset() {
        this.to('iotanalytics:UpdateDataset');
        return this;
    }
    /**
     * Updates the specified datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdateDatastore.html
     */
    toUpdateDatastore() {
        this.to('iotanalytics:UpdateDatastore');
        return this;
    }
    /**
     * Updates the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_UpdatePipeline.html
     */
    toUpdatePipeline() {
        this.to('iotanalytics:UpdatePipeline');
        return this;
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onChannel(channelName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:channel/${ChannelName}';
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onDataset(datasetName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:dataset/${DatasetName}';
        arn = arn.replace('${DatasetName}', datasetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type datastore to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param datastoreName - Identifier for the datastoreName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onDatastore(datastoreName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:datastore/${DatastoreName}';
        arn = arn.replace('${DatastoreName}', datastoreName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/iotanalytics/latest/userguide/welcome.html#aws-iot-analytics-how
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     */
    onPipeline(pipelineName, account, region, partition) {
        var arn = 'arn:${Partition}:iotanalytics:${Region}:${Account}:pipeline/${PipelineName}';
        arn = arn.replace('${PipelineName}', pipelineName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The preface string for a tag key and value pair attached to an IoT Analytics resource.
     *
     * Applies to resource types:
     * - channel
     * - dataset
     * - datastore
     * - pipeline
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`iotanalytics:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Iotanalytics = Iotanalytics;
//# sourceMappingURL=data:application/json;base64,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