"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotthingsgraph = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotthingsgraph](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotthingsgraph.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotthingsgraph extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotthingsgraph](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotthingsgraph.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotthingsgraph';
        this.accessLevelList = {
            "Write": [
                "AssociateEntityToThing",
                "CreateFlowTemplate",
                "CreateSystemTemplate",
                "DeleteFlowTemplate",
                "DeleteNamespace",
                "DeleteSystemInstance",
                "DeleteSystemTemplate",
                "DeploySystemInstance",
                "DeprecateFlowTemplate",
                "DeprecateSystemTemplate",
                "DissociateEntityFromThing",
                "UndeploySystemInstance",
                "UpdateFlowTemplate",
                "UpdateSystemTemplate",
                "UploadEntityDefinitions"
            ],
            "Tagging": [
                "CreateSystemInstance",
                "TagResource",
                "UntagResource"
            ],
            "Read": [
                "DescribeNamespace",
                "GetEntities",
                "GetFlowTemplate",
                "GetFlowTemplateRevisions",
                "GetNamespaceDeletionStatus",
                "GetSystemInstance",
                "GetSystemTemplate",
                "GetSystemTemplateRevisions",
                "GetUploadStatus",
                "SearchEntities",
                "SearchFlowExecutions",
                "SearchFlowTemplates",
                "SearchSystemInstances",
                "SearchSystemTemplates",
                "SearchThings"
            ],
            "List": [
                "ListFlowExecutionMessages",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Associates a device with a concrete thing that is in the user's registry. A thing can be associated with only one device at a time. If you associate a thing with a new device id, its previous association will be removed.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_AssociateEntityToThing.html
     */
    toAssociateEntityToThing() {
        this.to('iotthingsgraph:AssociateEntityToThing');
        return this;
    }
    /**
     * Creates a workflow template. Workflows can be created only in the user's namespace. (The public namespace contains only entities.) The workflow can contain only entities in the specified namespace. The workflow is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateFlowTemplate.html
     */
    toCreateFlowTemplate() {
        this.to('iotthingsgraph:CreateFlowTemplate');
        return this;
    }
    /**
     * Creates an instance of a system with specified configurations and Things.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemInstance.html
     */
    toCreateSystemInstance() {
        this.to('iotthingsgraph:CreateSystemInstance');
        return this;
    }
    /**
     * Creates a system. The system is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemTemplate.html
     */
    toCreateSystemTemplate() {
        this.to('iotthingsgraph:CreateSystemTemplate');
        return this;
    }
    /**
     * Deletes a workflow. Any new system or system instance that contains this workflow will fail to update or deploy. Existing system instances that contain the workflow will continue to run (since they use a snapshot of the workflow taken at the time of deploying the system instance).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteFlowTemplate.html
     */
    toDeleteFlowTemplate() {
        this.to('iotthingsgraph:DeleteFlowTemplate');
        return this;
    }
    /**
     * Deletes the specified namespace. This action deletes all of the entities in the namespace. Delete the systems and flows in the namespace before performing this action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteNamespace.html
     */
    toDeleteNamespace() {
        this.to('iotthingsgraph:DeleteNamespace');
        return this;
    }
    /**
     * Deletes a system instance. Only instances that have never been deployed, or that have been undeployed from the target can be deleted. Users can create a new system instance that has the same ID as a deleted system instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemInstance.html
     */
    toDeleteSystemInstance() {
        this.to('iotthingsgraph:DeleteSystemInstance');
        return this;
    }
    /**
     * Deletes a system. New system instances can't contain the system after its deletion. Existing system instances that contain the system will continue to work because they use a snapshot of the system that is taken when it is deployed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemTemplate.html
     */
    toDeleteSystemTemplate() {
        this.to('iotthingsgraph:DeleteSystemTemplate');
        return this;
    }
    /**
     * Deploys the system instance to the target specified in CreateSystemInstance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeploySystemInstance.html
     */
    toDeploySystemInstance() {
        this.to('iotthingsgraph:DeploySystemInstance');
        return this;
    }
    /**
     * Deprecates the specified workflow. This action marks the workflow for deletion. Deprecated flows can't be deployed, but existing system instances that use the flow will continue to run.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateFlowTemplate.html
     */
    toDeprecateFlowTemplate() {
        this.to('iotthingsgraph:DeprecateFlowTemplate');
        return this;
    }
    /**
     * Deprecates the specified system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateSystemTemplate.html
     */
    toDeprecateSystemTemplate() {
        this.to('iotthingsgraph:DeprecateSystemTemplate');
        return this;
    }
    /**
     * Gets the latest version of the user's namespace and the public version that it is tracking.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DescribeNamespace.html
     */
    toDescribeNamespace() {
        this.to('iotthingsgraph:DescribeNamespace');
        return this;
    }
    /**
     * Dissociates a device entity from a concrete thing. The action takes only the type of the entity that you need to dissociate because only one entity of a particular type can be associated with a thing.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DissociateEntityFromThing.html
     */
    toDissociateEntityFromThing() {
        this.to('iotthingsgraph:DissociateEntityFromThing');
        return this;
    }
    /**
     * Gets descriptions of the specified entities. Uses the latest version of the user's namespace by default.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetEntities.html
     */
    toGetEntities() {
        this.to('iotthingsgraph:GetEntities');
        return this;
    }
    /**
     * Gets the latest version of the DefinitionDocument and FlowTemplateSummary for the specified workflow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplate.html
     */
    toGetFlowTemplate() {
        this.to('iotthingsgraph:GetFlowTemplate');
        return this;
    }
    /**
     * Gets revisions of the specified workflow. Only the last 100 revisions are stored. If the workflow has been deprecated, this action will return revisions that occurred before the deprecation. This action won't work for workflows that have been deleted.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplateRevisions.html
     */
    toGetFlowTemplateRevisions() {
        this.to('iotthingsgraph:GetFlowTemplateRevisions');
        return this;
    }
    /**
     * Gets the status of a namespace deletion task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetNamespaceDeletionStatus.html
     */
    toGetNamespaceDeletionStatus() {
        this.to('iotthingsgraph:GetNamespaceDeletionStatus');
        return this;
    }
    /**
     * Gets a system instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemInstance.html
     */
    toGetSystemInstance() {
        this.to('iotthingsgraph:GetSystemInstance');
        return this;
    }
    /**
     * Gets a system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplate.html
     */
    toGetSystemTemplate() {
        this.to('iotthingsgraph:GetSystemTemplate');
        return this;
    }
    /**
     * Gets revisions made to the specified system template. Only the previous 100 revisions are stored. If the system has been deprecated, this action will return the revisions that occurred before its deprecation. This action won't work with systems that have been deleted.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplateRevisions.html
     */
    toGetSystemTemplateRevisions() {
        this.to('iotthingsgraph:GetSystemTemplateRevisions');
        return this;
    }
    /**
     * Gets the status of the specified upload.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetUploadStatus.html
     */
    toGetUploadStatus() {
        this.to('iotthingsgraph:GetUploadStatus');
        return this;
    }
    /**
     * Lists details of a single workflow execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListFlowExecutionMessages.html
     */
    toListFlowExecutionMessages() {
        this.to('iotthingsgraph:ListFlowExecutionMessages');
        return this;
    }
    /**
     * Lists all tags for a given resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('iotthingsgraph:ListTagsForResource');
        return this;
    }
    /**
     * Searches for entities of the specified type. You can search for entities in your namespace and the public namespace that you're tracking.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchEntities.html
     */
    toSearchEntities() {
        this.to('iotthingsgraph:SearchEntities');
        return this;
    }
    /**
     * Searches for workflow executions of a system instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowExecutions.html
     */
    toSearchFlowExecutions() {
        this.to('iotthingsgraph:SearchFlowExecutions');
        return this;
    }
    /**
     * Searches for summary information about workflows.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowTemplates.html
     */
    toSearchFlowTemplates() {
        this.to('iotthingsgraph:SearchFlowTemplates');
        return this;
    }
    /**
     * Searches for system instances in the user's account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemInstances.html
     */
    toSearchSystemInstances() {
        this.to('iotthingsgraph:SearchSystemInstances');
        return this;
    }
    /**
     * Searches for summary information about systems in the user's account. You can filter by the ID of a workflow to return only systems that use the specified workflow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemTemplates.html
     */
    toSearchSystemTemplates() {
        this.to('iotthingsgraph:SearchSystemTemplates');
        return this;
    }
    /**
     * Searches for things associated with the specified entity. You can search by both device and device model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchThings.html
     */
    toSearchThings() {
        this.to('iotthingsgraph:SearchThings');
        return this;
    }
    /**
     * Tag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_TagResource.html
     */
    toTagResource() {
        this.to('iotthingsgraph:TagResource');
        return this;
    }
    /**
     * Removes the system instance and associated triggers from the target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UndeploySystemInstance.html
     */
    toUndeploySystemInstance() {
        this.to('iotthingsgraph:UndeploySystemInstance');
        return this;
    }
    /**
     * Untag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UntagResource.html
     */
    toUntagResource() {
        this.to('iotthingsgraph:UntagResource');
        return this;
    }
    /**
     * Updates the specified workflow. All deployed systems and system instances that use the workflow will see the changes in the flow when it is redeployed. The workflow can contain only entities in the specified namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateFlowTemplate.html
     */
    toUpdateFlowTemplate() {
        this.to('iotthingsgraph:UpdateFlowTemplate');
        return this;
    }
    /**
     * Updates the specified system. You don't need to run this action after updating a workflow. Any system instance that uses the system will see the changes in the system when it is redeployed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateSystemTemplate.html
     */
    toUpdateSystemTemplate() {
        this.to('iotthingsgraph:UpdateSystemTemplate');
        return this;
    }
    /**
     * Asynchronously uploads one or more entity definitions to the user's namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UploadEntityDefinitions.html
     */
    toUploadEntityDefinitions() {
        this.to('iotthingsgraph:UploadEntityDefinitions');
        return this;
    }
    /**
     * Adds a resource of type Workflow to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-workflow.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkflow(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:Workflow/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type System to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-system.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSystem(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:System/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type SystemInstance to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-sdc-deployconfig.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSystemInstance(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:Deployment/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Iotthingsgraph = Iotthingsgraph;
//# sourceMappingURL=data:application/json;base64,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