"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awskeymanagementservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awskeymanagementservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kms';
        this.accessLevelList = {
            "Write": [
                "CancelKeyDeletion",
                "ConnectCustomKeyStore",
                "CreateAlias",
                "CreateCustomKeyStore",
                "CreateKey",
                "Decrypt",
                "DeleteAlias",
                "DeleteCustomKeyStore",
                "DeleteImportedKeyMaterial",
                "DisableKey",
                "DisableKeyRotation",
                "DisconnectCustomKeyStore",
                "EnableKey",
                "EnableKeyRotation",
                "Encrypt",
                "GenerateDataKey",
                "GenerateDataKeyPair",
                "GenerateDataKeyPairWithoutPlaintext",
                "GenerateDataKeyWithoutPlaintext",
                "GenerateRandom",
                "ImportKeyMaterial",
                "ReEncryptFrom",
                "ReEncryptTo",
                "ScheduleKeyDeletion",
                "Sign",
                "UpdateAlias",
                "UpdateCustomKeyStore",
                "UpdateKeyDescription",
                "Verify"
            ],
            "Permissions management": [
                "CreateGrant",
                "PutKeyPolicy",
                "RetireGrant",
                "RevokeGrant"
            ],
            "Read": [
                "DescribeCustomKeyStores",
                "DescribeKey",
                "GetKeyPolicy",
                "GetKeyRotationStatus",
                "GetParametersForImport",
                "GetPublicKey",
                "ListResourceTags"
            ],
            "List": [
                "ListAliases",
                "ListGrants",
                "ListKeyPolicies",
                "ListKeys",
                "ListRetirableGrants"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Controls permission to cancel the scheduled deletion of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html
     */
    toCancelKeyDeletion() {
        this.to('kms:CancelKeyDeletion');
        return this;
    }
    /**
     * Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html
     */
    toConnectCustomKeyStore() {
        this.to('kms:ConnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to create an alias for a customer master key (CMK). Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html
     */
    toCreateAlias() {
        this.to('kms:CreateAlias');
        return this;
    }
    /**
     * Controls permission to create a custom key store that is associated with an AWS CloudHSM cluster that you own and manage.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudhsm:DescribeClusters
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html
     */
    toCreateCustomKeyStore() {
        this.to('kms:CreateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to add a grant to a customer master key. You can use grants to add permissions without changing the key policy or IAM policy.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantConstraintType()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html
     */
    toCreateGrant() {
        this.to('kms:CreateGrant');
        return this;
    }
    /**
     * Controls permission to create a customer master key that can be used to protect data keys and other sensitive information.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCustomerMasterKeySpec()
     * - .ifCustomerMasterKeyUsage()
     * - .ifKeyOrigin()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html
     */
    toCreateKey() {
        this.to('kms:CreateKey');
        return this;
    }
    /**
     * Controls permission to decrypt ciphertext that was encrypted under a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html
     */
    toDecrypt() {
        this.to('kms:Decrypt');
        return this;
    }
    /**
     * Controls permission to delete an alias. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        this.to('kms:DeleteAlias');
        return this;
    }
    /**
     * Controls permission to delete a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html
     */
    toDeleteCustomKeyStore() {
        this.to('kms:DeleteCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete cryptographic material that you imported into a customer master key. This action makes the key unusable.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html
     */
    toDeleteImportedKeyMaterial() {
        this.to('kms:DeleteImportedKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view detailed information about custom key stores in the account and region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html
     */
    toDescribeCustomKeyStores() {
        this.to('kms:DescribeCustomKeyStores');
        return this;
    }
    /**
     * Controls permission to view detailed information about a customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html
     */
    toDescribeKey() {
        this.to('kms:DescribeKey');
        return this;
    }
    /**
     * Controls permission to disable a customer master key, which prevents it from being used in cryptographic operations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html
     */
    toDisableKey() {
        this.to('kms:DisableKey');
        return this;
    }
    /**
     * Controls permission to disable automatic rotation of a customer managed customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html
     */
    toDisableKeyRotation() {
        this.to('kms:DisableKeyRotation');
        return this;
    }
    /**
     * Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html
     */
    toDisconnectCustomKeyStore() {
        this.to('kms:DisconnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to change the state of a customer master key (CMK) to enabled. This allows the CMK to be used in cryptographic operations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html
     */
    toEnableKey() {
        this.to('kms:EnableKey');
        return this;
    }
    /**
     * Controls permission to enable automatic rotation of the cryptographic material in a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html
     */
    toEnableKeyRotation() {
        this.to('kms:EnableKeyRotation');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to encrypt data and data keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html
     */
    toEncrypt() {
        this.to('kms:Encrypt');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html
     */
    toGenerateDataKey() {
        this.to('kms:GenerateDataKey');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html
     */
    toGenerateDataKeyPair() {
        this.to('kms:GenerateDataKeyPair');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html
     */
    toGenerateDataKeyPairWithoutPlaintext() {
        this.to('kms:GenerateDataKeyPairWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html
     */
    toGenerateDataKeyWithoutPlaintext() {
        this.to('kms:GenerateDataKeyWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to get a cryptographically secure random byte string from AWS KMS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html
     */
    toGenerateRandom() {
        this.to('kms:GenerateRandom');
        return this;
    }
    /**
     * Controls permission to view the key policy for the specified customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html
     */
    toGetKeyPolicy() {
        this.to('kms:GetKeyPolicy');
        return this;
    }
    /**
     * Controls permission to determine whether automatic key rotation is enabled on the customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html
     */
    toGetKeyRotationStatus() {
        this.to('kms:GetKeyRotationStatus');
        return this;
    }
    /**
     * Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     * - .ifWrappingAlgorithm()
     * - .ifWrappingKeySpec()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html
     */
    toGetParametersForImport() {
        this.to('kms:GetParametersForImport');
        return this;
    }
    /**
     * Controls permission to download the public key of an asymmetric customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html
     */
    toGetPublicKey() {
        this.to('kms:GetPublicKey');
        return this;
    }
    /**
     * Controls permission to import cryptographic material into a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifExpirationModel()
     * - .ifValidTo()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html
     */
    toImportKeyMaterial() {
        this.to('kms:ImportKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        this.to('kms:ListAliases');
        return this;
    }
    /**
     * Controls permission to view all grants for a customer master key.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html
     */
    toListGrants() {
        this.to('kms:ListGrants');
        return this;
    }
    /**
     * Controls permission to view the names of key policies for a customer master key.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html
     */
    toListKeyPolicies() {
        this.to('kms:ListKeyPolicies');
        return this;
    }
    /**
     * Controls permission to view the key ID and Amazon Resource Name (ARN) of all customer master keys in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html
     */
    toListKeys() {
        this.to('kms:ListKeys');
        return this;
    }
    /**
     * Controls permission to view all tags that are attached to a customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html
     */
    toListResourceTags() {
        this.to('kms:ListResourceTags');
        return this;
    }
    /**
     * Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html
     */
    toListRetirableGrants() {
        this.to('kms:ListRetirableGrants');
        return this;
    }
    /**
     * Controls permission to replace the key policy for the specified customer master key.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html
     */
    toPutKeyPolicy() {
        this.to('kms:PutKeyPolicy');
        return this;
    }
    /**
     * Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    toReEncryptFrom() {
        this.to('kms:ReEncryptFrom');
        return this;
    }
    /**
     * Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    toReEncryptTo() {
        this.to('kms:ReEncryptTo');
        return this;
    }
    /**
     * Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html
     */
    toRetireGrant() {
        this.to('kms:RetireGrant');
        return this;
    }
    /**
     * Controls permission to revoke a grant, which denies permission for all operations that depend on the grant.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html
     */
    toRevokeGrant() {
        this.to('kms:RevokeGrant');
        return this;
    }
    /**
     * Controls permission to schedule deletion of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html
     */
    toScheduleKeyDeletion() {
        this.to('kms:ScheduleKeyDeletion');
        return this;
    }
    /**
     * Controls permission to produce a digital signature for a message.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html
     */
    toSign() {
        this.to('kms:Sign');
        return this;
    }
    /**
     * Controls permission to create or update tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('kms:TagResource');
        return this;
    }
    /**
     * Controls permission to delete tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('kms:UntagResource');
        return this;
    }
    /**
     * Controls permission to associate an alias with a different customer master key. An alias is an optional friendly name that you can associate with a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html
     */
    toUpdateAlias() {
        this.to('kms:UpdateAlias');
        return this;
    }
    /**
     * Controls permission to change the properties of a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html
     */
    toUpdateCustomKeyStore() {
        this.to('kms:UpdateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete or change the description of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html
     */
    toUpdateKeyDescription() {
        this.to('kms:UpdateKeyDescription');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to verify digital signatures.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html
     */
    toVerify() {
        this.to('kms:Verify');
        return this;
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/programming-aliases.html
     *
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAlias(alias, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:alias/${Alias}';
        arn = arn.replace('${Alias}', alias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onKey(keyId, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}';
        arn = arn.replace('${KeyId}', keyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Controls access to the CreateKey and PutKeyPolicy operations based on the value of the BypassPolicyLockoutSafetyCheck parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-bypass-policy-lockout-safety-check
     *
     * Applies to actions:
     * - .toCreateKey()
     * - .toPutKeyPolicy()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBypassPolicyLockoutSafetyCheck(value) {
        return this.if(`kms:BypassPolicyLockoutSafetyCheck`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to specified AWS KMS operations based on the AWS account ID of the caller. You can use this condition key to allow or deny access to all IAM users and roles in an AWS account in a single policy statement.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-caller-account
     *
     * Applies to actions:
     * - .toCancelKeyDeletion()
     * - .toCreateAlias()
     * - .toCreateGrant()
     * - .toDecrypt()
     * - .toDeleteAlias()
     * - .toDeleteImportedKeyMaterial()
     * - .toDescribeKey()
     * - .toDisableKey()
     * - .toDisableKeyRotation()
     * - .toEnableKey()
     * - .toEnableKeyRotation()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toGetKeyPolicy()
     * - .toGetKeyRotationStatus()
     * - .toGetParametersForImport()
     * - .toGetPublicKey()
     * - .toImportKeyMaterial()
     * - .toListGrants()
     * - .toListKeyPolicies()
     * - .toListResourceTags()
     * - .toPutKeyPolicy()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toRevokeGrant()
     * - .toScheduleKeyDeletion()
     * - .toSign()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAlias()
     * - .toUpdateKeyDescription()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCallerAccount(value, operator) {
        return this.if(`kms:CallerAccount`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the CustomerMasterKeySpec property of the CMK that is created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-spec
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeySpec(value, operator) {
        return this.if(`kms:CustomerMasterKeySpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the KeyUsage property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-usage
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeyUsage(value, operator) {
        return this.if(`kms:CustomerMasterKeyUsage`, value, operator || 'StringLike');
    }
    /**
     * Controls access to GenerateDataKeyPair and GenerateDataKeyPairWithoutPlaintext operations based on the value of the DataKeyPairSpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-data-key-pair-spec
     *
     * Applies to actions:
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataKeyPairSpec(value, operator) {
        return this.if(`kms:DataKeyPairSpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to encryption operations based on the value of the encryption algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm
     *
     * Applies to actions:
     * - .toDecrypt()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionAlgorithm(value, operator) {
        return this.if(`kms:EncryptionAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access based on the presence of specified keys in the encryption context. The encryption context is an optional element in a cryptographic operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-context-keys
     *
     * Applies to actions:
     * - .toDecrypt()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionContextKeys(value, operator) {
        return this.if(`kms:EncryptionContextKeys`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ExpirationModel parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-expiration-model
     *
     * Applies to actions:
     * - .toImportKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExpirationModel(value, operator) {
        return this.if(`kms:ExpirationModel`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grant constraint in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-constraint-type
     *
     * Applies to actions:
     * - .toCreateGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantConstraintType(value, operator) {
        return this.if(`kms:GrantConstraintType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation when the request comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-is-for-aws-resource
     *
     * Applies to actions:
     * - .toCreateGrant()
     * - .toListGrants()
     * - .toRevokeGrant()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifGrantIsForAWSResource(value) {
        return this.if(`kms:GrantIsForAWSResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the operations in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-operations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantOperations(value, operator) {
        return this.if(`kms:GrantOperations`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grantee principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grantee-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGranteePrincipal(value, operator) {
        return this.if(`kms:GranteePrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the Origin property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-key-origin
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyOrigin(value, operator) {
        return this.if(`kms:KeyOrigin`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the value of the MessageType parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-message-type
     *
     * Applies to actions:
     * - .toSign()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMessageType(value, operator) {
        return this.if(`kms:MessageType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ReEncrypt operation when it uses the same customer master key that was used for the Encrypt operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-reencrypt-on-same-key
     *
     * Applies to actions:
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifReEncryptOnSameKey(value) {
        return this.if(`kms:ReEncryptOnSameKey`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the retiring principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-retiring-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRetiringPrincipal(value, operator) {
        return this.if(`kms:RetiringPrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the signing algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm
     *
     * Applies to actions:
     * - .toSign()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSigningAlgorithm(value, operator) {
        return this.if(`kms:SigningAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ValidTo parameter in the request. You can use this condition key to allow users to import key material only when it expires by the specified date.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-valid-to
     *
     * Applies to actions:
     * - .toImportKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifValidTo(value, operator) {
        return this.if(`kms:ValidTo`, value, operator || 'NumericEquals');
    }
    /**
     * Controls access when a request made on the principal's behalf comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-via-service
     *
     * Applies to actions:
     * - .toCancelKeyDeletion()
     * - .toCreateAlias()
     * - .toCreateGrant()
     * - .toDecrypt()
     * - .toDeleteAlias()
     * - .toDeleteImportedKeyMaterial()
     * - .toDescribeKey()
     * - .toDisableKey()
     * - .toDisableKeyRotation()
     * - .toEnableKey()
     * - .toEnableKeyRotation()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toGetKeyPolicy()
     * - .toGetKeyRotationStatus()
     * - .toGetParametersForImport()
     * - .toGetPublicKey()
     * - .toImportKeyMaterial()
     * - .toListGrants()
     * - .toListKeyPolicies()
     * - .toListResourceTags()
     * - .toPutKeyPolicy()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toRevokeGrant()
     * - .toScheduleKeyDeletion()
     * - .toSign()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAlias()
     * - .toUpdateKeyDescription()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifViaService(value, operator) {
        return this.if(`kms:ViaService`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingAlgorithm parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-algorithm
     *
     * Applies to actions:
     * - .toGetParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingAlgorithm(value, operator) {
        return this.if(`kms:WrappingAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingKeySpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-key-spec
     *
     * Applies to actions:
     * - .toGetParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingKeySpec(value, operator) {
        return this.if(`kms:WrappingKeySpec`, value, operator || 'StringLike');
    }
}
exports.Kms = Kms;
//# sourceMappingURL=data:application/json;base64,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