"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsMarketplace = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [aws-marketplace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplace.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AwsMarketplace extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [aws-marketplace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplace.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'aws-marketplace';
        this.accessLevelList = {
            "Write": [
                "AcceptAgreementApprovalRequest",
                "CancelAgreementRequest",
                "RejectAgreementApprovalRequest",
                "Subscribe",
                "Unsubscribe",
                "UpdateAgreementApprovalRequest"
            ],
            "Read": [
                "DescribeAgreement",
                "GetAgreementApprovalRequest",
                "GetAgreementRequest"
            ],
            "List": [
                "GetAgreementTerms",
                "ListAgreementApprovalRequests",
                "ListAgreementRequests",
                "SearchAgreements",
                "ViewSubscriptions"
            ]
        };
    }
    /**
     * Allows users to approve an incoming subscription request (for providers who provide products that require subscription verification).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toAcceptAgreementApprovalRequest() {
        this.to('aws-marketplace:AcceptAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to cancel pending subscription requests for products that require subscription verification.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toCancelAgreementRequest() {
        this.to('aws-marketplace:CancelAgreementRequest');
        return this;
    }
    /**
     * Returns metadata about the agreement.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toDescribeAgreement() {
        this.to('aws-marketplace:DescribeAgreement');
        return this;
    }
    /**
     * Allows users to view the details of their incoming subscription requests (for providers who provide products that require subscription verification).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toGetAgreementApprovalRequest() {
        this.to('aws-marketplace:GetAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to view the details of their subscription requests for data products that require subscription verification.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toGetAgreementRequest() {
        this.to('aws-marketplace:GetAgreementRequest');
        return this;
    }
    /**
     * Returns a list of terms for an agreement.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toGetAgreementTerms() {
        this.to('aws-marketplace:GetAgreementTerms');
        return this;
    }
    /**
     * Allows users to list their incoming subscription requests (for providers who provide products that require subscription verification).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toListAgreementApprovalRequests() {
        this.to('aws-marketplace:ListAgreementApprovalRequests');
        return this;
    }
    /**
     * Allows users to list their subscription requests for products that require subscription verification.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toListAgreementRequests() {
        this.to('aws-marketplace:ListAgreementRequests');
        return this;
    }
    /**
     * Allows users to decline an incoming subscription requests (for providers who provide products that require subscription verification).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toRejectAgreementApprovalRequest() {
        this.to('aws-marketplace:RejectAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to search their agreements.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toSearchAgreements() {
        this.to('aws-marketplace:SearchAgreements');
        return this;
    }
    /**
     * Allows users to subscribe to AWS Marketplace products. Includes the ability to send a subscription request for products that require subscription verification. Includes the ability to enable auto-renewal for an existing subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toSubscribe() {
        this.to('aws-marketplace:Subscribe');
        return this;
    }
    /**
     * Allows users to remove subscriptions to AWS Marketplace products. Includes the ability to disable auto-renewal for an existing subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toUnsubscribe() {
        this.to('aws-marketplace:Unsubscribe');
        return this;
    }
    /**
     * Allows users to make changes to an incoming subscription request, including the ability to delete the prospective subscriber's information (for providers who provide products that require subscription verification).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toUpdateAgreementApprovalRequest() {
        this.to('aws-marketplace:UpdateAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to see their account's subscriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    toViewSubscriptions() {
        this.to('aws-marketplace:ViewSubscriptions');
        return this;
    }
    /**
     * Enables you to control access based on the type of the agreement.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAgreementType(value, operator) {
        return this.if(`aws-marketplace:AgreementType`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the party type of the agreement.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPartyType(value, operator) {
        return this.if(`aws-marketplace:PartyType`, value, operator || 'StringLike');
    }
}
exports.AwsMarketplace = AwsMarketplace;
//# sourceMappingURL=data:application/json;base64,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