"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsMarketplaceCatalog = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [aws-marketplace-catalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacecatalog.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AwsMarketplaceCatalog extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [aws-marketplace-catalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacecatalog.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'aws-marketplace-catalog';
        this.accessLevelList = {
            "Write": [
                "CancelChangeSet",
                "CompleteTask",
                "StartChangeSet",
                "UpdateTask"
            ],
            "Read": [
                "DescribeChangeSet",
                "DescribeEntity",
                "DescribeTask",
                "ListChangeSets",
                "ListEntities"
            ],
            "List": [
                "ListTasks"
            ]
        };
    }
    /**
     * Cancels a running change set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_CancelChangeSet.html
     */
    toCancelChangeSet() {
        this.to('aws-marketplace-catalog:CancelChangeSet');
        return this;
    }
    /**
     * Complete an existing task and submit the content to the associated change.
     *
     * Access Level: Write
     */
    toCompleteTask() {
        this.to('aws-marketplace-catalog:CompleteTask');
        return this;
    }
    /**
     * Returns the details of an existing change set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeChangeSet.html
     */
    toDescribeChangeSet() {
        this.to('aws-marketplace-catalog:DescribeChangeSet');
        return this;
    }
    /**
     * Returns the details of an existing entity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeEntity.html
     */
    toDescribeEntity() {
        this.to('aws-marketplace-catalog:DescribeEntity');
        return this;
    }
    /**
     * Returns the details of an existing task.
     *
     * Access Level: Read
     */
    toDescribeTask() {
        this.to('aws-marketplace-catalog:DescribeTask');
        return this;
    }
    /**
     * Lists existing change sets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListChangeSets.html
     */
    toListChangeSets() {
        this.to('aws-marketplace-catalog:ListChangeSets');
        return this;
    }
    /**
     * Lists existing entities.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListEntities.html
     */
    toListEntities() {
        this.to('aws-marketplace-catalog:ListEntities');
        return this;
    }
    /**
     * Lists existing tasks.
     *
     * Access Level: List
     */
    toListTasks() {
        this.to('aws-marketplace-catalog:ListTasks');
        return this;
    }
    /**
     * Requests a new change set.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeType()
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_StartChangeSet.html
     */
    toStartChangeSet() {
        this.to('aws-marketplace-catalog:StartChangeSet');
        return this;
    }
    /**
     * Update the content of an existing task.
     *
     * Access Level: Write
     */
    toUpdateTask() {
        this.to('aws-marketplace-catalog:UpdateTask');
        return this;
    }
    /**
     * Adds a resource of type Entity to the statement
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_DescribeEntity.html#API_DescribeEntity_ResponseSyntax
     *
     * @param catalog - Identifier for the catalog.
     * @param entityType - Identifier for the entityType.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEntity(catalog, entityType, resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/${EntityType}/${ResourceId}';
        arn = arn.replace('${Catalog}', catalog);
        arn = arn.replace('${EntityType}', entityType);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ChangeSet to the statement
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_StartChangeSet.html#API_StartChangeSet_ResponseSyntax
     *
     * @param catalog - Identifier for the catalog.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChangeSet(catalog, resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/ChangeSet/${ResourceId}';
        arn = arn.replace('${Catalog}', catalog);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to verify change type in the StartChangeSet request.
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/api-access-control.html
     *
     * Applies to actions:
     * - .toStartChangeSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeType(value, operator) {
        return this.if(`catalog:ChangeType`, value, operator || 'StringLike');
    }
}
exports.AwsMarketplaceCatalog = AwsMarketplaceCatalog;
//# sourceMappingURL=data:application/json;base64,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