"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mobilehub = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mobilehub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmobilehub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mobilehub extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mobilehub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmobilehub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mobilehub';
        this.accessLevelList = {
            "Write": [
                "CreateProject",
                "CreateServiceRole",
                "DeleteProject",
                "DeleteProjectSnapshot",
                "DeployToStage",
                "GenerateProjectParameters",
                "ImportProject",
                "InstallBundle",
                "SynchronizeProject",
                "UpdateProject"
            ],
            "Read": [
                "DescribeBundle",
                "ExportBundle",
                "ExportProject",
                "GetProject",
                "GetProjectSnapshot",
                "ValidateProject",
                "VerifyServiceRole"
            ],
            "List": [
                "ListAvailableConnectors",
                "ListAvailableFeatures",
                "ListAvailableRegions",
                "ListBundles",
                "ListProjectSnapshots",
                "ListProjects"
            ]
        };
    }
    /**
     * Create a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toCreateProject() {
        this.to('mobilehub:CreateProject');
        return this;
    }
    /**
     * Enable AWS Mobile Hub in the account by creating the required service role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toCreateServiceRole() {
        this.to('mobilehub:CreateServiceRole');
        return this;
    }
    /**
     * Delete the specified project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDeleteProject() {
        this.to('mobilehub:DeleteProject');
        return this;
    }
    /**
     * Delete a saved snapshot of project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDeleteProjectSnapshot() {
        this.to('mobilehub:DeleteProjectSnapshot');
        return this;
    }
    /**
     * Deploy changes to the specified stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDeployToStage() {
        this.to('mobilehub:DeployToStage');
        return this;
    }
    /**
     * Describe the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDescribeBundle() {
        this.to('mobilehub:DescribeBundle');
        return this;
    }
    /**
     * Export the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toExportBundle() {
        this.to('mobilehub:ExportBundle');
        return this;
    }
    /**
     * Export the project configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toExportProject() {
        this.to('mobilehub:ExportProject');
        return this;
    }
    /**
     * Generate project parameters required for code generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toGenerateProjectParameters() {
        this.to('mobilehub:GenerateProjectParameters');
        return this;
    }
    /**
     * Get project configuration and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toGetProject() {
        this.to('mobilehub:GetProject');
        return this;
    }
    /**
     * Fetch the previously exported project configuration snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toGetProjectSnapshot() {
        this.to('mobilehub:GetProjectSnapshot');
        return this;
    }
    /**
     * Create a new project from the previously exported project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toImportProject() {
        this.to('mobilehub:ImportProject');
        return this;
    }
    /**
     * Install a bundle in the project deployments S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toInstallBundle() {
        this.to('mobilehub:InstallBundle');
        return this;
    }
    /**
     * List the available SaaS (Software as a Service) connectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListAvailableConnectors() {
        this.to('mobilehub:ListAvailableConnectors');
        return this;
    }
    /**
     * List available features
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListAvailableFeatures() {
        this.to('mobilehub:ListAvailableFeatures');
        return this;
    }
    /**
     * List available regions for projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListAvailableRegions() {
        this.to('mobilehub:ListAvailableRegions');
        return this;
    }
    /**
     * List the available download bundles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListBundles() {
        this.to('mobilehub:ListBundles');
        return this;
    }
    /**
     * List saved snapshots of project configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListProjectSnapshots() {
        this.to('mobilehub:ListProjectSnapshots');
        return this;
    }
    /**
     * List projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListProjects() {
        this.to('mobilehub:ListProjects');
        return this;
    }
    /**
     * Synchronize state of resources into project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toSynchronizeProject() {
        this.to('mobilehub:SynchronizeProject');
        return this;
    }
    /**
     * Update project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toUpdateProject() {
        this.to('mobilehub:UpdateProject');
        return this;
    }
    /**
     * Validate a mobile hub project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toValidateProject() {
        this.to('mobilehub:ValidateProject');
        return this;
    }
    /**
     * Verify AWS Mobile Hub is enabled in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toVerifyServiceRole() {
        this.to('mobilehub:VerifyServiceRole');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/reference-mobile-hub-iam-managed-policies.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectId, account, region, partition) {
        var arn = 'arn:${Partition}:mobilehub:${Region}:${Account}:project/${ProjectId}';
        arn = arn.replace('${ProjectId}', projectId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mobilehub = Mobilehub;
//# sourceMappingURL=data:application/json;base64,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