"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceGroups = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [resource-groups](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourcegroups.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ResourceGroups extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [resource-groups](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourcegroups.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'resource-groups';
        this.accessLevelList = {
            "Write": [
                "CreateGroup",
                "DeleteGroup",
                "UpdateGroup",
                "UpdateGroupQuery"
            ],
            "Read": [
                "GetGroup",
                "GetGroupQuery",
                "GetTags"
            ],
            "List": [
                "ListGroupResources",
                "ListGroups",
                "SearchResources"
            ],
            "Tagging": [
                "Tag",
                "Untag"
            ]
        };
    }
    /**
     * Creates a group with a specified name, description, and resource query.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        this.to('resource-groups:CreateGroup');
        return this;
    }
    /**
     * Deletes a specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.to('resource-groups:DeleteGroup');
        return this;
    }
    /**
     * Gets information of a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroup.html
     */
    toGetGroup() {
        this.to('resource-groups:GetGroup');
        return this;
    }
    /**
     * Gets the query associated with a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroupQuery.html
     */
    toGetGroupQuery() {
        this.to('resource-groups:GetGroupQuery');
        return this;
    }
    /**
     * Gets the tags associated with a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetTags.html
     */
    toGetTags() {
        this.to('resource-groups:GetTags');
        return this;
    }
    /**
     * Lists the resources that are member of a specified resource group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroupResources.html
     */
    toListGroupResources() {
        this.to('resource-groups:ListGroupResources');
        return this;
    }
    /**
     * Lists all resource groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        this.to('resource-groups:ListGroups');
        return this;
    }
    /**
     * Returns a list of AWS resource identifiers matching the given query
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_SearchResources.html
     */
    toSearchResources() {
        this.to('resource-groups:SearchResources');
        return this;
    }
    /**
     * Tags a specified resource group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_Tag.html
     */
    toTag() {
        this.to('resource-groups:Tag');
        return this;
    }
    /**
     * Removes tags associated with a specified resource group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_Untag.html
     */
    toUntag() {
        this.to('resource-groups:Untag');
        return this;
    }
    /**
     * Updates a specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        this.to('resource-groups:UpdateGroup');
        return this;
    }
    /**
     * Updates the query associated with a specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroupQuery.html
     */
    toUpdateGroupQuery() {
        this.to('resource-groups:UpdateGroupQuery');
        return this;
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/ARG/latest/userguide/welcome.html
     *
     * @param groupName - Identifier for the groupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupName, account, region, partition) {
        var arn = 'arn:${Partition}:resource-groups:${Region}:${Account}:group/${GroupName}';
        arn = arn.replace('${GroupName}', groupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.ResourceGroups = ResourceGroups;
//# sourceMappingURL=data:application/json;base64,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