"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicecatalog = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservicecatalog.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicecatalog extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservicecatalog.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicecatalog';
        this.accessLevelList = {
            "Write": [
                "AcceptPortfolioShare",
                "AssociateAttributeGroup",
                "AssociateBudgetWithResource",
                "AssociatePrincipalWithPortfolio",
                "AssociateProductWithPortfolio",
                "AssociateResource",
                "AssociateServiceActionWithProvisioningArtifact",
                "AssociateTagOptionWithResource",
                "BatchAssociateServiceActionWithProvisioningArtifact",
                "BatchDisassociateServiceActionFromProvisioningArtifact",
                "CopyProduct",
                "CreateApplication",
                "CreateAttributeGroup",
                "CreateConstraint",
                "CreatePortfolio",
                "CreateProduct",
                "CreateProvisionedProductPlan",
                "CreateProvisioningArtifact",
                "CreateServiceAction",
                "CreateTagOption",
                "DeleteApplication",
                "DeleteAttributeGroup",
                "DeleteConstraint",
                "DeletePortfolio",
                "DeleteProduct",
                "DeleteProvisionedProductPlan",
                "DeleteProvisioningArtifact",
                "DeleteServiceAction",
                "DeleteTagOption",
                "DisableAWSOrganizationsAccess",
                "DisassociateAttributeGroup",
                "DisassociateBudgetFromResource",
                "DisassociatePrincipalFromPortfolio",
                "DisassociateProductFromPortfolio",
                "DisassociateResource",
                "DisassociateServiceActionFromProvisioningArtifact",
                "DisassociateTagOptionFromResource",
                "EnableAWSOrganizationsAccess",
                "ExecuteProvisionedProductPlan",
                "ExecuteProvisionedProductServiceAction",
                "ImportAsProvisionedProduct",
                "ProvisionProduct",
                "RejectPortfolioShare",
                "TerminateProvisionedProduct",
                "UpdateApplication",
                "UpdateAttributeGroup",
                "UpdateConstraint",
                "UpdatePortfolio",
                "UpdateProduct",
                "UpdateProvisionedProduct",
                "UpdateProvisionedProductProperties",
                "UpdateProvisioningArtifact",
                "UpdateServiceAction",
                "UpdateTagOption"
            ],
            "Permissions management": [
                "CreatePortfolioShare",
                "DeletePortfolioShare"
            ],
            "Read": [
                "DescribeConstraint",
                "DescribeCopyProductStatus",
                "DescribePortfolio",
                "DescribePortfolioShareStatus",
                "DescribeProduct",
                "DescribeProductAsAdmin",
                "DescribeProductView",
                "DescribeProvisionedProduct",
                "DescribeProvisionedProductPlan",
                "DescribeProvisioningArtifact",
                "DescribeProvisioningParameters",
                "DescribeRecord",
                "DescribeServiceAction",
                "DescribeServiceActionExecutionParameters",
                "DescribeTagOption",
                "GetAWSOrganizationsAccessStatus",
                "GetApplication",
                "GetAttributeGroup"
            ],
            "List": [
                "ListAcceptedPortfolioShares",
                "ListApplications",
                "ListAssociatedAttributeGroups",
                "ListAssociatedResources",
                "ListAttributeGroups",
                "ListBudgetsForResource",
                "ListConstraintsForPortfolio",
                "ListLaunchPaths",
                "ListOrganizationPortfolioAccess",
                "ListPortfolioAccess",
                "ListPortfolios",
                "ListPortfoliosForProduct",
                "ListPrincipalsForPortfolio",
                "ListProvisionedProductPlans",
                "ListProvisioningArtifacts",
                "ListProvisioningArtifactsForServiceAction",
                "ListRecordHistory",
                "ListResourcesForTagOption",
                "ListServiceActions",
                "ListServiceActionsForProvisioningArtifact",
                "ListStackInstancesForProvisionedProduct",
                "ListTagOptions",
                "ScanProvisionedProducts",
                "SearchProducts",
                "SearchProductsAsAdmin",
                "SearchProvisionedProducts"
            ]
        };
    }
    /**
     * Grants permission to accept a portfolio that has been shared with you
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AcceptPortfolioShare.html
     */
    toAcceptPortfolioShare() {
        this.to('servicecatalog:AcceptPortfolioShare');
        return this;
    }
    /**
     * Grants permission to associate an attribute group with an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateAttributeGroup.html
     */
    toAssociateAttributeGroup() {
        this.to('servicecatalog:AssociateAttributeGroup');
        return this;
    }
    /**
     * Grants permission to associate a budget with a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateBudgetWithResource.html
     */
    toAssociateBudgetWithResource() {
        this.to('servicecatalog:AssociateBudgetWithResource');
        return this;
    }
    /**
     * Grants permission to associate an IAM principal with a portfolio, giving the specified principal access to any products associated with the specified portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociatePrincipalWithPortfolio.html
     */
    toAssociatePrincipalWithPortfolio() {
        this.to('servicecatalog:AssociatePrincipalWithPortfolio');
        return this;
    }
    /**
     * Grants permission to associate a product with a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateProductWithPortfolio.html
     */
    toAssociateProductWithPortfolio() {
        this.to('servicecatalog:AssociateProductWithPortfolio');
        return this;
    }
    /**
     * Grants permission to associate a resource with an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateResource.html
     */
    toAssociateResource() {
        this.to('servicecatalog:AssociateResource');
        return this;
    }
    /**
     * Grants permission to associate an action with a provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateServiceActionWithProvisioningArtifact.html
     */
    toAssociateServiceActionWithProvisioningArtifact() {
        this.to('servicecatalog:AssociateServiceActionWithProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to associate the specified TagOption with the specified portfolio or product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateTagOptionWithResource.html
     */
    toAssociateTagOptionWithResource() {
        this.to('servicecatalog:AssociateTagOptionWithResource');
        return this;
    }
    /**
     * Grants permission to associate multiple self-service actions with provisioning artifacts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchAssociateServiceActionWithProvisioningArtifact.html
     */
    toBatchAssociateServiceActionWithProvisioningArtifact() {
        this.to('servicecatalog:BatchAssociateServiceActionWithProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to disassociate a batch of self-service actions from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchDisassociateServiceActionFromProvisioningArtifact.html
     */
    toBatchDisassociateServiceActionFromProvisioningArtifact() {
        this.to('servicecatalog:BatchDisassociateServiceActionFromProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to copy the specified source product to the specified target product or a new product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CopyProduct.html
     */
    toCopyProduct() {
        this.to('servicecatalog:CopyProduct');
        return this;
    }
    /**
     * Grants permission to create an application.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateApplication.html
     */
    toCreateApplication() {
        this.to('servicecatalog:CreateApplication');
        return this;
    }
    /**
     * Grants permission to create an attribute group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateAttributeGroup.html
     */
    toCreateAttributeGroup() {
        this.to('servicecatalog:CreateAttributeGroup');
        return this;
    }
    /**
     * Grants permission to create a constraint on an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateConstraint.html
     */
    toCreateConstraint() {
        this.to('servicecatalog:CreateConstraint');
        return this;
    }
    /**
     * Grants permission to create a portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolio.html
     */
    toCreatePortfolio() {
        this.to('servicecatalog:CreatePortfolio');
        return this;
    }
    /**
     * Grants permission to share a portfolio you own with another AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolioShare.html
     */
    toCreatePortfolioShare() {
        this.to('servicecatalog:CreatePortfolioShare');
        return this;
    }
    /**
     * Grants permission to create a product and that product's first provisioning artifact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProduct.html
     */
    toCreateProduct() {
        this.to('servicecatalog:CreateProduct');
        return this;
    }
    /**
     * Grants permission to add a new provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisionedProductPlan.html
     */
    toCreateProvisionedProductPlan() {
        this.to('servicecatalog:CreateProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to add a new provisioning artifact to an existing product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisioningArtifact.html
     */
    toCreateProvisioningArtifact() {
        this.to('servicecatalog:CreateProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to create a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateServiceAction.html
     */
    toCreateServiceAction() {
        this.to('servicecatalog:CreateServiceAction');
        return this;
    }
    /**
     * Grants permission to create a TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateTagOption.html
     */
    toCreateTagOption() {
        this.to('servicecatalog:CreateTagOption');
        return this;
    }
    /**
     * Grants permission to delete an application if all associations have been removed from the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteApplication.html
     */
    toDeleteApplication() {
        this.to('servicecatalog:DeleteApplication');
        return this;
    }
    /**
     * Grants permission to delete an attribute group if all associations have been removed from the attribute group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteAttributeGroup.html
     */
    toDeleteAttributeGroup() {
        this.to('servicecatalog:DeleteAttributeGroup');
        return this;
    }
    /**
     * Grants permission to remove and delete an existing constraint from an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteConstraint.html
     */
    toDeleteConstraint() {
        this.to('servicecatalog:DeleteConstraint');
        return this;
    }
    /**
     * Grants permission to delete a portfolio if all associations and shares have been removed from the portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolio.html
     */
    toDeletePortfolio() {
        this.to('servicecatalog:DeletePortfolio');
        return this;
    }
    /**
     * Grants permission to unshare a portfolio you own from an AWS account you previously shared the portfolio with
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolioShare.html
     */
    toDeletePortfolioShare() {
        this.to('servicecatalog:DeletePortfolioShare');
        return this;
    }
    /**
     * Grants permission to delete a product if all associations have been removed from the product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProduct.html
     */
    toDeleteProduct() {
        this.to('servicecatalog:DeleteProduct');
        return this;
    }
    /**
     * Grants permission to delete a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisionedProductPlan.html
     */
    toDeleteProvisionedProductPlan() {
        this.to('servicecatalog:DeleteProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to delete a provisioning artifact from a product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisioningArtifact.html
     */
    toDeleteProvisioningArtifact() {
        this.to('servicecatalog:DeleteProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to delete a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteServiceAction.html
     */
    toDeleteServiceAction() {
        this.to('servicecatalog:DeleteServiceAction');
        return this;
    }
    /**
     * Grants permission to delete the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteTagOption.html
     */
    toDeleteTagOption() {
        this.to('servicecatalog:DeleteTagOption');
        return this;
    }
    /**
     * Grants permission to describe a constraint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeConstraint.html
     */
    toDescribeConstraint() {
        this.to('servicecatalog:DescribeConstraint');
        return this;
    }
    /**
     * Grants permission to get the status of the specified copy product operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeCopyProductStatus.html
     */
    toDescribeCopyProductStatus() {
        this.to('servicecatalog:DescribeCopyProductStatus');
        return this;
    }
    /**
     * Grants permission to describe a portfolio
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolio.html
     */
    toDescribePortfolio() {
        this.to('servicecatalog:DescribePortfolio');
        return this;
    }
    /**
     * Grants permission to get the status of the specified portfolio share operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolioShareStatus.html
     */
    toDescribePortfolioShareStatus() {
        this.to('servicecatalog:DescribePortfolioShareStatus');
        return this;
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProduct.html
     */
    toDescribeProduct() {
        this.to('servicecatalog:DescribeProduct');
        return this;
    }
    /**
     * Grants permission to describe a product as an admin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductAsAdmin.html
     */
    toDescribeProductAsAdmin() {
        this.to('servicecatalog:DescribeProductAsAdmin');
        return this;
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductView.html
     */
    toDescribeProductView() {
        this.to('servicecatalog:DescribeProductView');
        return this;
    }
    /**
     * Grants permission to describe a provisioned product
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProduct.html
     */
    toDescribeProvisionedProduct() {
        this.to('servicecatalog:DescribeProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to describe a provisioned product plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProductPlan.html
     */
    toDescribeProvisionedProductPlan() {
        this.to('servicecatalog:DescribeProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to describe a provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningArtifact.html
     */
    toDescribeProvisioningArtifact() {
        this.to('servicecatalog:DescribeProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to describe the parameters that you need to specify to successfully provision a specified provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningParameters.html
     */
    toDescribeProvisioningParameters() {
        this.to('servicecatalog:DescribeProvisioningParameters');
        return this;
    }
    /**
     * Grants permission to describe a record and lists any outputs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeRecord.html
     */
    toDescribeRecord() {
        this.to('servicecatalog:DescribeRecord');
        return this;
    }
    /**
     * Grants permission to describe a self-service action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceAction.html
     */
    toDescribeServiceAction() {
        this.to('servicecatalog:DescribeServiceAction');
        return this;
    }
    /**
     * Grants permission to get the default parameters if you executed the specified Service Action on the specified Provisioned Product
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceActionExecutionParameters.html
     */
    toDescribeServiceActionExecutionParameters() {
        this.to('servicecatalog:DescribeServiceActionExecutionParameters');
        return this;
    }
    /**
     * Grants permission to get information about the specified TagOption
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeTagOption.html
     */
    toDescribeTagOption() {
        this.to('servicecatalog:DescribeTagOption');
        return this;
    }
    /**
     * Grants permission to disable portfolio sharing through AWS Organizations feature
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisableAWSOrganizationsAccess.html
     */
    toDisableAWSOrganizationsAccess() {
        this.to('servicecatalog:DisableAWSOrganizationsAccess');
        return this;
    }
    /**
     * Grants permission to disassociate an attribute group from an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateAttributeGroup.html
     */
    toDisassociateAttributeGroup() {
        this.to('servicecatalog:DisassociateAttributeGroup');
        return this;
    }
    /**
     * Grants permission to disassociate a budget from a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateBudgetFromResource.html
     */
    toDisassociateBudgetFromResource() {
        this.to('servicecatalog:DisassociateBudgetFromResource');
        return this;
    }
    /**
     * Grants permission to disassociate an IAM principal from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociatePrincipalFromPortfolio.html
     */
    toDisassociatePrincipalFromPortfolio() {
        this.to('servicecatalog:DisassociatePrincipalFromPortfolio');
        return this;
    }
    /**
     * Grants permission to disassociate a product from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateProductFromPortfolio.html
     */
    toDisassociateProductFromPortfolio() {
        this.to('servicecatalog:DisassociateProductFromPortfolio');
        return this;
    }
    /**
     * Grants permission to disassociate a resource from an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateResource.html
     */
    toDisassociateResource() {
        this.to('servicecatalog:DisassociateResource');
        return this;
    }
    /**
     * Grants permission to disassociate the specified self-service action association from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateServiceActionFromProvisioningArtifact.html
     */
    toDisassociateServiceActionFromProvisioningArtifact() {
        this.to('servicecatalog:DisassociateServiceActionFromProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to disassociate the specified TagOption from the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateTagOptionFromResource.html
     */
    toDisassociateTagOptionFromResource() {
        this.to('servicecatalog:DisassociateTagOptionFromResource');
        return this;
    }
    /**
     * Grants permission to enable portfolio sharing feature through AWS Organizations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_EnableAWSOrganizationsAccess.html
     */
    toEnableAWSOrganizationsAccess() {
        this.to('servicecatalog:EnableAWSOrganizationsAccess');
        return this;
    }
    /**
     * Grants permission to execute a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductPlan.html
     */
    toExecuteProvisionedProductPlan() {
        this.to('servicecatalog:ExecuteProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to executes a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductServiceAction.html
     */
    toExecuteProvisionedProductServiceAction() {
        this.to('servicecatalog:ExecuteProvisionedProductServiceAction');
        return this;
    }
    /**
     * Grants permission to get the access status of AWS Organization portfolio share feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetAWSOrganizationsAccessStatus.html
     */
    toGetAWSOrganizationsAccessStatus() {
        this.to('servicecatalog:GetAWSOrganizationsAccessStatus');
        return this;
    }
    /**
     * Grants permission to get an application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetApplication.html
     */
    toGetApplication() {
        this.to('servicecatalog:GetApplication');
        return this;
    }
    /**
     * Grants permission to get an attribute group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetAttributeGroup.html
     */
    toGetAttributeGroup() {
        this.to('servicecatalog:GetAttributeGroup');
        return this;
    }
    /**
     * Grants permission to import a resource into a provisioned product.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ImportAsProvisionedProduct.html
     */
    toImportAsProvisionedProduct() {
        this.to('servicecatalog:ImportAsProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to list the portfolios that have been shared with you and you have accepted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAcceptedPortfolioShares.html
     */
    toListAcceptedPortfolioShares() {
        this.to('servicecatalog:ListAcceptedPortfolioShares');
        return this;
    }
    /**
     * Grants permission to list the applications in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListApplications.html
     */
    toListApplications() {
        this.to('servicecatalog:ListApplications');
        return this;
    }
    /**
     * Grants permission to list the attribute groups associated with an application.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAssociatedAttributeGroups.html
     */
    toListAssociatedAttributeGroups() {
        this.to('servicecatalog:ListAssociatedAttributeGroups');
        return this;
    }
    /**
     * Grants permission to list the resources associated with an application.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAssociatedResources.html
     */
    toListAssociatedResources() {
        this.to('servicecatalog:ListAssociatedResources');
        return this;
    }
    /**
     * Grants permission to list the attribute groups in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAttributeGroups.html
     */
    toListAttributeGroups() {
        this.to('servicecatalog:ListAttributeGroups');
        return this;
    }
    /**
     * Grants permission to list all the budgets associated to a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListBudgetsForResource.html
     */
    toListBudgetsForResource() {
        this.to('servicecatalog:ListBudgetsForResource');
        return this;
    }
    /**
     * Grants permission to list constraints associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListConstraintsForPortfolio.html
     */
    toListConstraintsForPortfolio() {
        this.to('servicecatalog:ListConstraintsForPortfolio');
        return this;
    }
    /**
     * Grants permission to list the different ways to launch a given product as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListLaunchPaths.html
     */
    toListLaunchPaths() {
        this.to('servicecatalog:ListLaunchPaths');
        return this;
    }
    /**
     * Grants permission to list the organization nodes that have access to the specified portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListOrganizationPortfolioAccess.html
     */
    toListOrganizationPortfolioAccess() {
        this.to('servicecatalog:ListOrganizationPortfolioAccess');
        return this;
    }
    /**
     * Grants permission to list the AWS accounts you have shared a given portfolio with
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolioAccess.html
     */
    toListPortfolioAccess() {
        this.to('servicecatalog:ListPortfolioAccess');
        return this;
    }
    /**
     * Grants permission to list the portfolios in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolios.html
     */
    toListPortfolios() {
        this.to('servicecatalog:ListPortfolios');
        return this;
    }
    /**
     * Grants permission to list the portfolios associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfoliosForProduct.html
     */
    toListPortfoliosForProduct() {
        this.to('servicecatalog:ListPortfoliosForProduct');
        return this;
    }
    /**
     * Grants permission to list the IAM principals associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPrincipalsForPortfolio.html
     */
    toListPrincipalsForPortfolio() {
        this.to('servicecatalog:ListPrincipalsForPortfolio');
        return this;
    }
    /**
     * Grants permission to list the provisioned product plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisionedProductPlans.html
     */
    toListProvisionedProductPlans() {
        this.to('servicecatalog:ListProvisionedProductPlans');
        return this;
    }
    /**
     * Grants permission to list the provisioning artifacts associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifacts.html
     */
    toListProvisioningArtifacts() {
        this.to('servicecatalog:ListProvisioningArtifacts');
        return this;
    }
    /**
     * Grants permission to list all provisioning artifacts for the specified self-service action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifactsForServiceAction.html
     */
    toListProvisioningArtifactsForServiceAction() {
        this.to('servicecatalog:ListProvisioningArtifactsForServiceAction');
        return this;
    }
    /**
     * Grants permission to list all the records in your account or all the records related to a given provisioned product
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListRecordHistory.html
     */
    toListRecordHistory() {
        this.to('servicecatalog:ListRecordHistory');
        return this;
    }
    /**
     * Grants permission to list the resources associated with the specified TagOption
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListResourcesForTagOption.html
     */
    toListResourcesForTagOption() {
        this.to('servicecatalog:ListResourcesForTagOption');
        return this;
    }
    /**
     * Grants permission to list all self-service actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActions.html
     */
    toListServiceActions() {
        this.to('servicecatalog:ListServiceActions');
        return this;
    }
    /**
     * Grants permission to list all the service actions associated with the specified provisioning artifact in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActionsForProvisioningArtifact.html
     */
    toListServiceActionsForProvisioningArtifact() {
        this.to('servicecatalog:ListServiceActionsForProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to list account, region and status of each stack instances that are associated with a CFN_STACKSET type provisioned product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListStackInstancesForProvisionedProduct.html
     */
    toListStackInstancesForProvisionedProduct() {
        this.to('servicecatalog:ListStackInstancesForProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to list the specified TagOptions or all TagOptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListTagOptions.html
     */
    toListTagOptions() {
        this.to('servicecatalog:ListTagOptions');
        return this;
    }
    /**
     * Grants permission to provision a product with a specified provisioning artifact and launch parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProvisionProduct.html
     */
    toProvisionProduct() {
        this.to('servicecatalog:ProvisionProduct');
        return this;
    }
    /**
     * Grants permission to reject a portfolio that has been shared with you that you previously accepted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_RejectPortfolioShare.html
     */
    toRejectPortfolioShare() {
        this.to('servicecatalog:RejectPortfolioShare');
        return this;
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ScanProvisionedProducts.html
     */
    toScanProvisionedProducts() {
        this.to('servicecatalog:ScanProvisionedProducts');
        return this;
    }
    /**
     * Grants permission to list the products available to you as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProducts.html
     */
    toSearchProducts() {
        this.to('servicecatalog:SearchProducts');
        return this;
    }
    /**
     * Grants permission to list all the products in your account or all the products associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProductsAsAdmin.html
     */
    toSearchProductsAsAdmin() {
        this.to('servicecatalog:SearchProductsAsAdmin');
        return this;
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProvisionedProducts.html
     */
    toSearchProvisionedProducts() {
        this.to('servicecatalog:SearchProvisionedProducts');
        return this;
    }
    /**
     * Grants permission to terminate an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_TerminateProvisionedProduct.html
     */
    toTerminateProvisionedProduct() {
        this.to('servicecatalog:TerminateProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to update the attributes of an existing application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateApplication.html
     */
    toUpdateApplication() {
        this.to('servicecatalog:UpdateApplication');
        return this;
    }
    /**
     * Grants permission to update the attributes of an existing attribute group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateAttributeGroup.html
     */
    toUpdateAttributeGroup() {
        this.to('servicecatalog:UpdateAttributeGroup');
        return this;
    }
    /**
     * Grants permission to update the metadata fields of an existing constraint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateConstraint.html
     */
    toUpdateConstraint() {
        this.to('servicecatalog:UpdateConstraint');
        return this;
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdatePortfolio.html
     */
    toUpdatePortfolio() {
        this.to('servicecatalog:UpdatePortfolio');
        return this;
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProduct.html
     */
    toUpdateProduct() {
        this.to('servicecatalog:UpdateProduct');
        return this;
    }
    /**
     * Grants permission to update an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProduct.html
     */
    toUpdateProvisionedProduct() {
        this.to('servicecatalog:UpdateProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to update the properties of an existing provisioned product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProductProperties.html
     */
    toUpdateProvisionedProductProperties() {
        this.to('servicecatalog:UpdateProvisionedProductProperties');
        return this;
    }
    /**
     * Grants permission to update the metadata fields of an existing provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisioningArtifact.html
     */
    toUpdateProvisioningArtifact() {
        this.to('servicecatalog:UpdateProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to update a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateServiceAction.html
     */
    toUpdateServiceAction() {
        this.to('servicecatalog:UpdateServiceAction');
        return this;
    }
    /**
     * Grants permission to update the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateTagOption.html
     */
    toUpdateTagOption() {
        this.to('servicecatalog:UpdateTagOption');
        return this;
    }
    /**
     * Adds a resource of type Application to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateApplication.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationId, account, region, partition) {
        var arn = 'arn:${Partition}:servicecatalog:${Region}:${Account}:/applications/${ApplicationId}';
        arn = arn.replace('${ApplicationId}', applicationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type AttributeGroup to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateAttributeGroup.html
     *
     * @param attributeGroupId - Identifier for the attributeGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAttributeGroup(attributeGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:servicecatalog:${Region}:${Account}:/attribute-groups/${AttributeGroupId}';
        arn = arn.replace('${AttributeGroupId}', attributeGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Portfolio to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_PortfolioDetail.html
     *
     * @param portfolioId - Identifier for the portfolioId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPortfolio(portfolioId, account, region, partition) {
        var arn = 'arn:${Partition}:catalog:${Region}:${Account}:portfolio/${PortfolioId}';
        arn = arn.replace('${PortfolioId}', portfolioId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Product to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProductViewDetail.html
     *
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProduct(productId, account, region, partition) {
        var arn = 'arn:${Partition}:catalog:${Region}:${Account}:product/${ProductId}';
        arn = arn.replace('${ProductId}', productId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters users to see and perform actions on resources created by anyone in the account
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccountLevel(value, operator) {
        return this.if(`servicecatalog:accountLevel`, value, operator || 'StringLike');
    }
    /**
     * Filters users to see and perform actions on resources created either by them or by anyone federating into the same role as them
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoleLevel(value, operator) {
        return this.if(`servicecatalog:roleLevel`, value, operator || 'StringLike');
    }
    /**
     * Filters users to see and perform actions on only resources that they created
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserLevel(value, operator) {
        return this.if(`servicecatalog:userLevel`, value, operator || 'StringLike');
    }
}
exports.Servicecatalog = Servicecatalog;
//# sourceMappingURL=data:application/json;base64,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