"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Snowball = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [snowball](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssnowball.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Snowball extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [snowball](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssnowball.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'snowball';
        this.accessLevelList = {
            "Write": [
                "CancelCluster",
                "CancelJob",
                "CreateAddress",
                "CreateCluster",
                "CreateJob",
                "UpdateCluster",
                "UpdateJob"
            ],
            "Read": [
                "DescribeAddress",
                "DescribeCluster",
                "DescribeJob",
                "GetJobManifest",
                "GetJobUnlockCode",
                "GetSnowballUsage"
            ],
            "List": [
                "DescribeAddresses",
                "ListClusterJobs",
                "ListClusters",
                "ListJobs"
            ]
        };
    }
    /**
     * Cancels a cluster job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CancelCluster.html
     */
    toCancelCluster() {
        this.to('snowball:CancelCluster');
        return this;
    }
    /**
     * Cancels the specified job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CancelJob.html
     */
    toCancelJob() {
        this.to('snowball:CancelJob');
        return this;
    }
    /**
     * Creates an address for a Snowball to be shipped to.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateAddress.html
     */
    toCreateAddress() {
        this.to('snowball:CreateAddress');
        return this;
    }
    /**
     * Creates an empty cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateCluster.html
     */
    toCreateCluster() {
        this.to('snowball:CreateCluster');
        return this;
    }
    /**
     * Creates a job to import or export data between Amazon S3 and your on-premises data center.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateJob.html
     */
    toCreateJob() {
        this.to('snowball:CreateJob');
        return this;
    }
    /**
     * Takes an AddressId and returns specific details about that address in the form of an Address object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeAddress.html
     */
    toDescribeAddress() {
        this.to('snowball:DescribeAddress');
        return this;
    }
    /**
     * Returns a specified number of ADDRESS objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeAddresses.html
     */
    toDescribeAddresses() {
        this.to('snowball:DescribeAddresses');
        return this;
    }
    /**
     * Returns information about a specific cluster including shipping information, cluster status, and other important metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeCluster.html
     */
    toDescribeCluster() {
        this.to('snowball:DescribeCluster');
        return this;
    }
    /**
     * Returns information about a specific job including shipping information, job status, and other important metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeJob.html
     */
    toDescribeJob() {
        this.to('snowball:DescribeJob');
        return this;
    }
    /**
     * Returns a link to an Amazon S3 presigned URL for the manifest file associated with the specified JobId value.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetJobManifest.html
     */
    toGetJobManifest() {
        this.to('snowball:GetJobManifest');
        return this;
    }
    /**
     * Returns the UnlockCode code value for the specified job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetJobUnlockCode.html
     */
    toGetJobUnlockCode() {
        this.to('snowball:GetJobUnlockCode');
        return this;
    }
    /**
     * Returns information about the Snowball service limit for your account, and also the number of Snowballs your account has in use.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetSnowballUsage.html
     */
    toGetSnowballUsage() {
        this.to('snowball:GetSnowballUsage');
        return this;
    }
    /**
     * Returns an array of JobListEntry objects of the specified length.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListClusterJobs.html
     */
    toListClusterJobs() {
        this.to('snowball:ListClusterJobs');
        return this;
    }
    /**
     * Returns an array of ClusterListEntry objects of the specified length.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListClusters.html
     */
    toListClusters() {
        this.to('snowball:ListClusters');
        return this;
    }
    /**
     * Returns an array of JobListEntry objects of the specified length.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListJobs.html
     */
    toListJobs() {
        this.to('snowball:ListJobs');
        return this;
    }
    /**
     * While a cluster's ClusterState value is in the AwaitingQuorum state, you can update some of the information associated with a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_UpdateCluster.html
     */
    toUpdateCluster() {
        this.to('snowball:UpdateCluster');
        return this;
    }
    /**
     * While a job's JobState value is New, you can update some of the information associated with a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_UpdateJob.html
     */
    toUpdateJob() {
        this.to('snowball:UpdateJob');
        return this;
    }
}
exports.Snowball = Snowball;
//# sourceMappingURL=data:application/json;base64,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