"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sso = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sso](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssso.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sso extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sso](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssso.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sso';
        this.accessLevelList = {
            "Write": [
                "AssociateDirectory",
                "AssociateProfile",
                "AttachManagedPolicyToPermissionSet",
                "CreateAccountAssignment",
                "CreateApplicationInstance",
                "CreateApplicationInstanceCertificate",
                "CreateManagedApplicationInstance",
                "CreatePermissionSet",
                "CreateProfile",
                "CreateTrust",
                "DeleteAccountAssignment",
                "DeleteApplicationInstance",
                "DeleteApplicationInstanceCertificate",
                "DeleteInlinePolicyFromPermissionSet",
                "DeleteManagedApplicationInstance",
                "DeletePermissionSet",
                "DeletePermissionsPolicy",
                "DeleteProfile",
                "DetachManagedPolicyFromPermissionSet",
                "DisassociateDirectory",
                "DisassociateProfile",
                "ImportApplicationInstanceServiceProviderMetadata",
                "ProvisionPermissionSet",
                "PutInlinePolicyToPermissionSet",
                "PutMfaDeviceManagementForDirectory",
                "PutPermissionsPolicy",
                "StartSSO",
                "UpdateApplicationInstanceActiveCertificate",
                "UpdateApplicationInstanceDisplayData",
                "UpdateApplicationInstanceResponseConfiguration",
                "UpdateApplicationInstanceResponseSchemaConfiguration",
                "UpdateApplicationInstanceSecurityConfiguration",
                "UpdateApplicationInstanceServiceProviderConfiguration",
                "UpdateApplicationInstanceStatus",
                "UpdateDirectoryAssociation",
                "UpdateManagedApplicationInstanceStatus",
                "UpdatePermissionSet",
                "UpdateProfile",
                "UpdateSSOConfiguration",
                "UpdateTrust"
            ],
            "Read": [
                "DescribeAccountAssignmentCreationStatus",
                "DescribeAccountAssignmentDeletionStatus",
                "DescribePermissionSet",
                "DescribePermissionSetProvisioningStatus",
                "DescribePermissionsPolicies",
                "DescribeRegisteredRegions",
                "GetApplicationInstance",
                "GetApplicationTemplate",
                "GetInlinePolicyForPermissionSet",
                "GetManagedApplicationInstance",
                "GetMfaDeviceManagementForDirectory",
                "GetPermissionSet",
                "GetPermissionsPolicy",
                "GetProfile",
                "GetSSOStatus",
                "GetSharedSsoConfiguration",
                "GetSsoConfiguration",
                "GetTrust",
                "SearchGroups",
                "SearchUsers"
            ],
            "List": [
                "ListAccountAssignmentCreationStatus",
                "ListAccountAssignmentDeletionStatus",
                "ListAccountAssignments",
                "ListAccountsForProvisionedPermissionSet",
                "ListApplicationInstanceCertificates",
                "ListApplicationInstances",
                "ListApplicationTemplates",
                "ListApplications",
                "ListDirectoryAssociations",
                "ListInstances",
                "ListManagedPoliciesInPermissionSet",
                "ListPermissionSetProvisioningStatus",
                "ListPermissionSets",
                "ListPermissionSetsProvisionedToAccount",
                "ListProfileAssociations",
                "ListProfiles",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Connect a directory to be used by AWS Single Sign-On
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateDirectory() {
        this.to('sso:AssociateDirectory');
        return this;
    }
    /**
     * Create an association between a directory user or group and a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateProfile() {
        this.to('sso:AssociateProfile');
        return this;
    }
    /**
     * Attaches an AWS managed policy to a permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAttachManagedPolicyToPermissionSet() {
        this.to('sso:AttachManagedPolicyToPermissionSet');
        return this;
    }
    /**
     * Assigns access to a Principal for a specified AWS account using a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateAccountAssignment() {
        this.to('sso:CreateAccountAssignment');
        return this;
    }
    /**
     * Add an application instance to AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstance() {
        this.to('sso:CreateApplicationInstance');
        return this;
    }
    /**
     * Add a new certificate for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstanceCertificate() {
        this.to('sso:CreateApplicationInstanceCertificate');
        return this;
    }
    /**
     * Add a managed application instance to AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateManagedApplicationInstance() {
        this.to('sso:CreateManagedApplicationInstance');
        return this;
    }
    /**
     * Create a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreatePermissionSet() {
        this.to('sso:CreatePermissionSet');
        return this;
    }
    /**
     * Create a profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateProfile() {
        this.to('sso:CreateProfile');
        return this;
    }
    /**
     * Create a federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateTrust() {
        this.to('sso:CreateTrust');
        return this;
    }
    /**
     * Deletes a Principal's access from a specified AWS account using a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteAccountAssignment() {
        this.to('sso:DeleteAccountAssignment');
        return this;
    }
    /**
     * Delete the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstance() {
        this.to('sso:DeleteApplicationInstance');
        return this;
    }
    /**
     * Delete an inactive or expired certificate from the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstanceCertificate() {
        this.to('sso:DeleteApplicationInstanceCertificate');
        return this;
    }
    /**
     * Deletes the inline policy from a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteInlinePolicyFromPermissionSet() {
        this.to('sso:DeleteInlinePolicyFromPermissionSet');
        return this;
    }
    /**
     * Delete the managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteManagedApplicationInstance() {
        this.to('sso:DeleteManagedApplicationInstance');
        return this;
    }
    /**
     * Delete a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeletePermissionSet() {
        this.to('sso:DeletePermissionSet');
        return this;
    }
    /**
     * Delete the permission policy associated with a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeletePermissionsPolicy() {
        this.to('sso:DeletePermissionsPolicy');
        return this;
    }
    /**
     * Delete the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteProfile() {
        this.to('sso:DeleteProfile');
        return this;
    }
    /**
     * Describes the status of the assignment creation request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeAccountAssignmentCreationStatus() {
        this.to('sso:DescribeAccountAssignmentCreationStatus');
        return this;
    }
    /**
     * Describes the status of an assignment deletion request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeAccountAssignmentDeletionStatus() {
        this.to('sso:DescribeAccountAssignmentDeletionStatus');
        return this;
    }
    /**
     * Describes a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionSet() {
        this.to('sso:DescribePermissionSet');
        return this;
    }
    /**
     * Describes the status for the given Permission Set Provisioning request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionSetProvisioningStatus() {
        this.to('sso:DescribePermissionSetProvisioningStatus');
        return this;
    }
    /**
     * Retrieve all the permissions policies associated with a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionsPolicies() {
        this.to('sso:DescribePermissionsPolicies');
        return this;
    }
    /**
     * Obtains the regions where your organization has enabled AWS Single Sign-on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeRegisteredRegions() {
        this.to('sso:DescribeRegisteredRegions');
        return this;
    }
    /**
     * Detaches the attached AWS managed policy from the specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDetachManagedPolicyFromPermissionSet() {
        this.to('sso:DetachManagedPolicyFromPermissionSet');
        return this;
    }
    /**
     * Disassociate a directory to be used by AWS Single Sign-On
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:UnauthorizeApplication
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateDirectory() {
        this.to('sso:DisassociateDirectory');
        return this;
    }
    /**
     * Disassociate a directory user or group from a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateProfile() {
        this.to('sso:DisassociateProfile');
        return this;
    }
    /**
     * Retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationInstance() {
        this.to('sso:GetApplicationInstance');
        return this;
    }
    /**
     * Retrieve application template details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationTemplate() {
        this.to('sso:GetApplicationTemplate');
        return this;
    }
    /**
     * Obtains the inline policy assigned to the permission set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetInlinePolicyForPermissionSet() {
        this.to('sso:GetInlinePolicyForPermissionSet');
        return this;
    }
    /**
     * Retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetManagedApplicationInstance() {
        this.to('sso:GetManagedApplicationInstance');
        return this;
    }
    /**
     * Retrieve Mfa Device Management settings for the directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetMfaDeviceManagementForDirectory() {
        this.to('sso:GetMfaDeviceManagementForDirectory');
        return this;
    }
    /**
     * Retrieve details of a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionSet() {
        this.to('sso:GetPermissionSet');
        return this;
    }
    /**
     * Retrieve all permission policies associated with a permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:DescribePermissionsPolicies
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionsPolicy() {
        this.to('sso:GetPermissionsPolicy');
        return this;
    }
    /**
     * Retrieve a profile for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetProfile() {
        this.to('sso:GetProfile');
        return this;
    }
    /**
     * Check if AWS Single Sign-On is enabled
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSSOStatus() {
        this.to('sso:GetSSOStatus');
        return this;
    }
    /**
     * Retrieve shared configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSharedSsoConfiguration() {
        this.to('sso:GetSharedSsoConfiguration');
        return this;
    }
    /**
     * Retrieve configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSsoConfiguration() {
        this.to('sso:GetSsoConfiguration');
        return this;
    }
    /**
     * Retrieve the federation trust in a target account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetTrust() {
        this.to('sso:GetTrust');
        return this;
    }
    /**
     * Update the application instance by uploading an application SAML metadata file provided by the service provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toImportApplicationInstanceServiceProviderMetadata() {
        this.to('sso:ImportApplicationInstanceServiceProviderMetadata');
        return this;
    }
    /**
     * Lists the status of the AWS account assignment creation requests for a specified SSO instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignmentCreationStatus() {
        this.to('sso:ListAccountAssignmentCreationStatus');
        return this;
    }
    /**
     * Lists the status of the AWS account assignment deletion requests for a specified SSO instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignmentDeletionStatus() {
        this.to('sso:ListAccountAssignmentDeletionStatus');
        return this;
    }
    /**
     * Lists the assignee of the specified AWS account with the specified permission set.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignments() {
        this.to('sso:ListAccountAssignments');
        return this;
    }
    /**
     * Lists all the AWS accounts where the specified permission set is provisioned.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountsForProvisionedPermissionSet() {
        this.to('sso:ListAccountsForProvisionedPermissionSet');
        return this;
    }
    /**
     * Retrieve all of the certificates for a given application instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstanceCertificates() {
        this.to('sso:ListApplicationInstanceCertificates');
        return this;
    }
    /**
     * Retrieve all application instances
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetApplicationInstance
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstances() {
        this.to('sso:ListApplicationInstances');
        return this;
    }
    /**
     * Retrieve all supported application templates
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetApplicationTemplate
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationTemplates() {
        this.to('sso:ListApplicationTemplates');
        return this;
    }
    /**
     * Retrieve all supported applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplications() {
        this.to('sso:ListApplications');
        return this;
    }
    /**
     * Retrieve details about the directory connected to AWS Single Sign-On
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListDirectoryAssociations() {
        this.to('sso:ListDirectoryAssociations');
        return this;
    }
    /**
     * Lists the SSO Instances that the caller has access to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListInstances() {
        this.to('sso:ListInstances');
        return this;
    }
    /**
     * Lists the AWS managed policies that are attached to a specified permission set.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListManagedPoliciesInPermissionSet() {
        this.to('sso:ListManagedPoliciesInPermissionSet');
        return this;
    }
    /**
     * Lists the status of the Permission Set Provisioning requests for a specified SSO instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSetProvisioningStatus() {
        this.to('sso:ListPermissionSetProvisioningStatus');
        return this;
    }
    /**
     * Retrieve all permission sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSets() {
        this.to('sso:ListPermissionSets');
        return this;
    }
    /**
     * Lists all the permission sets that are provisioned to a specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSetsProvisionedToAccount() {
        this.to('sso:ListPermissionSetsProvisionedToAccount');
        return this;
    }
    /**
     * Retrieve the directory user or group associated with the profile
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfileAssociations() {
        this.to('sso:ListProfileAssociations');
        return this;
    }
    /**
     * Retrieve all profiles for an application instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetProfile
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfiles() {
        this.to('sso:ListProfiles');
        return this;
    }
    /**
     * Lists the tags that are attached to a specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListTagsForResource() {
        this.to('sso:ListTagsForResource');
        return this;
    }
    /**
     * The process by which a specified permission set is provisioned to the specified target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toProvisionPermissionSet() {
        this.to('sso:ProvisionPermissionSet');
        return this;
    }
    /**
     * Attaches an IAM inline policy to a permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutInlinePolicyToPermissionSet() {
        this.to('sso:PutInlinePolicyToPermissionSet');
        return this;
    }
    /**
     * Put Mfa Device Management settings for the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutMfaDeviceManagementForDirectory() {
        this.to('sso:PutMfaDeviceManagementForDirectory');
        return this;
    }
    /**
     * Add a policy to a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutPermissionsPolicy() {
        this.to('sso:PutPermissionsPolicy');
        return this;
    }
    /**
     * Search for groups within the associated directory
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchGroups() {
        this.to('sso:SearchGroups');
        return this;
    }
    /**
     * Search for users within the associated directory
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchUsers() {
        this.to('sso:SearchUsers');
        return this;
    }
    /**
     * Initialize AWS Single Sign-On
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organization:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toStartSSO() {
        this.to('sso:StartSSO');
        return this;
    }
    /**
     * Associates a set of tags with a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toTagResource() {
        this.to('sso:TagResource');
        return this;
    }
    /**
     * Disassociates a set of tags from a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUntagResource() {
        this.to('sso:UntagResource');
        return this;
    }
    /**
     * Set a certificate as the active one for this application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceActiveCertificate() {
        this.to('sso:UpdateApplicationInstanceActiveCertificate');
        return this;
    }
    /**
     * Update display data of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceDisplayData() {
        this.to('sso:UpdateApplicationInstanceDisplayData');
        return this;
    }
    /**
     * Update federation response configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseConfiguration() {
        this.to('sso:UpdateApplicationInstanceResponseConfiguration');
        return this;
    }
    /**
     * Update federation response schema configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseSchemaConfiguration() {
        this.to('sso:UpdateApplicationInstanceResponseSchemaConfiguration');
        return this;
    }
    /**
     * Update security details for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceSecurityConfiguration() {
        this.to('sso:UpdateApplicationInstanceSecurityConfiguration');
        return this;
    }
    /**
     * Update service provider related configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceServiceProviderConfiguration() {
        this.to('sso:UpdateApplicationInstanceServiceProviderConfiguration');
        return this;
    }
    /**
     * Update the status of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceStatus() {
        this.to('sso:UpdateApplicationInstanceStatus');
        return this;
    }
    /**
     * Update the user attribute mappings for your connected directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateDirectoryAssociation() {
        this.to('sso:UpdateDirectoryAssociation');
        return this;
    }
    /**
     * Update the status of a managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateManagedApplicationInstanceStatus() {
        this.to('sso:UpdateManagedApplicationInstanceStatus');
        return this;
    }
    /**
     * Update the permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdatePermissionSet() {
        this.to('sso:UpdatePermissionSet');
        return this;
    }
    /**
     * Update the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateProfile() {
        this.to('sso:UpdateProfile');
        return this;
    }
    /**
     * Update the configuration for the current SSO instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateSSOConfiguration() {
        this.to('sso:UpdateSSOConfiguration');
        return this;
    }
    /**
     * Update the federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateTrust() {
        this.to('sso:UpdateTrust');
        return this;
    }
    /**
     * Adds a resource of type PermissionSet to the statement
     *
     * @param instanceId - Identifier for the instanceId.
     * @param permissionSetId - Identifier for the permissionSetId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPermissionSet(instanceId, permissionSetId, partition) {
        var arn = 'arn:${Partition}:sso:::permissionSet/${InstanceId}/${PermissionSetId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${PermissionSetId}', permissionSetId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Account to the statement
     *
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccount(accountId, partition) {
        var arn = 'arn:${Partition}:sso:::account/${AccountId}';
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * @param instanceId - Identifier for the instanceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstance(instanceId, partition) {
        var arn = 'arn:${Partition}:sso:::instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Sso = Sso;
//# sourceMappingURL=data:application/json;base64,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