"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsoDirectory = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sso-directory](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsssodirectory.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SsoDirectory extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sso-directory](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsssodirectory.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sso-directory';
        this.accessLevelList = {
            "Write": [
                "AddMemberToGroup",
                "CompleteVirtualMfaDeviceRegistration",
                "CreateAlias",
                "CreateBearerToken",
                "CreateExternalIdPConfigurationForDirectory",
                "CreateGroup",
                "CreateProvisioningTenant",
                "CreateUser",
                "DeleteBearerToken",
                "DeleteExternalIdPCertificate",
                "DeleteExternalIdPConfigurationForDirectory",
                "DeleteGroup",
                "DeleteMfaDeviceForUser",
                "DeleteProvisioningTenant",
                "DeleteUser",
                "DisableExternalIdPConfigurationForDirectory",
                "DisableUser",
                "EnableExternalIdPConfigurationForDirectory",
                "EnableUser",
                "ImportExternalIdPCertificate",
                "RemoveMemberFromGroup",
                "StartVirtualMfaDeviceRegistration",
                "UpdateExternalIdPConfigurationForDirectory",
                "UpdateGroup",
                "UpdatePassword",
                "UpdateUser",
                "VerifyEmail"
            ],
            "Read": [
                "DescribeDirectory",
                "DescribeUser",
                "GetAWSSPConfigurationForDirectory",
                "IsMemberInGroup",
                "SearchGroups",
                "SearchUsers"
            ],
            "List": [
                "DescribeGroups",
                "DescribeUsers",
                "ListBearerTokens",
                "ListExternalIdPCertificates",
                "ListExternalIdPConfigurationsForDirectory",
                "ListGroupsForUser",
                "ListMembersInGroup",
                "ListMfaDevicesForUser",
                "ListProvisioningTenants"
            ]
        };
    }
    /**
     * Grants permission to add a member to a group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAddMemberToGroup() {
        this.to('sso-directory:AddMemberToGroup');
        return this;
    }
    /**
     * Grants permission to complete the creation process of a virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCompleteVirtualMfaDeviceRegistration() {
        this.to('sso-directory:CompleteVirtualMfaDeviceRegistration');
        return this;
    }
    /**
     * Grants permission to create an alias for the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateAlias() {
        this.to('sso-directory:CreateAlias');
        return this;
    }
    /**
     * Grants permission to create a bearer token for a given provisioning tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateBearerToken() {
        this.to('sso-directory:CreateBearerToken');
        return this;
    }
    /**
     * Grants permission to create an External Identity Provider configuration for the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:CreateExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Grants permission to create a group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateGroup() {
        this.to('sso-directory:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a provisioning tenant for a given directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateProvisioningTenant() {
        this.to('sso-directory:CreateProvisioningTenant');
        return this;
    }
    /**
     * Grants permission to create a user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateUser() {
        this.to('sso-directory:CreateUser');
        return this;
    }
    /**
     * Grants permission to delete a bearer token
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteBearerToken() {
        this.to('sso-directory:DeleteBearerToken');
        return this;
    }
    /**
     * Grants permission to delete the given external IdP certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteExternalIdPCertificate() {
        this.to('sso-directory:DeleteExternalIdPCertificate');
        return this;
    }
    /**
     * Grants permission to delete an External Identity Provider configuration associated with the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:DeleteExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Grants permission to delete a group from the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteGroup() {
        this.to('sso-directory:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete a MFA device by device name for a given user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteMfaDeviceForUser() {
        this.to('sso-directory:DeleteMfaDeviceForUser');
        return this;
    }
    /**
     * Grants permission to delete the provisioning tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteProvisioningTenant() {
        this.to('sso-directory:DeleteProvisioningTenant');
        return this;
    }
    /**
     * Grants permission to delete a user from the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteUser() {
        this.to('sso-directory:DeleteUser');
        return this;
    }
    /**
     * Grants permission to retrieve information about the directory that AWS SSO provides by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeDirectory() {
        this.to('sso-directory:DescribeDirectory');
        return this;
    }
    /**
     * Grants permission to retrieve information about a group from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeGroups() {
        this.to('sso-directory:DescribeGroups');
        return this;
    }
    /**
     * Grants permission to retrieve information about a user from the directory that AWS SSO provides by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeUser() {
        this.to('sso-directory:DescribeUser');
        return this;
    }
    /**
     * Grants permission to retrieve information about user from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeUsers() {
        this.to('sso-directory:DescribeUsers');
        return this;
    }
    /**
     * Grants permission to disable authentication of end users with an External Identity Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisableExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:DisableExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Grants permission to deactivate a user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisableUser() {
        this.to('sso-directory:DisableUser');
        return this;
    }
    /**
     * Grants permission to enable authentication of end users with an External Identity Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toEnableExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:EnableExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Grants permission to activate user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toEnableUser() {
        this.to('sso-directory:EnableUser');
        return this;
    }
    /**
     * Grants permission to retrieve the AWS SSO Service Provider configurations for the directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetAWSSPConfigurationForDirectory() {
        this.to('sso-directory:GetAWSSPConfigurationForDirectory');
        return this;
    }
    /**
     * Grants permission to import the IdP certificate used for verifying external IdP responses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toImportExternalIdPCertificate() {
        this.to('sso-directory:ImportExternalIdPCertificate');
        return this;
    }
    /**
     * Grants permission to check if a member is a part of the group in the directory that AWS SSO provides by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toIsMemberInGroup() {
        this.to('sso-directory:IsMemberInGroup');
        return this;
    }
    /**
     * Grants permission to list bearer tokens for a given provisioning tenant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListBearerTokens() {
        this.to('sso-directory:ListBearerTokens');
        return this;
    }
    /**
     * Grants permission to list the external IdP certificates of a given directory and IdP
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListExternalIdPCertificates() {
        this.to('sso-directory:ListExternalIdPCertificates');
        return this;
    }
    /**
     * Grants permission to list all the External Identity Provider configurations created for the directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListExternalIdPConfigurationsForDirectory() {
        this.to('sso-directory:ListExternalIdPConfigurationsForDirectory');
        return this;
    }
    /**
     * Grants permission to list groups for a user from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListGroupsForUser() {
        this.to('sso-directory:ListGroupsForUser');
        return this;
    }
    /**
     * Grants permission to retrieve all members that are part of a group in the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListMembersInGroup() {
        this.to('sso-directory:ListMembersInGroup');
        return this;
    }
    /**
     * Grants permission to list all active MFA devices and their MFA device metadata for a user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListMfaDevicesForUser() {
        this.to('sso-directory:ListMfaDevicesForUser');
        return this;
    }
    /**
     * Grants permission to list provisioning tenants for a given directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProvisioningTenants() {
        this.to('sso-directory:ListProvisioningTenants');
        return this;
    }
    /**
     * Grants permission to remove a member that is part of a group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toRemoveMemberFromGroup() {
        this.to('sso-directory:RemoveMemberFromGroup');
        return this;
    }
    /**
     * Grants permission to search for groups within the associated directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchGroups() {
        this.to('sso-directory:SearchGroups');
        return this;
    }
    /**
     * Grants permission to search for users within the associated directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchUsers() {
        this.to('sso-directory:SearchUsers');
        return this;
    }
    /**
     * Grants permission to begin the creation process of virtual mfa device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toStartVirtualMfaDeviceRegistration() {
        this.to('sso-directory:StartVirtualMfaDeviceRegistration');
        return this;
    }
    /**
     * Grants permission to update an External Identity Provider configuration associated with the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateExternalIdPConfigurationForDirectory() {
        this.to('sso-directory:UpdateExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Grants permission to update information about a group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateGroup() {
        this.to('sso-directory:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to update a password by sending password reset link via email or generating one time password for a user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdatePassword() {
        this.to('sso-directory:UpdatePassword');
        return this;
    }
    /**
     * Grants permission to update user information in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateUser() {
        this.to('sso-directory:UpdateUser');
        return this;
    }
    /**
     * Grants permission to verify an email address of an User
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toVerifyEmail() {
        this.to('sso-directory:VerifyEmail');
        return this;
    }
}
exports.SsoDirectory = SsoDirectory;
//# sourceMappingURL=data:application/json;base64,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