"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.States = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [states](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsstepfunctions.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class States extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [states](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsstepfunctions.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'states';
        this.accessLevelList = {
            "Write": [
                "CreateActivity",
                "CreateStateMachine",
                "DeleteActivity",
                "DeleteStateMachine",
                "GetActivityTask",
                "SendTaskFailure",
                "SendTaskHeartbeat",
                "SendTaskSuccess",
                "StartExecution",
                "StopExecution",
                "UpdateStateMachine"
            ],
            "Read": [
                "DescribeActivity",
                "DescribeExecution",
                "DescribeStateMachine",
                "DescribeStateMachineForExecution",
                "GetExecutionHistory",
                "ListExecutions",
                "ListTagsForResource"
            ],
            "List": [
                "ListActivities",
                "ListStateMachines"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates an activity. Activities must poll Step Functions using the GetActivityTask and respond using SendTask* API calls.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_CreateActivity.html
     */
    toCreateActivity() {
        this.to('states:CreateActivity');
        return this;
    }
    /**
     * Creates a state machine.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_CreateStateMachine.html
     */
    toCreateStateMachine() {
        this.to('states:CreateStateMachine');
        return this;
    }
    /**
     * Deletes an activity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DeleteActivity.html
     */
    toDeleteActivity() {
        this.to('states:DeleteActivity');
        return this;
    }
    /**
     * Deletes a state machine.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DeleteStateMachine.html
     */
    toDeleteStateMachine() {
        this.to('states:DeleteStateMachine');
        return this;
    }
    /**
     * Describes an activity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeActivity.html
     */
    toDescribeActivity() {
        this.to('states:DescribeActivity');
        return this;
    }
    /**
     * Describes an execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeExecution.html
     */
    toDescribeExecution() {
        this.to('states:DescribeExecution');
        return this;
    }
    /**
     * Describes a state machine.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeStateMachine.html
     */
    toDescribeStateMachine() {
        this.to('states:DescribeStateMachine');
        return this;
    }
    /**
     * Describes state machine for an execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeStateMachineForExecution.html
     */
    toDescribeStateMachineForExecution() {
        this.to('states:DescribeStateMachineForExecution');
        return this;
    }
    /**
     * Used by workers to retrieve a task (with the specified activity ARN) which has been scheduled for execution by a running state machine.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_GetActivityTask.html
     */
    toGetActivityTask() {
        this.to('states:GetActivityTask');
        return this;
    }
    /**
     * Returns the history of the specified execution as a list of events. By default, the results are returned in ascending order of the timeStamp of the events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_GetExecutionHistory.html
     */
    toGetExecutionHistory() {
        this.to('states:GetExecutionHistory');
        return this;
    }
    /**
     * Lists the existing activities. The results may be split into multiple pages.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListActivities.html
     */
    toListActivities() {
        this.to('states:ListActivities');
        return this;
    }
    /**
     * Lists the executions of a state machine that meet the filtering criteria. The results may be split into multiple pages.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListExecutions.html
     */
    toListExecutions() {
        this.to('states:ListExecutions');
        return this;
    }
    /**
     * Lists the existing state machines. The results may be split into multiple pages.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListStateMachines.html
     */
    toListStateMachines() {
        this.to('states:ListStateMachines');
        return this;
    }
    /**
     * This action lists tags for an AWS Step Functions resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('states:ListTagsForResource');
        return this;
    }
    /**
     * Used by workers to report that the task identified by the taskToken failed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_SendTaskFailure.html
     */
    toSendTaskFailure() {
        this.to('states:SendTaskFailure');
        return this;
    }
    /**
     * Used by workers to report to the service that the task represented by the specified taskToken is still making progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_SendTaskHeartbeat.html
     */
    toSendTaskHeartbeat() {
        this.to('states:SendTaskHeartbeat');
        return this;
    }
    /**
     * Used by workers to report that the task identified by the taskToken completed successfully.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_SendTaskSuccess.html
     */
    toSendTaskSuccess() {
        this.to('states:SendTaskSuccess');
        return this;
    }
    /**
     * Starts a state machine execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html
     */
    toStartExecution() {
        this.to('states:StartExecution');
        return this;
    }
    /**
     * Stops an execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_StopExecution.html
     */
    toStopExecution() {
        this.to('states:StopExecution');
        return this;
    }
    /**
     * This action tags an AWS Step Functions resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        this.to('states:TagResource');
        return this;
    }
    /**
     * This action removes a tag from an AWS Step Functions resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('states:UntagResource');
        return this;
    }
    /**
     * Updates a state machine.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_UpdateStateMachine.html
     */
    toUpdateStateMachine() {
        this.to('states:UpdateStateMachine');
        return this;
    }
    /**
     * Adds a resource of type activity to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-activities.html
     *
     * @param activityName - Identifier for the activityName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onActivity(activityName, account, region, partition) {
        var arn = 'arn:${Partition}:states:${Region}:${Account}:activity:${ActivityName}';
        arn = arn.replace('${ActivityName}', activityName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type execution to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-executions.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param executionId - Identifier for the executionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExecution(stateMachineName, executionId, account, region, partition) {
        var arn = 'arn:${Partition}:states:${Region}:${Account}:execution:${StateMachineName}:${ExecutionId}';
        arn = arn.replace('${StateMachineName}', stateMachineName);
        arn = arn.replace('${ExecutionId}', executionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type statemachine to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatemachine(stateMachineName, account, region, partition) {
        var arn = 'arn:${Partition}:states:${Region}:${Account}:stateMachine:${StateMachineName}';
        arn = arn.replace('${StateMachineName}', stateMachineName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.States = States;
//# sourceMappingURL=data:application/json;base64,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