"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Support = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [support](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupport.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Support extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [support](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupport.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'support';
        this.accessLevelList = {
            "Write": [
                "AddAttachmentsToSet",
                "AddCommunicationToCase",
                "CreateCase",
                "InitiateCallForCase",
                "InitiateChatForCase",
                "PutCaseAttributes",
                "RateCaseCommunication",
                "RefreshTrustedAdvisorCheck",
                "ResolveCase"
            ],
            "Read": [
                "DescribeAttachment",
                "DescribeCaseAttributes",
                "DescribeCases",
                "DescribeCommunications",
                "DescribeIssueTypes",
                "DescribeServices",
                "DescribeSeverityLevels",
                "DescribeSupportLevel",
                "DescribeTrustedAdvisorCheckRefreshStatuses",
                "DescribeTrustedAdvisorCheckResult",
                "DescribeTrustedAdvisorCheckSummaries",
                "DescribeTrustedAdvisorChecks",
                "SearchForCases"
            ]
        };
    }
    /**
     * Adds one or more attachments to an AWS Support case.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_AddAttachmentsToSet.html
     */
    toAddAttachmentsToSet() {
        this.to('support:AddAttachmentsToSet');
        return this;
    }
    /**
     * Adds a customer communication to an AWS Support case.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_AddCommunicationToCase.html
     */
    toAddCommunicationToCase() {
        this.to('support:AddCommunicationToCase');
        return this;
    }
    /**
     * Creates a new AWS Support case.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_CreateCase.html
     */
    toCreateCase() {
        this.to('support:CreateCase');
        return this;
    }
    /**
     * Returns the description for an attachment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeAttachment.html
     */
    toDescribeAttachment() {
        this.to('support:DescribeAttachment');
        return this;
    }
    /**
     * This is an internally managed function which allows secondary services to read AWS Support case attributes.
     *
     * Access Level: Read
     */
    toDescribeCaseAttributes() {
        this.to('support:DescribeCaseAttributes');
        return this;
    }
    /**
     * Returns a list of AWS Support cases that matches the given inputs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeCases.html
     */
    toDescribeCases() {
        this.to('support:DescribeCases');
        return this;
    }
    /**
     * Returns the communications and attachments for one or more AWS Support cases.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeCommunications.html
     */
    toDescribeCommunications() {
        this.to('support:DescribeCommunications');
        return this;
    }
    /**
     * Returns issue types for AWS Support cases.
     *
     * Access Level: Read
     */
    toDescribeIssueTypes() {
        this.to('support:DescribeIssueTypes');
        return this;
    }
    /**
     * Returns the current list of AWS services and categories that applies to each service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeServices.html
     */
    toDescribeServices() {
        this.to('support:DescribeServices');
        return this;
    }
    /**
     * Returns the list of severity levels that can be assigned to an AWS Support case.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeSeverityLevels.html
     */
    toDescribeSeverityLevels() {
        this.to('support:DescribeSeverityLevels');
        return this;
    }
    /**
     * Returns the support level for an AWS Account identifier.
     *
     * Access Level: Read
     */
    toDescribeSupportLevel() {
        this.to('support:DescribeSupportLevel');
        return this;
    }
    /**
     * Returns the status of a Trusted Advisor refresh check based on a list of check identifiers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeTrustedAdvisorCheckRefreshStatuses.html
     */
    toDescribeTrustedAdvisorCheckRefreshStatuses() {
        this.to('support:DescribeTrustedAdvisorCheckRefreshStatuses');
        return this;
    }
    /**
     * Returns the results of the Trusted Advisor check that has the specified check identifier.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeTrustedAdvisorCheckResult.html
     */
    toDescribeTrustedAdvisorCheckResult() {
        this.to('support:DescribeTrustedAdvisorCheckResult');
        return this;
    }
    /**
     * Returns the summaries of the results of the Trusted Advisor checks that have the specified check identifiers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeTrustedAdvisorCheckSummaries.html
     */
    toDescribeTrustedAdvisorCheckSummaries() {
        this.to('support:DescribeTrustedAdvisorCheckSummaries');
        return this;
    }
    /**
     * Returns a list of all available Trusted Advisor checks, including name, identifier, category and description.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_DescribeTrustedAdvisorChecks.html
     */
    toDescribeTrustedAdvisorChecks() {
        this.to('support:DescribeTrustedAdvisorChecks');
        return this;
    }
    /**
     * This is an internally managed function to initiate a call on AWS Support Center.
     *
     * Access Level: Write
     */
    toInitiateCallForCase() {
        this.to('support:InitiateCallForCase');
        return this;
    }
    /**
     * This is an internally managed function to initiate a chat on AWS Support Center.
     *
     * Access Level: Write
     */
    toInitiateChatForCase() {
        this.to('support:InitiateChatForCase');
        return this;
    }
    /**
     * This is an internally managed function which allows secondary services to attach attributes to AWS Support cases.
     *
     * Access Level: Write
     */
    toPutCaseAttributes() {
        this.to('support:PutCaseAttributes');
        return this;
    }
    /**
     * Rate an AWS Support case communication.
     *
     * Access Level: Write
     */
    toRateCaseCommunication() {
        this.to('support:RateCaseCommunication');
        return this;
    }
    /**
     * Requests a refresh of the Trusted Advisor check that has the specified check identifier.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_RefreshTrustedAdvisorCheck.html
     */
    toRefreshTrustedAdvisorCheck() {
        this.to('support:RefreshTrustedAdvisorCheck');
        return this;
    }
    /**
     * Resolves an AWS Support case.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_ResolveCase.html
     */
    toResolveCase() {
        this.to('support:ResolveCase');
        return this;
    }
    /**
     * Returns a list of AWS Support cases that matches the given inputs.
     *
     * Access Level: Read
     */
    toSearchForCases() {
        this.to('support:SearchForCases');
        return this;
    }
}
exports.Support = Support;
//# sourceMappingURL=data:application/json;base64,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