"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transfer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [transfer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstransferforsftp.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Transfer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [transfer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstransferforsftp.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'transfer';
        this.accessLevelList = {
            "Write": [
                "CreateServer",
                "CreateUser",
                "DeleteServer",
                "DeleteSshPublicKey",
                "DeleteUser",
                "ImportSshPublicKey",
                "StartServer",
                "StopServer",
                "UpdateServer",
                "UpdateUser"
            ],
            "Read": [
                "DescribeServer",
                "DescribeUser",
                "ListTagsForResource",
                "TestIdentityProvider"
            ],
            "List": [
                "ListServers",
                "ListUsers"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Enables the caller to create a server.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateServer.html
     */
    toCreateServer() {
        this.to('transfer:CreateServer');
        return this;
    }
    /**
     * Enables the caller to add a user associated with a server.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateUser.html
     */
    toCreateUser() {
        this.to('transfer:CreateUser');
        return this;
    }
    /**
     * Enables the caller to delete a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteServer.html
     */
    toDeleteServer() {
        this.to('transfer:DeleteServer');
        return this;
    }
    /**
     * Enables the caller to delete an SSH public key from a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteSshPublicKey.html
     */
    toDeleteSshPublicKey() {
        this.to('transfer:DeleteSshPublicKey');
        return this;
    }
    /**
     * Enables the caller to delete a user associated with a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('transfer:DeleteUser');
        return this;
    }
    /**
     * Enables the caller to describe a server.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeServer.html
     */
    toDescribeServer() {
        this.to('transfer:DescribeServer');
        return this;
    }
    /**
     * Enables the caller to describe a user associated with a server.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeUser.html
     */
    toDescribeUser() {
        this.to('transfer:DescribeUser');
        return this;
    }
    /**
     * Enables the caller to add an SSH public key to a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ImportSshPublicKey.html
     */
    toImportSshPublicKey() {
        this.to('transfer:ImportSshPublicKey');
        return this;
    }
    /**
     * Enables the caller to list servers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListServers.html
     */
    toListServers() {
        this.to('transfer:ListServers');
        return this;
    }
    /**
     * Enables the caller to list tags for a server or a user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('transfer:ListTagsForResource');
        return this;
    }
    /**
     * Enables the caller to list users associated with a server.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListUsers.html
     */
    toListUsers() {
        this.to('transfer:ListUsers');
        return this;
    }
    /**
     * Enables the caller to start a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StartServer.html
     */
    toStartServer() {
        this.to('transfer:StartServer');
        return this;
    }
    /**
     * Enables the caller to stop a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StopServer.html
     */
    toStopServer() {
        this.to('transfer:StopServer');
        return this;
    }
    /**
     * Enables the caller to tag a server or a user.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_TagResource.html
     */
    toTagResource() {
        this.to('transfer:TagResource');
        return this;
    }
    /**
     * Enables the caller to test a server's custom identity provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_TestIdentityProvider.html
     */
    toTestIdentityProvider() {
        this.to('transfer:TestIdentityProvider');
        return this;
    }
    /**
     * Enables the caller to untag a server or a user.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UntagResource.html
     */
    toUntagResource() {
        this.to('transfer:UntagResource');
        return this;
    }
    /**
     * Enables the caller to update the configuration of a server
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateServer.html
     */
    toUpdateServer() {
        this.to('transfer:UpdateServer');
        return this;
    }
    /**
     * Enables the caller to update the configuration of a user
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateUser.html
     */
    toUpdateUser() {
        this.to('transfer:UpdateUser');
        return this;
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/users.html
     *
     * @param serverId - Identifier for the serverId.
     * @param username - Identifier for the username.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(serverId, username, account, region, partition) {
        var arn = 'arn:${Partition}:transfer:${Region}:${Account}:user/${ServerId}/${Username}';
        arn = arn.replace('${ServerId}', serverId);
        arn = arn.replace('${Username}', username);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type server to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/configuring-servers.html
     *
     * @param serverId - Identifier for the serverId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onServer(serverId, account, region, partition) {
        var arn = 'arn:${Partition}:transfer:${Region}:${Account}:server/${ServerId}';
        arn = arn.replace('${ServerId}', serverId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Transfer = Transfer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzdHJhbnNmZXJmb3JzZnRwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzdHJhbnNmZXJmb3JzZnRwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHNDQUE0QztBQUU1Qzs7OztHQUlHO0FBQ0gsTUFBYSxRQUFTLFNBQVEsd0JBQWU7SUFHM0M7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFSTixrQkFBYSxHQUFHLFVBQVUsQ0FBQztRQThQeEIsb0JBQWUsR0FBb0I7WUFDM0MsT0FBTyxFQUFFO2dCQUNQLGNBQWM7Z0JBQ2QsWUFBWTtnQkFDWixjQUFjO2dCQUNkLG9CQUFvQjtnQkFDcEIsWUFBWTtnQkFDWixvQkFBb0I7Z0JBQ3BCLGFBQWE7Z0JBQ2IsWUFBWTtnQkFDWixjQUFjO2dCQUNkLFlBQVk7YUFDYjtZQUNELE1BQU0sRUFBRTtnQkFDTixnQkFBZ0I7Z0JBQ2hCLGNBQWM7Z0JBQ2QscUJBQXFCO2dCQUNyQixzQkFBc0I7YUFDdkI7WUFDRCxNQUFNLEVBQUU7Z0JBQ04sYUFBYTtnQkFDYixXQUFXO2FBQ1o7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQWhSRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQStCRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksTUFBTSxDQUFDLFFBQWdCLEVBQUUsUUFBZ0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxJQUFJLEdBQUcsR0FBRyw2RUFBNkUsQ0FBQztRQUN4RixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDM0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksUUFBUSxDQUFDLFFBQWdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckYsSUFBSSxHQUFHLEdBQUcsbUVBQW1FLENBQUM7UUFDOUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUF6VUQsNEJBeVVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSBcIi4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWxcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCIuLi9zaGFyZWRcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3RyYW5zZmVyXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3N0cmFuc2ZlcmZvcnNmdHAuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBUcmFuc2ZlciBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3RyYW5zZmVyJztcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFt0cmFuc2Zlcl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzdHJhbnNmZXJmb3JzZnRwLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBjcmVhdGUgYSBzZXJ2ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9DcmVhdGVTZXJ2ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU2VydmVyKCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOkNyZWF0ZVNlcnZlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBhZGQgYSB1c2VyIGFzc29jaWF0ZWQgd2l0aCBhIHNlcnZlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RyYW5zZmVyL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0NyZWF0ZVVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVXNlcigpIHtcbiAgICB0aGlzLnRvKCd0cmFuc2ZlcjpDcmVhdGVVc2VyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlcyB0aGUgY2FsbGVyIHRvIGRlbGV0ZSBhIHNlcnZlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdHJhbnNmZXIvbGF0ZXN0L3VzZXJndWlkZS9BUElfRGVsZXRlU2VydmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVNlcnZlcigpIHtcbiAgICB0aGlzLnRvKCd0cmFuc2ZlcjpEZWxldGVTZXJ2ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIHRoZSBjYWxsZXIgdG8gZGVsZXRlIGFuIFNTSCBwdWJsaWMga2V5IGZyb20gYSB1c2VyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZWxldGVTc2hQdWJsaWNLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlU3NoUHVibGljS2V5KCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOkRlbGV0ZVNzaFB1YmxpY0tleScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBkZWxldGUgYSB1c2VyIGFzc29jaWF0ZWQgd2l0aCBhIHNlcnZlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdHJhbnNmZXIvbGF0ZXN0L3VzZXJndWlkZS9BUElfRGVsZXRlVXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVVc2VyKCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOkRlbGV0ZVVzZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIHRoZSBjYWxsZXIgdG8gZGVzY3JpYmUgYSBzZXJ2ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdHJhbnNmZXIvbGF0ZXN0L3VzZXJndWlkZS9BUElfRGVzY3JpYmVTZXJ2ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTZXJ2ZXIoKSB7XG4gICAgdGhpcy50bygndHJhbnNmZXI6RGVzY3JpYmVTZXJ2ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIHRoZSBjYWxsZXIgdG8gZGVzY3JpYmUgYSB1c2VyIGFzc29jaWF0ZWQgd2l0aCBhIHNlcnZlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZXNjcmliZVVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVVc2VyKCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOkRlc2NyaWJlVXNlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBhZGQgYW4gU1NIIHB1YmxpYyBrZXkgdG8gYSB1c2VyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9JbXBvcnRTc2hQdWJsaWNLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvSW1wb3J0U3NoUHVibGljS2V5KCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOkltcG9ydFNzaFB1YmxpY0tleScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBsaXN0IHNlcnZlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9MaXN0U2VydmVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2VydmVycygpIHtcbiAgICB0aGlzLnRvKCd0cmFuc2ZlcjpMaXN0U2VydmVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBsaXN0IHRhZ3MgZm9yIGEgc2VydmVyIG9yIGEgdXNlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygndHJhbnNmZXI6TGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBsaXN0IHVzZXJzIGFzc29jaWF0ZWQgd2l0aCBhIHNlcnZlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9MaXN0VXNlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFVzZXJzKCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOkxpc3RVc2VycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byBzdGFydCBhIHNlcnZlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdHJhbnNmZXIvbGF0ZXN0L3VzZXJndWlkZS9BUElfU3RhcnRTZXJ2ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRTZXJ2ZXIoKSB7XG4gICAgdGhpcy50bygndHJhbnNmZXI6U3RhcnRTZXJ2ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIHRoZSBjYWxsZXIgdG8gc3RvcCBhIHNlcnZlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdHJhbnNmZXIvbGF0ZXN0L3VzZXJndWlkZS9BUElfU3RvcFNlcnZlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wU2VydmVyKCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOlN0b3BTZXJ2ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIHRoZSBjYWxsZXIgdG8gdGFnIGEgc2VydmVyIG9yIGEgdXNlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCd0cmFuc2ZlcjpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byB0ZXN0IGEgc2VydmVyJ3MgY3VzdG9tIGlkZW50aXR5IHByb3ZpZGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RyYW5zZmVyL2xhdGVzdC91c2VyZ3VpZGUvQVBJX1Rlc3RJZGVudGl0eVByb3ZpZGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Rlc3RJZGVudGl0eVByb3ZpZGVyKCkge1xuICAgIHRoaXMudG8oJ3RyYW5zZmVyOlRlc3RJZGVudGl0eVByb3ZpZGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlcyB0aGUgY2FsbGVyIHRvIHVudGFnIGEgc2VydmVyIG9yIGEgdXNlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90cmFuc2Zlci9sYXRlc3QvdXNlcmd1aWRlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygndHJhbnNmZXI6VW50YWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIGNhbGxlciB0byB1cGRhdGUgdGhlIGNvbmZpZ3VyYXRpb24gb2YgYSBzZXJ2ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RyYW5zZmVyL2xhdGVzdC91c2VyZ3VpZGUvQVBJX1VwZGF0ZVNlcnZlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTZXJ2ZXIoKSB7XG4gICAgdGhpcy50bygndHJhbnNmZXI6VXBkYXRlU2VydmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlcyB0aGUgY2FsbGVyIHRvIHVwZGF0ZSB0aGUgY29uZmlndXJhdGlvbiBvZiBhIHVzZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RyYW5zZmVyL2xhdGVzdC91c2VyZ3VpZGUvQVBJX1VwZGF0ZVVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVXNlcigpIHtcbiAgICB0aGlzLnRvKCd0cmFuc2ZlcjpVcGRhdGVVc2VyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgXCJXcml0ZVwiOiBbXG4gICAgICBcIkNyZWF0ZVNlcnZlclwiLFxuICAgICAgXCJDcmVhdGVVc2VyXCIsXG4gICAgICBcIkRlbGV0ZVNlcnZlclwiLFxuICAgICAgXCJEZWxldGVTc2hQdWJsaWNLZXlcIixcbiAgICAgIFwiRGVsZXRlVXNlclwiLFxuICAgICAgXCJJbXBvcnRTc2hQdWJsaWNLZXlcIixcbiAgICAgIFwiU3RhcnRTZXJ2ZXJcIixcbiAgICAgIFwiU3RvcFNlcnZlclwiLFxuICAgICAgXCJVcGRhdGVTZXJ2ZXJcIixcbiAgICAgIFwiVXBkYXRlVXNlclwiXG4gICAgXSxcbiAgICBcIlJlYWRcIjogW1xuICAgICAgXCJEZXNjcmliZVNlcnZlclwiLFxuICAgICAgXCJEZXNjcmliZVVzZXJcIixcbiAgICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiLFxuICAgICAgXCJUZXN0SWRlbnRpdHlQcm92aWRlclwiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJMaXN0U2VydmVyc1wiLFxuICAgICAgXCJMaXN0VXNlcnNcIlxuICAgIF0sXG4gICAgXCJUYWdnaW5nXCI6IFtcbiAgICAgIFwiVGFnUmVzb3VyY2VcIixcbiAgICAgIFwiVW50YWdSZXNvdXJjZVwiXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB1c2VyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RyYW5zZmVyL2xhdGVzdC91c2VyZ3VpZGUvdXNlcnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc2VydmVySWQgLSBJZGVudGlmaWVyIGZvciB0aGUgc2VydmVySWQuXG4gICAqIEBwYXJhbSB1c2VybmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB1c2VybmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Vc2VyKHNlcnZlcklkOiBzdHJpbmcsIHVzZXJuYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnRyYW5zZmVyOiR7UmVnaW9ufToke0FjY291bnR9OnVzZXIvJHtTZXJ2ZXJJZH0vJHtVc2VybmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1NlcnZlcklkfScsIHNlcnZlcklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtVc2VybmFtZX0nLCB1c2VybmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc2VydmVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RyYW5zZmVyL2xhdGVzdC91c2VyZ3VpZGUvY29uZmlndXJpbmctc2VydmVycy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzZXJ2ZXJJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzZXJ2ZXJJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TZXJ2ZXIoc2VydmVySWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06dHJhbnNmZXI6JHtSZWdpb259OiR7QWNjb3VudH06c2VydmVyLyR7U2VydmVySWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtTZXJ2ZXJJZH0nLCBzZXJ2ZXJJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cbn1cbiJdfQ==