"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Trustedadvisor = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstrustedadvisor.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Trustedadvisor extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstrustedadvisor.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'trustedadvisor';
        this.accessLevelList = {
            "Read": [
                "DescribeAccount",
                "DescribeAccountAccess",
                "DescribeCheckItems",
                "DescribeCheckRefreshStatuses",
                "DescribeCheckSummaries",
                "DescribeChecks",
                "DescribeNotificationPreferences",
                "DescribeOrganization",
                "DescribeOrganizationAccounts",
                "DescribeReports",
                "DescribeServiceMetadata",
                "ListAccountsForParent",
                "ListOrganizationalUnitsForParent",
                "ListRoots"
            ],
            "Write": [
                "ExcludeCheckItems",
                "GenerateReport",
                "IncludeCheckItems",
                "RefreshCheck",
                "SetAccountAccess",
                "SetOrganizationAccess",
                "UpdateNotificationPreferences"
            ]
        };
    }
    /**
     * Grants permission to view the AWS Support plan and various AWS Trusted Advisor preferences
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeAccount() {
        this.to('trustedadvisor:DescribeAccount');
        return this;
    }
    /**
     * Grants permission to view if the AWS account has enabled or disabled AWS Trusted Advisor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeAccountAccess() {
        this.to('trustedadvisor:DescribeAccountAccess');
        return this;
    }
    /**
     * Grants permission to view details for the check items
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckItems() {
        this.to('trustedadvisor:DescribeCheckItems');
        return this;
    }
    /**
     * Grants permission to view the refresh statuses for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckRefreshStatuses() {
        this.to('trustedadvisor:DescribeCheckRefreshStatuses');
        return this;
    }
    /**
     * Grants permission to view AWS Trusted Advisor check summaries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckSummaries() {
        this.to('trustedadvisor:DescribeCheckSummaries');
        return this;
    }
    /**
     * Grants permission to view details for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeChecks() {
        this.to('trustedadvisor:DescribeChecks');
        return this;
    }
    /**
     * Grants permission to view the notification preferences for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeNotificationPreferences() {
        this.to('trustedadvisor:DescribeNotificationPreferences');
        return this;
    }
    /**
     * Grants permission to view if the AWS account meets the requirements to enable the organizational view feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeOrganization() {
        this.to('trustedadvisor:DescribeOrganization');
        return this;
    }
    /**
     * Grants permission to view the linked AWS accounts that are in the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeOrganizationAccounts() {
        this.to('trustedadvisor:DescribeOrganizationAccounts');
        return this;
    }
    /**
     * Grants permission to view details for organizational view reports, such as the report name, runtime, date created, status, and format
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeReports() {
        this.to('trustedadvisor:DescribeReports');
        return this;
    }
    /**
     * Grants permission to view information about organizational view reports, such as the AWS Regions, check categories, check names, and resource statuses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeServiceMetadata() {
        this.to('trustedadvisor:DescribeServiceMetadata');
        return this;
    }
    /**
     * Grants permission to exclude recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toExcludeCheckItems() {
        this.to('trustedadvisor:ExcludeCheckItems');
        return this;
    }
    /**
     * Grants permission to create a report for AWS Trusted Advisor checks in your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toGenerateReport() {
        this.to('trustedadvisor:GenerateReport');
        return this;
    }
    /**
     * Grants permission to include recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toIncludeCheckItems() {
        this.to('trustedadvisor:IncludeCheckItems');
        return this;
    }
    /**
     * Grants permission to view, in the Trusted Advisor console, all of the accounts in an AWS organization that are contained by a root or organizational unit (OU)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toListAccountsForParent() {
        this.to('trustedadvisor:ListAccountsForParent');
        return this;
    }
    /**
     * Grants permission to view, in the Trusted Advisor console, all of the organizational units (OUs) in a parent organizational unit or root
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toListOrganizationalUnitsForParent() {
        this.to('trustedadvisor:ListOrganizationalUnitsForParent');
        return this;
    }
    /**
     * Grants permission to view, in the Trusted Advisor console, all of the roots that are defined in an AWS organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toListRoots() {
        this.to('trustedadvisor:ListRoots');
        return this;
    }
    /**
     * Grants permission to refresh an AWS Trusted Advisor check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toRefreshCheck() {
        this.to('trustedadvisor:RefreshCheck');
        return this;
    }
    /**
     * Grants permission to enable or disable AWS Trusted Advisor for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toSetAccountAccess() {
        this.to('trustedadvisor:SetAccountAccess');
        return this;
    }
    /**
     * Grants permission to enable the organizational view feature for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toSetOrganizationAccess() {
        this.to('trustedadvisor:SetOrganizationAccess');
        return this;
    }
    /**
     * Grants permission to update notification preferences for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toUpdateNotificationPreferences() {
        this.to('trustedadvisor:UpdateNotificationPreferences');
        return this;
    }
    /**
     * Adds a resource of type checks to the statement
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html
     *
     * @param categoryCode - Identifier for the categoryCode.
     * @param checkId - Identifier for the checkId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChecks(categoryCode, checkId, account, region, partition) {
        var arn = 'arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}';
        arn = arn.replace('${CategoryCode}', categoryCode);
        arn = arn.replace('${CheckId}', checkId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Trustedadvisor = Trustedadvisor;
//# sourceMappingURL=data:application/json;base64,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