"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WafRegional = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [waf-regional](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswafregional.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class WafRegional extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [waf-regional](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswafregional.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'waf-regional';
        this.accessLevelList = {
            "Write": [
                "AssociateWebACL",
                "CreateByteMatchSet",
                "CreateGeoMatchSet",
                "CreateIPSet",
                "CreateRateBasedRule",
                "CreateRegexMatchSet",
                "CreateRegexPatternSet",
                "CreateRule",
                "CreateRuleGroup",
                "CreateSizeConstraintSet",
                "CreateSqlInjectionMatchSet",
                "CreateWebACLMigrationStack",
                "CreateXssMatchSet",
                "DeleteByteMatchSet",
                "DeleteGeoMatchSet",
                "DeleteIPSet",
                "DeleteLoggingConfiguration",
                "DeleteRateBasedRule",
                "DeleteRegexMatchSet",
                "DeleteRegexPatternSet",
                "DeleteRule",
                "DeleteRuleGroup",
                "DeleteSizeConstraintSet",
                "DeleteSqlInjectionMatchSet",
                "DeleteXssMatchSet",
                "DisassociateWebACL",
                "PutLoggingConfiguration",
                "UpdateByteMatchSet",
                "UpdateGeoMatchSet",
                "UpdateIPSet",
                "UpdateRateBasedRule",
                "UpdateRegexMatchSet",
                "UpdateRegexPatternSet",
                "UpdateRule",
                "UpdateRuleGroup",
                "UpdateSizeConstraintSet",
                "UpdateSqlInjectionMatchSet",
                "UpdateXssMatchSet"
            ],
            "Permissions management": [
                "CreateWebACL",
                "DeletePermissionPolicy",
                "DeleteWebACL",
                "PutPermissionPolicy",
                "UpdateWebACL"
            ],
            "Read": [
                "GetByteMatchSet",
                "GetChangeToken",
                "GetChangeTokenStatus",
                "GetGeoMatchSet",
                "GetIPSet",
                "GetLoggingConfiguration",
                "GetPermissionPolicy",
                "GetRateBasedRule",
                "GetRateBasedRuleManagedKeys",
                "GetRegexMatchSet",
                "GetRegexPatternSet",
                "GetRule",
                "GetRuleGroup",
                "GetSampledRequests",
                "GetSizeConstraintSet",
                "GetSqlInjectionMatchSet",
                "GetWebACL",
                "GetWebACLForResource",
                "GetXssMatchSet",
                "ListTagsForResource"
            ],
            "List": [
                "ListActivatedRulesInRuleGroup",
                "ListByteMatchSets",
                "ListGeoMatchSets",
                "ListIPSets",
                "ListLoggingConfigurations",
                "ListRateBasedRules",
                "ListRegexMatchSets",
                "ListRegexPatternSets",
                "ListResourcesForWebACL",
                "ListRuleGroups",
                "ListRules",
                "ListSizeConstraintSets",
                "ListSqlInjectionMatchSets",
                "ListSubscribedRuleGroups",
                "ListWebACLs",
                "ListXssMatchSets"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Associates a WebACL with a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_AssociateWebACL.html
     */
    toAssociateWebACL() {
        this.to('waf-regional:AssociateWebACL');
        return this;
    }
    /**
     * Creates a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateByteMatchSet.html
     */
    toCreateByteMatchSet() {
        this.to('waf-regional:CreateByteMatchSet');
        return this;
    }
    /**
     * Creates a GeoMatchSet, which you use to specify which web requests you want to allow or block based on the country that the requests originate rom.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateGeoMatchSet.html
     */
    toCreateGeoMatchSet() {
        this.to('waf-regional:CreateGeoMatchSet');
        return this;
    }
    /**
     * Creates an IPSet, which you use to specify which web requests you want to allow or block based on the IP addresses that the requests originate rom.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateIPSet.html
     */
    toCreateIPSet() {
        this.to('waf-regional:CreateIPSet');
        return this;
    }
    /**
     * Creates a RateBasedRule, which contains a RateLimit specifying the maximum number of requests that AWS WAF allows from a specified IP address n a five-minute period.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateRateBasedRule.html
     */
    toCreateRateBasedRule() {
        this.to('waf-regional:CreateRateBasedRule');
        return this;
    }
    /**
     * Creates a RegexMatchSet, which you use to specify which web requests you want to allow or block based on the regex patterns you specified in a egexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateRegexMatchSet.html
     */
    toCreateRegexMatchSet() {
        this.to('waf-regional:CreateRegexMatchSet');
        return this;
    }
    /**
     * Creates a RegexPatternSet, which you use to specify the regular expression (regex) pattern that you want AWS WAF to search for.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateRegexPatternSet.html
     */
    toCreateRegexPatternSet() {
        this.to('waf-regional:CreateRegexPatternSet');
        return this;
    }
    /**
     * Creates a Rule, which contains the IPSet objects, ByteMatchSet objects, and other predicates that identify the requests that you want to lock.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateRule.html
     */
    toCreateRule() {
        this.to('waf-regional:CreateRule');
        return this;
    }
    /**
     * Creates a RuleGroup. A rule group is a collection of predefined rules that you add to a WebACL.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        this.to('waf-regional:CreateRuleGroup');
        return this;
    }
    /**
     * Creates a SizeConstraintSet, which you use to identify the part of a web request that you want to check for length.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateSizeConstraintSet.html
     */
    toCreateSizeConstraintSet() {
        this.to('waf-regional:CreateSizeConstraintSet');
        return this;
    }
    /**
     * Creates a SqlInjectionMatchSet, which you use to allow, block, or count requests that contain snippets of SQL code in a specified part of web equests.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateSqlInjectionMatchSet.html
     */
    toCreateSqlInjectionMatchSet() {
        this.to('waf-regional:CreateSqlInjectionMatchSet');
        return this;
    }
    /**
     * Creates a WebACL, which contains the Rules that identify the CloudFront web requests that you want to allow, block, or count.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateWebACL.html
     */
    toCreateWebACL() {
        this.to('waf-regional:CreateWebACL');
        return this;
    }
    /**
     * Create and store a CloudFormation tempalte that creates an equivalent WAF v2 WebACL from the given WAF Classic WebACL in the given S3 bucket.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateWebACLMigrationStack.html
     */
    toCreateWebACLMigrationStack() {
        this.to('waf-regional:CreateWebACLMigrationStack');
        return this;
    }
    /**
     * Creates an XssMatchSet, which you use to allow, block, or count requests that contain cross-site scripting attacks in the specified part of web equests.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_CreateXssMatchSet.html
     */
    toCreateXssMatchSet() {
        this.to('waf-regional:CreateXssMatchSet');
        return this;
    }
    /**
     * Permanently deletes a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteByteMatchSet.html
     */
    toDeleteByteMatchSet() {
        this.to('waf-regional:DeleteByteMatchSet');
        return this;
    }
    /**
     * Permanently deletes an GeoMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteGeoMatchSet.html
     */
    toDeleteGeoMatchSet() {
        this.to('waf-regional:DeleteGeoMatchSet');
        return this;
    }
    /**
     * Permanently deletes an IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteIPSet.html
     */
    toDeleteIPSet() {
        this.to('waf-regional:DeleteIPSet');
        return this;
    }
    /**
     * Permanently deletes the LoggingConfiguration from the specified web ACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteLoggingConfiguration.html
     */
    toDeleteLoggingConfiguration() {
        this.to('waf-regional:DeleteLoggingConfiguration');
        return this;
    }
    /**
     * Permanently deletes an IAM policy from the specified RuleGroup.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeletePermissionPolicy.html
     */
    toDeletePermissionPolicy() {
        this.to('waf-regional:DeletePermissionPolicy');
        return this;
    }
    /**
     * Permanently deletes a RateBasedRule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteRateBasedRule.html
     */
    toDeleteRateBasedRule() {
        this.to('waf-regional:DeleteRateBasedRule');
        return this;
    }
    /**
     * Permanently deletes an RegexMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteRegexMatchSet.html
     */
    toDeleteRegexMatchSet() {
        this.to('waf-regional:DeleteRegexMatchSet');
        return this;
    }
    /**
     * Permanently deletes an RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteRegexPatternSet.html
     */
    toDeleteRegexPatternSet() {
        this.to('waf-regional:DeleteRegexPatternSet');
        return this;
    }
    /**
     * Permanently deletes a Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteRule.html
     */
    toDeleteRule() {
        this.to('waf-regional:DeleteRule');
        return this;
    }
    /**
     * Permanently deletes a RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        this.to('waf-regional:DeleteRuleGroup');
        return this;
    }
    /**
     * Permanently deletes a SizeConstraintSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteSizeConstraintSet.html
     */
    toDeleteSizeConstraintSet() {
        this.to('waf-regional:DeleteSizeConstraintSet');
        return this;
    }
    /**
     * Permanently deletes a SqlInjectionMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteSqlInjectionMatchSet.html
     */
    toDeleteSqlInjectionMatchSet() {
        this.to('waf-regional:DeleteSqlInjectionMatchSet');
        return this;
    }
    /**
     * Permanently deletes a WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteWebACL.html
     */
    toDeleteWebACL() {
        this.to('waf-regional:DeleteWebACL');
        return this;
    }
    /**
     * Permanently deletes an XssMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DeleteXssMatchSet.html
     */
    toDeleteXssMatchSet() {
        this.to('waf-regional:DeleteXssMatchSet');
        return this;
    }
    /**
     * Removes a WebACL from the specified resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_DisassociateWebACL.html
     */
    toDisassociateWebACL() {
        this.to('waf-regional:DisassociateWebACL');
        return this;
    }
    /**
     * Returns the ByteMatchSet specified by ByteMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetByteMatchSet.html
     */
    toGetByteMatchSet() {
        this.to('waf-regional:GetByteMatchSet');
        return this;
    }
    /**
     * When you want to create, update, or delete AWS WAF objects, get a change token and include the change token in the create, update, or delete equest.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetChangeToken.html
     */
    toGetChangeToken() {
        this.to('waf-regional:GetChangeToken');
        return this;
    }
    /**
     * Returns the status of a ChangeToken that you got by calling GetChangeToken.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetChangeTokenStatus.html
     */
    toGetChangeTokenStatus() {
        this.to('waf-regional:GetChangeTokenStatus');
        return this;
    }
    /**
     * Returns the GeoMatchSet specified by GeoMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetGeoMatchSet.html
     */
    toGetGeoMatchSet() {
        this.to('waf-regional:GetGeoMatchSet');
        return this;
    }
    /**
     * Returns the IPSet that is specified by IPSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetIPSet.html
     */
    toGetIPSet() {
        this.to('waf-regional:GetIPSet');
        return this;
    }
    /**
     * Returns the LoggingConfiguration for the specified web ACL.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetLoggingConfiguration.html
     */
    toGetLoggingConfiguration() {
        this.to('waf-regional:GetLoggingConfiguration');
        return this;
    }
    /**
     * Returns the IAM policy attached to the RuleGroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetPermissionPolicy.html
     */
    toGetPermissionPolicy() {
        this.to('waf-regional:GetPermissionPolicy');
        return this;
    }
    /**
     * Returns the RateBasedRule that is specified by the RuleId that you included in the GetRateBasedRule request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetRateBasedRule.html
     */
    toGetRateBasedRule() {
        this.to('waf-regional:GetRateBasedRule');
        return this;
    }
    /**
     * Returns an array of IP addresses currently being blocked by the RateBasedRule that is specified by the RuleId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetRateBasedRuleManagedKeys.html
     */
    toGetRateBasedRuleManagedKeys() {
        this.to('waf-regional:GetRateBasedRuleManagedKeys');
        return this;
    }
    /**
     * Returns the RegexMatchSet specified by RegexMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetRegexMatchSet.html
     */
    toGetRegexMatchSet() {
        this.to('waf-regional:GetRegexMatchSet');
        return this;
    }
    /**
     * Returns the RegexPatternSet specified by RegexPatternSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetRegexPatternSet.html
     */
    toGetRegexPatternSet() {
        this.to('waf-regional:GetRegexPatternSet');
        return this;
    }
    /**
     * Returns the Rule that is specified by the RuleId that you included in the GetRule request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetRule.html
     */
    toGetRule() {
        this.to('waf-regional:GetRule');
        return this;
    }
    /**
     * Returns the RuleGroup that is specified by the RuleGroupId that you included in the GetRuleGroup request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetRuleGroup.html
     */
    toGetRuleGroup() {
        this.to('waf-regional:GetRuleGroup');
        return this;
    }
    /**
     * Gets detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests hat your AWS resource received during a time range that you choose.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetSampledRequests.html
     */
    toGetSampledRequests() {
        this.to('waf-regional:GetSampledRequests');
        return this;
    }
    /**
     * Returns the SizeConstraintSet specified by SizeConstraintSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetSizeConstraintSet.html
     */
    toGetSizeConstraintSet() {
        this.to('waf-regional:GetSizeConstraintSet');
        return this;
    }
    /**
     * Returns the SqlInjectionMatchSet that is specified by SqlInjectionMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetSqlInjectionMatchSet.html
     */
    toGetSqlInjectionMatchSet() {
        this.to('waf-regional:GetSqlInjectionMatchSet');
        return this;
    }
    /**
     * Returns the WebACL that is specified by WebACLId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetWebACL.html
     */
    toGetWebACL() {
        this.to('waf-regional:GetWebACL');
        return this;
    }
    /**
     * Returns the WebACL for the specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetWebACLForResource.html
     */
    toGetWebACLForResource() {
        this.to('waf-regional:GetWebACLForResource');
        return this;
    }
    /**
     * Returns the XssMatchSet that is specified by XssMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GetXssMatchSet.html
     */
    toGetXssMatchSet() {
        this.to('waf-regional:GetXssMatchSet');
        return this;
    }
    /**
     * Returns an array of ActivatedRule objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListActivatedRulesInRuleGroup.html
     */
    toListActivatedRulesInRuleGroup() {
        this.to('waf-regional:ListActivatedRulesInRuleGroup');
        return this;
    }
    /**
     * Returns an array of ByteMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListByteMatchSets.html
     */
    toListByteMatchSets() {
        this.to('waf-regional:ListByteMatchSets');
        return this;
    }
    /**
     * Returns an array of GeoMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListGeoMatchSets.html
     */
    toListGeoMatchSets() {
        this.to('waf-regional:ListGeoMatchSets');
        return this;
    }
    /**
     * Returns an array of IPSetSummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListIPSets.html
     */
    toListIPSets() {
        this.to('waf-regional:ListIPSets');
        return this;
    }
    /**
     * Returns an array of LoggingConfiguration objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListLoggingConfigurations.html
     */
    toListLoggingConfigurations() {
        this.to('waf-regional:ListLoggingConfigurations');
        return this;
    }
    /**
     * Returns an array of RuleSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListRateBasedRules.html
     */
    toListRateBasedRules() {
        this.to('waf-regional:ListRateBasedRules');
        return this;
    }
    /**
     * Returns an array of RegexMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListRegexMatchSets.html
     */
    toListRegexMatchSets() {
        this.to('waf-regional:ListRegexMatchSets');
        return this;
    }
    /**
     * Returns an array of RegexPatternSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListRegexPatternSets.html
     */
    toListRegexPatternSets() {
        this.to('waf-regional:ListRegexPatternSets');
        return this;
    }
    /**
     * Returns an array of resources associated with the specified WebACL.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListResourcesForWebACL.html
     */
    toListResourcesForWebACL() {
        this.to('waf-regional:ListResourcesForWebACL');
        return this;
    }
    /**
     * Returns an array of RuleGroup objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListRuleGroups.html
     */
    toListRuleGroups() {
        this.to('waf-regional:ListRuleGroups');
        return this;
    }
    /**
     * Returns an array of RuleSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListRules.html
     */
    toListRules() {
        this.to('waf-regional:ListRules');
        return this;
    }
    /**
     * Returns an array of SizeConstraintSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListSizeConstraintSets.html
     */
    toListSizeConstraintSets() {
        this.to('waf-regional:ListSizeConstraintSets');
        return this;
    }
    /**
     * Returns an array of SqlInjectionMatchSet objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListSqlInjectionMatchSets.html
     */
    toListSqlInjectionMatchSets() {
        this.to('waf-regional:ListSqlInjectionMatchSets');
        return this;
    }
    /**
     * Returns an array of RuleGroup objects that you are subscribed to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListSubscribedRuleGroups.html
     */
    toListSubscribedRuleGroups() {
        this.to('waf-regional:ListSubscribedRuleGroups');
        return this;
    }
    /**
     * Lists the Tags for a given resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('waf-regional:ListTagsForResource');
        return this;
    }
    /**
     * Returns an array of WebACLSummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListWebACLs.html
     */
    toListWebACLs() {
        this.to('waf-regional:ListWebACLs');
        return this;
    }
    /**
     * Returns an array of XssMatchSet objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ListXssMatchSets.html
     */
    toListXssMatchSets() {
        this.to('waf-regional:ListXssMatchSets');
        return this;
    }
    /**
     * Associates a LoggingConfiguration with a specified web ACL.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_PutLoggingConfiguration.html
     */
    toPutLoggingConfiguration() {
        this.to('waf-regional:PutLoggingConfiguration');
        return this;
    }
    /**
     * Attaches a IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_PutPermissionPolicy.html
     */
    toPutPermissionPolicy() {
        this.to('waf-regional:PutPermissionPolicy');
        return this;
    }
    /**
     * Adds a Tag to a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_TagResource.html
     */
    toTagResource() {
        this.to('waf-regional:TagResource');
        return this;
    }
    /**
     * Removes a Tag from a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UntagResource.html
     */
    toUntagResource() {
        this.to('waf-regional:UntagResource');
        return this;
    }
    /**
     * Inserts or deletes ByteMatchTuple objects (filters) in a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateByteMatchSet.html
     */
    toUpdateByteMatchSet() {
        this.to('waf-regional:UpdateByteMatchSet');
        return this;
    }
    /**
     * Inserts or deletes GeoMatchConstraint objects in a GeoMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateGeoMatchSet.html
     */
    toUpdateGeoMatchSet() {
        this.to('waf-regional:UpdateGeoMatchSet');
        return this;
    }
    /**
     * Inserts or deletes IPSetDescriptor objects in an IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateIPSet.html
     */
    toUpdateIPSet() {
        this.to('waf-regional:UpdateIPSet');
        return this;
    }
    /**
     * Inserts or deletes Predicate objects in a rule and updates the RateLimit in the rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateRateBasedRule.html
     */
    toUpdateRateBasedRule() {
        this.to('waf-regional:UpdateRateBasedRule');
        return this;
    }
    /**
     * Inserts or deletes RegexMatchTuple objects (filters) in a RegexMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateRegexMatchSet.html
     */
    toUpdateRegexMatchSet() {
        this.to('waf-regional:UpdateRegexMatchSet');
        return this;
    }
    /**
     * Inserts or deletes RegexPatternStrings in a RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateRegexPatternSet.html
     */
    toUpdateRegexPatternSet() {
        this.to('waf-regional:UpdateRegexPatternSet');
        return this;
    }
    /**
     * Inserts or deletes Predicate objects in a Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateRule.html
     */
    toUpdateRule() {
        this.to('waf-regional:UpdateRule');
        return this;
    }
    /**
     * Inserts or deletes ActivatedRule objects in a RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        this.to('waf-regional:UpdateRuleGroup');
        return this;
    }
    /**
     * Inserts or deletes SizeConstraint objects (filters) in a SizeConstraintSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateSizeConstraintSet.html
     */
    toUpdateSizeConstraintSet() {
        this.to('waf-regional:UpdateSizeConstraintSet');
        return this;
    }
    /**
     * Inserts or deletes SqlInjectionMatchTuple objects (filters) in a SqlInjectionMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateSqlInjectionMatchSet.html
     */
    toUpdateSqlInjectionMatchSet() {
        this.to('waf-regional:UpdateSqlInjectionMatchSet');
        return this;
    }
    /**
     * Inserts or deletes ActivatedRule objects in a WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateWebACL.html
     */
    toUpdateWebACL() {
        this.to('waf-regional:UpdateWebACL');
        return this;
    }
    /**
     * Inserts or deletes XssMatchTuple objects (filters) in an XssMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_UpdateXssMatchSet.html
     */
    toUpdateXssMatchSet() {
        this.to('waf-regional:UpdateXssMatchSet');
        return this;
    }
    /**
     * Adds a resource of type bytematchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_ByteMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBytematchset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:bytematchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_IPSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIpset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:ipset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loadbalancer/app/ to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_WebACL.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadbalancerApp(loadBalancerName, loadBalancerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/app/${LoadBalancerName}/${LoadBalancerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ratebasedrule to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_RateBasedRule.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRatebasedrule(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:ratebasedrule/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_Rule.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:rule/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sizeconstraintset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_SizeConstraintSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSizeconstraintset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:sizeconstraintset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sqlinjectionmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_SqlInjectionMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSqlinjectionmatchset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:sqlinjectionset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_WebACL.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebacl(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:webacl/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type xssmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_XssMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onXssmatchset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:xssmatchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_RegexMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRegexmatchset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:regexmatch/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexpatternset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_RegexPatternSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRegexpatternset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:regexpatternset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type geomatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_GeoMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGeomatchset(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:geomatchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rulegroup to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_wafRegional_RuleGroup.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRulegroup(id, account, region, partition) {
        var arn = 'arn:${Partition}:waf-regional:${Region}:${Account}:rulegroup/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.WafRegional = WafRegional;
//# sourceMappingURL=data:application/json;base64,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