"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wafv2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [wafv2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswafv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wafv2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [wafv2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswafv2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'wafv2';
        this.accessLevelList = {
            "Write": [
                "AssociateWebACL",
                "CreateIPSet",
                "CreateRegexPatternSet",
                "CreateRuleGroup",
                "DeleteFirewallManagerRuleGroups",
                "DeleteIPSet",
                "DeleteLoggingConfiguration",
                "DeleteRegexPatternSet",
                "DeleteRuleGroup",
                "DisassociateFirewallManager",
                "DisassociateWebACL",
                "PutFirewallManagerRuleGroups",
                "PutLoggingConfiguration",
                "UpdateIPSet",
                "UpdateRegexPatternSet",
                "UpdateRuleGroup"
            ],
            "Read": [
                "CheckCapacity",
                "GetIPSet",
                "GetLoggingConfiguration",
                "GetPermissionPolicy",
                "GetRateBasedStatementManagedKeys",
                "GetRegexPatternSet",
                "GetRuleGroup",
                "GetSampledRequests",
                "GetWebACL",
                "GetWebACLForResource",
                "ListTagsForResource"
            ],
            "Permissions management": [
                "CreateWebACL",
                "DeletePermissionPolicy",
                "DeleteWebACL",
                "PutPermissionPolicy",
                "UpdateWebACL"
            ],
            "List": [
                "DescribeManagedRuleGroup",
                "ListAvailableManagedRuleGroups",
                "ListIPSets",
                "ListLoggingConfigurations",
                "ListRegexPatternSets",
                "ListResourcesForWebACL",
                "ListRuleGroups",
                "ListWebACLs"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate a WebACL with a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_AssociateWebACL.html
     */
    toAssociateWebACL() {
        this.to('wafv2:AssociateWebACL');
        return this;
    }
    /**
     * Grants permission to calculate web ACL capacity unit (WCU) requirements for a specified scope and set of rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CheckCapacity.html
     */
    toCheckCapacity() {
        this.to('wafv2:CheckCapacity');
        return this;
    }
    /**
     * Grants permission to create an IPSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        this.to('wafv2:CreateIPSet');
        return this;
    }
    /**
     * Grants permission to create a RegexPatternSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRegexPatternSet.html
     */
    toCreateRegexPatternSet() {
        this.to('wafv2:CreateRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to create a RuleGroup.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        this.to('wafv2:CreateRuleGroup');
        return this;
    }
    /**
     * Grants permission to create a WebACL.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateWebACL.html
     */
    toCreateWebACL() {
        this.to('wafv2:CreateWebACL');
        return this;
    }
    /**
     * Grants permission to delete specified FirewallManagedRulesGroups from the specified WebACL if not managed by Firewall Manager anymore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteFirewallManagerRuleGroups.html
     */
    toDeleteFirewallManagerRuleGroups() {
        this.to('wafv2:DeleteFirewallManagerRuleGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        this.to('wafv2:DeleteIPSet');
        return this;
    }
    /**
     * Grants permission to delete the LoggingConfiguration from the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteLoggingConfiguration.html
     */
    toDeleteLoggingConfiguration() {
        this.to('wafv2:DeleteLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the PermissionPolicy on the specified RuleGroup.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeletePermissionPolicy.html
     */
    toDeletePermissionPolicy() {
        this.to('wafv2:DeletePermissionPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRegexPatternSet.html
     */
    toDeleteRegexPatternSet() {
        this.to('wafv2:DeleteRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to delete the specified RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        this.to('wafv2:DeleteRuleGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteWebACL.html
     */
    toDeleteWebACL() {
        this.to('wafv2:DeleteWebACL');
        return this;
    }
    /**
     * Grants permission to view high-level information for a managed rule group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DescribeManagedRuleGroup.html
     */
    toDescribeManagedRuleGroup() {
        this.to('wafv2:DescribeManagedRuleGroup');
        return this;
    }
    /**
     * Grants permission to disassociate Firewall Manager from the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateFirewallManager.html
     */
    toDisassociateFirewallManager() {
        this.to('wafv2:DisassociateFirewallManager');
        return this;
    }
    /**
     * Grants permission disassociate a WebACL from an application resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateWebACL.html
     */
    toDisassociateWebACL() {
        this.to('wafv2:DisassociateWebACL');
        return this;
    }
    /**
     * Grants permission to view details about the specified IPSet.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        this.to('wafv2:GetIPSet');
        return this;
    }
    /**
     * Grants permission to view LoggingConfiguration about the specified WebACL.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetLoggingConfiguration.html
     */
    toGetLoggingConfiguration() {
        this.to('wafv2:GetLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to view PermissionPolicy on the specified RuleGroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetPermissionPolicy.html
     */
    toGetPermissionPolicy() {
        this.to('wafv2:GetPermissionPolicy');
        return this;
    }
    /**
     * Grants permission to view the keys that are currently blocked by a rate-based rule.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRateBasedStatementManagedKeys.html
     */
    toGetRateBasedStatementManagedKeys() {
        this.to('wafv2:GetRateBasedStatementManagedKeys');
        return this;
    }
    /**
     * Grants permission to view details about the specified RegexPatternSet.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRegexPatternSet.html
     */
    toGetRegexPatternSet() {
        this.to('wafv2:GetRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to view details about the specified RuleGroup.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRuleGroup.html
     */
    toGetRuleGroup() {
        this.to('wafv2:GetRuleGroup');
        return this;
    }
    /**
     * Grants permission to view detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetSampledRequests.html
     */
    toGetSampledRequests() {
        this.to('wafv2:GetSampledRequests');
        return this;
    }
    /**
     * Grants permission to view details about the specified GetWebACL.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACL.html
     */
    toGetWebACL() {
        this.to('wafv2:GetWebACL');
        return this;
    }
    /**
     * Grants permission to view the WebACL for the specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACLForResource.html
     */
    toGetWebACLForResource() {
        this.to('wafv2:GetWebACLForResource');
        return this;
    }
    /**
     * Grants permission to view an array of managed rule groups that are available for you to use.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListAvailableManagedRuleGroups.html
     */
    toListAvailableManagedRuleGroups() {
        this.to('wafv2:ListAvailableManagedRuleGroups');
        return this;
    }
    /**
     * Grants permission to view an array of IPSetSummary objects for the IP sets that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListIPSets.html
     */
    toListIPSets() {
        this.to('wafv2:ListIPSets');
        return this;
    }
    /**
     * Grants permission to view an array of your LoggingConfiguration objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListLoggingConfigurations.html
     */
    toListLoggingConfigurations() {
        this.to('wafv2:ListLoggingConfigurations');
        return this;
    }
    /**
     * Grants permission to view an array of RegexPatternSetSummary objects for the regex pattern sets that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRegexPatternSets.html
     */
    toListRegexPatternSets() {
        this.to('wafv2:ListRegexPatternSets');
        return this;
    }
    /**
     * Grants permission to view an array of the Amazon Resource Names (ARNs) for the resources that are associated with the specified web ACL.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListResourcesForWebACL.html
     */
    toListResourcesForWebACL() {
        this.to('wafv2:ListResourcesForWebACL');
        return this;
    }
    /**
     * Grants permission to view an array of RuleGroupSummary objects for the rule groups that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRuleGroups.html
     */
    toListRuleGroups() {
        this.to('wafv2:ListRuleGroups');
        return this;
    }
    /**
     * Grants permission to lists tag for the specified resource.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('wafv2:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to view an array of WebACLSummary objects for the web ACLs that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListWebACLs.html
     */
    toListWebACLs() {
        this.to('wafv2:ListWebACLs');
        return this;
    }
    /**
     * Grants permission to create FirewallManagedRulesGroups in the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutFirewallManagerRuleGroups.html
     */
    toPutFirewallManagerRuleGroups() {
        this.to('wafv2:PutFirewallManagerRuleGroups');
        return this;
    }
    /**
     * Grants permission to enables the specified LoggingConfiguration, to start logging from a web ACL.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutLoggingConfiguration.html
     */
    toPutLoggingConfiguration() {
        this.to('wafv2:PutLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to attach the specified IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutPermissionPolicy.html
     */
    toPutPermissionPolicy() {
        this.to('wafv2:PutPermissionPolicy');
        return this;
    }
    /**
     * Grants permission to associates tags with the specified AWS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('wafv2:TagResource');
        return this;
    }
    /**
     * Grants permission to disassociates tags from an AWS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('wafv2:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the specified IPSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        this.to('wafv2:UpdateIPSet');
        return this;
    }
    /**
     * Grants permission to update the specified RegexPatternSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRegexPatternSet.html
     */
    toUpdateRegexPatternSet() {
        this.to('wafv2:UpdateRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to update the specified RuleGroup.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        this.to('wafv2:UpdateRuleGroup');
        return this;
    }
    /**
     * Grants permission to update the specified WebACL.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateWebACL.html
     */
    toUpdateWebACL() {
        this.to('wafv2:UpdateWebACL');
        return this;
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebacl(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/webacl/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_IPSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/ipset/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rulegroup to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RuleGroup.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRulegroup(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/rulegroup/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexpatternset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RegexPatternSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegexpatternset(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/regexpatternset/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loadbalancer/app/ to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadbalancerApp(loadBalancerName, loadBalancerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/app/${LoadBalancerName}/${LoadBalancerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type apigateway to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param apiId - Identifier for the apiId.
     * @param stageName - Identifier for the stageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApigateway(apiId, stageName, account, region, partition) {
        var arn = 'arn:${Partition}:apigateway:${Region}:${Account}:/restapis/${ApiId}/stages/${StageName}';
        arn = arn.replace('${ApiId}', apiId);
        arn = arn.replace('${StageName}', stageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type appsync to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAppsync(graphQLAPIId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Wafv2 = Wafv2;
//# sourceMappingURL=data:application/json;base64,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